import { Section, SectionDiff, SectionId } from './section'
import { documentToJsonReplacer, uniqueid, UniqueId } from './util'

/**
 * Type of a step in a history state node.
 */
export const enum HistoryStepType {
    create = 0,
    update = 1,
    remove = 2,
}

/**
 * Step for creating a new section.
 */
export type HistoryStepCreate = {
    type: HistoryStepType.create
    section: Section
    after?: SectionId
}
/**
 * Step for updating an existing section.
 */
export type HistoryStepUpdate = {
    type: HistoryStepType.update
    diff: SectionDiff
}
/**
 * Step for removing a section.
 */
export type HistoryStepRemove = {
    type: HistoryStepType.remove
    previous: Section
    after?: SectionId
}

/**
 * Unification of all types of steps.
 */
export type HistoryStep = HistoryStepCreate | HistoryStepUpdate | HistoryStepRemove

/**
 * Map from section id to history step.
 * Every section can have at most one step per history node.
 */
export type ChangeMap = Map<SectionId, HistoryStep>

/**
 * Unique identifier of a history state node.
 */
export type HistoryStateId = UniqueId

/**
 * A single state node in the history tree.
 */
export class HistoryNode {
    id: HistoryStateId
    parent: HistoryStateId | undefined
    children: Set<HistoryStateId>
    changes: ChangeMap

    constructor(parent?: HistoryStateId) {
        this.id = uniqueid()
        this.parent = parent
        this.children = new Set()
        this.changes = new Map()
    }

    toString() {
        return 'HistoryNode ' + JSON.stringify(this.toJSON(), documentToJsonReplacer, 4)
    }

    toJSON() {
        return {
            id: this.id,
            parent: this.parent,
            children: this.children,
            changes: this.changes,
        }
    }
}

/**
 * Root of the history tree.
 */
export class HistoryRoot {
    private root: HistoryStateId
    private current: HistoryStateId
    private nodes: Map<HistoryStateId, HistoryNode>

    constructor() {
        const root = new HistoryNode()
        this.root = root.id
        this.current = this.root
        this.nodes = new Map()
        this.nodes.set(root.id, root)
    }

    getNode(id: HistoryStateId) {
        return this.nodes.get(id)
    }

    getCurrentNode() {
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        return this.getNode(this.current)!
    }

    getChangeSet(id: HistoryStateId) {
        return this.getNode(id)?.changes
    }

    getCurrentChangeSet() {
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        return this.getChangeSet(this.current)!
    }

    isCurrentRoot() {
        return this.current === this.root
    }

    appendChanges(changes: ChangeMap) {
        if (changes.size === 0) return
        const changeSet = this.getCurrentChangeSet()
        for (const [id, change] of changes.entries()) {
            changeSet.set(id, change)
        }
    }

    pushState() {
        console.log('push history state')
        const newHistoryNode = new HistoryNode(this.current)
        this.nodes.set(newHistoryNode.id, newHistoryNode)
        this.getCurrentNode().children.add(newHistoryNode.id)
        this.current = newHistoryNode.id
        return this.current
    }

    popState() {
        const parent = this.getCurrentNode().parent
        if (!parent) return
        this.current = parent
        return this.current
    }

    descendState(branch: HistoryStateId) {
        if (!this.getCurrentNode().children.has(branch)) {
            return
        }
        this.current = branch
        return this.getCurrentNode()
    }

    toString() {
        return 'HistoryRoot ' + JSON.stringify(this.toJSON(), documentToJsonReplacer, 4)
    }

    toJSON() {
        return {
            root: this.root,
            current: this.current,
            nodes: this.nodes,
        }
    }
}
