import { canJoin, liftTarget, canSplit, ReplaceAroundStep, replaceStep, Step } from 'prosemirror-transform'
import { Slice, Fragment, Schema, ResolvedPos, Node, ContentMatch, NodeType } from 'prosemirror-model'
import { Selection, TextSelection, NodeSelection, AllSelection, EditorState, Transaction } from 'prosemirror-state'
import { isMacOS } from '../../util/compat'
import { EditorView } from 'prosemirror-view'

export interface Command<S extends Schema = any> {
    (state: EditorState<S>, dispatch?: (tr: Transaction<S>) => void, view?: EditorView<S>): boolean
}
export interface Keymap<S extends Schema = any> {
    [key: string]: Command<S>
}
export type KeymapBindings<S extends Schema = any> = Record<string, Command<S>>
export type DispatchTransaction<S extends Schema = any> = (tr: Transaction<S>) => void
export type AnyStep<S extends Schema = any, T extends Step<S> = any> = T

function textblockAt<S extends Schema = any>(node: Node<S>, side: string, only?: boolean) {
    for (
        let tempNode: Node<S> | null | undefined = node;
        tempNode;
        tempNode = side == 'start' ? tempNode.firstChild : tempNode.lastChild
    ) {
        if (tempNode.isTextblock) return true
        if (only && tempNode.childCount != 1) return false
    }
    return false
}

function deleteBarrier<S extends Schema = any>(
    state: EditorState<S>,
    $cut: ResolvedPos<S>,
    dispatch?: DispatchTransaction<S>
) {
    const before = $cut.nodeBefore
    const after = $cut.nodeAfter

    if (!before || !after) {
        console.error('undefined before or after:', before, after, 'in state', state)
        return false
    }

    if (before.type.spec.isolating || after.type.spec.isolating) return false
    if (joinMaybeClear(state, $cut, dispatch)) return true

    const canDelAfter = $cut.parent.canReplace($cut.index(), $cut.index() + 1)
    if (canDelAfter) {
        const match = before.contentMatchAt(before.childCount)
        const conn = match.findWrapping(after.type)
        if (conn && match.matchType(conn[0] || after.type)?.validEnd) {
            if (dispatch) {
                const end = $cut.pos + after.nodeSize
                let wrap = Fragment.empty
                for (let i = conn.length - 1; i >= 0; i--) wrap = Fragment.from(conn[i].create(undefined, wrap))
                wrap = Fragment.from(before.copy(wrap))
                const tr = state.tr.step(
                    new ReplaceAroundStep($cut.pos - 1, end, $cut.pos, end, new Slice(wrap, 1, 0), conn.length, true)
                )
                const joinAt = end + 2 * conn.length
                if (canJoin(tr.doc, joinAt)) tr.join(joinAt)
                dispatch(tr.scrollIntoView())
            }
            return true
        }
    }

    const selAfter = Selection.findFrom($cut, 1)
    const range = selAfter && selAfter.$from.blockRange(selAfter.$to)
    const target = range && liftTarget(range)
    if (range && target != undefined && target >= $cut.depth) {
        if (dispatch) dispatch(state.tr.lift(range, target).scrollIntoView())
        return true
    }

    if (canDelAfter && textblockAt(after, 'start', true) && textblockAt(before, 'end')) {
        let at = before
        const wrap = []
        for (;;) {
            wrap.push(at)
            if (at.isTextblock || !at.lastChild) break
            at = at.lastChild
        }
        let afterDepth = 1
        let afterText = after
        for (; !afterText.isTextblock && !!afterText.firstChild; afterText = afterText.firstChild) {
            afterDepth++
        }
        if (at.canReplace(at.childCount, at.childCount, afterText.content)) {
            if (dispatch) {
                let end = Fragment.empty
                for (let i = wrap.length - 1; i >= 0; i--) end = Fragment.from(wrap[i].copy(end))
                const tr = state.tr.step(
                    new ReplaceAroundStep(
                        $cut.pos - wrap.length,
                        $cut.pos + after.nodeSize,
                        $cut.pos + afterDepth,
                        $cut.pos + after.nodeSize - afterDepth,
                        new Slice(end, wrap.length, 0),
                        0,
                        true
                    )
                )
                dispatch(tr.scrollIntoView())
            }
            return true
        }
    }

    return false
}

function joinMaybeClear<S extends Schema = any>(
    state: EditorState<S>,
    $pos: ResolvedPos<S>,
    dispatch?: DispatchTransaction<S>
) {
    const before = $pos.nodeBefore,
        after = $pos.nodeAfter,
        index = $pos.index()
    // @ts-ignore
    if (!before || !after || !before.type.compatibleContent(after.type)) return false
    if (before.content.size === 0 && $pos.parent.canReplace(index - 1, index)) {
        if (dispatch) dispatch(state.tr.delete($pos.pos - before.nodeSize, $pos.pos).scrollIntoView())
        return true
    }
    if (!$pos.parent.canReplace(index, index + 1) || !(after.isTextblock || canJoin(state.doc, $pos.pos))) return false
    if (dispatch)
        dispatch(
            state.tr
                .clearIncompatible($pos.pos, before.type, before.contentMatchAt(before.childCount))
                .join($pos.pos)
                .scrollIntoView()
        )
    return true
}

function findCutBefore<S extends Schema = any>($pos: ResolvedPos<S>) {
    if (!$pos.parent.type.spec.isolating)
        for (let i = $pos.depth - 1; i >= 0; i--) {
            if ($pos.index(i) > 0) return $pos.doc.resolve($pos.before(i + 1))
            if ($pos.node(i).type.spec.isolating) break
        }
    return
}

function findCutAfter<S extends Schema = any>($pos: ResolvedPos<S>) {
    if (!$pos.parent.type.spec.isolating)
        for (let i = $pos.depth - 1; i >= 0; i--) {
            const parent = $pos.node(i)
            if ($pos.index(i) + 1 < parent.childCount) return $pos.doc.resolve($pos.after(i + 1))
            if (parent.type.spec.isolating) break
        }
    return
}

function defaultBlockAt<S extends Schema = any>(match: ContentMatch<S>) {
    for (let i = 0; i < match.edgeCount; i++) {
        const { type } = match.edge(i)
        if (type.isTextblock && !type.hasRequiredAttrs()) return type
    }
    return
}

// Select the whole document.
export function selectAll<S extends Schema = any>(state: EditorState<S>, dispatch?: DispatchTransaction<S>) {
    if (dispatch) dispatch(state.tr.setSelection(new AllSelection(state.doc)))
    return true
}

// Delete the selection, if there is one.
export function deleteSelection<S extends Schema = any>(state: EditorState<S>, dispatch?: DispatchTransaction<S>) {
    if (state.selection.empty) return false
    if (dispatch) dispatch(state.tr.deleteSelection().scrollIntoView())
    return true
}

// If the selection is empty and the cursor is at the end of a
// textblock, try to reduce or remove the boundary between that block
// and the one after it, either by joining them or by moving the other
// block closer to this one in the tree structure. Will use the view
// for accurate start-of-textblock detection if given.
export function joinForward<S extends Schema = any>(
    state: EditorState<S>,
    dispatch?: DispatchTransaction<S>,
    view?: EditorView<S>
) {
    if (!(state.selection instanceof TextSelection)) return false
    const selection = state.selection as TextSelection<S>

    const { $cursor } = selection
    if (
        !$cursor ||
        (view ? !view.endOfTextblock('forward', state) : $cursor.parentOffset < $cursor.parent.content.size)
    )
        return false

    const $cut = findCutAfter($cursor)

    // If there is no node after this, there's nothing to do
    if (!$cut) return false

    const after = $cut.nodeAfter
    if (!after) {
        console.error('no node after')
        return false
    }

    // Try the joining algorithm
    if (deleteBarrier(state, $cut, dispatch)) return true

    // If the node above has no content and the node below is
    // selectable, delete the node above and select the one below.
    if ($cursor.parent.content.size === 0 && (textblockAt(after, 'start') || NodeSelection.isSelectable(after))) {
        const delStep = replaceStep(state.doc, $cursor.before(), $cursor.after(), Slice.empty) as AnyStep<S>
        if (delStep && delStep.slice.size < delStep.to - delStep.from) {
            if (dispatch) {
                const tr = state.tr.step(delStep)
                const selection = textblockAt(after, 'start')
                    ? // eslint-disable-next-line unicorn/no-array-callback-reference
                      Selection.findFrom(tr.doc.resolve(tr.mapping.map($cut.pos)), 1)
                    : // eslint-disable-next-line unicorn/no-array-callback-reference
                      NodeSelection.create(tr.doc, tr.mapping.map($cut.pos))
                if (selection) {
                    tr.setSelection(selection)
                } else {
                    console.error('could not set selection')
                }
                dispatch(tr.scrollIntoView())
            }
            return true
        }
    }

    // If the next node is an atom, delete it
    if (after.isAtom && $cut.depth == $cursor.depth - 1) {
        if (dispatch) dispatch(state.tr.delete($cut.pos, $cut.pos + after.nodeSize).scrollIntoView())
        return true
    }

    return false
}

// If the selection is empty and at the start of a textblock, try to
// reduce the distance between that block and the one before it—if
// there's a block directly before it that can be joined, join them.
// If not, try to move the selected block closer to the next one in
// the document structure by lifting it out of its parent or moving it
// into a parent of the previous block. Will use the view for accurate
// (bidi-aware) start-of-textblock detection if given.
export function joinBackward<S extends Schema = any>(
    state: EditorState<S>,
    dispatch?: DispatchTransaction<S>,
    view?: EditorView<S>
) {
    if (!(state.selection instanceof TextSelection)) return false
    const selection = state.selection as TextSelection<S>

    const { $cursor } = selection
    if (!$cursor || (view ? !view.endOfTextblock('backward', state) : $cursor.parentOffset > 0)) return false

    const $cut = findCutBefore($cursor)

    // If there is no node before this, try to lift
    if (!$cut) {
        const range = $cursor.blockRange()
        if (!range) return false
        const target = range && liftTarget(range)
        if (target == undefined) return false
        if (dispatch) dispatch(state.tr.lift(range, target).scrollIntoView())
        return true
    }

    const before = $cut.nodeBefore
    if (!before) {
        console.error('no node before')
        return false
    }

    // Apply the joining algorithm
    if (!before.type.spec.isolating && deleteBarrier(state, $cut, dispatch)) return true

    // If the node below has no content and the node above is
    // selectable, delete the node below and select the one above.
    if ($cursor.parent.content.size === 0 && (textblockAt(before, 'end') || NodeSelection.isSelectable(before))) {
        const delStep = replaceStep(state.doc, $cursor.before(), $cursor.after(), Slice.empty) as AnyStep<S>
        if (delStep && delStep.slice.size < delStep.to - delStep.from) {
            if (dispatch) {
                const tr = state.tr.step(delStep)
                const selection = textblockAt(before, 'end')
                    ? // eslint-disable-next-line unicorn/no-array-callback-reference, unicorn/no-array-method-this-argument
                      Selection.findFrom(tr.doc.resolve(tr.mapping.map($cut.pos, -1)), -1)
                    : NodeSelection.create(tr.doc, $cut.pos - before.nodeSize)
                if (selection) {
                    tr.setSelection(selection)
                } else {
                    console.error('could not set selection')
                }
                dispatch(tr.scrollIntoView())
            }
            return true
        }
    }

    // If the node before is an atom, delete it
    if (before.isAtom && $cut.depth == $cursor.depth - 1) {
        if (dispatch) dispatch(state.tr.delete($cut.pos - before.nodeSize, $cut.pos).scrollIntoView())
        return true
    }

    return false
}

// When the selection is empty and at the end of a textblock, select
// the node coming after that textblock, if possible. This is intended
// to be bound to keys like delete, after
// [`joinForward`](#commands.joinForward) and similar deleting
// commands, to provide a fall-back behavior when the schema doesn't
// allow deletion at the selected point.
export function selectNodeForward<S extends Schema = any>(
    state: EditorState<S>,
    dispatch?: DispatchTransaction<S>,
    view?: EditorView<S>
) {
    const { $head, empty } = state.selection
    let $cut: ResolvedPos<S> | null | undefined = $head
    if (!empty) return false
    if ($head.parent.isTextblock) {
        if (view ? !view.endOfTextblock('forward', state) : $head.parentOffset < $head.parent.content.size) return false
        $cut = findCutAfter($head)
    }
    const node = $cut && $cut.nodeAfter
    if (!$cut || !node || !NodeSelection.isSelectable(node)) return false
    if (dispatch) dispatch(state.tr.setSelection(NodeSelection.create(state.doc, $cut.pos)).scrollIntoView())
    return true
}

// When the selection is empty and at the start of a textblock, select
// the node before that textblock, if possible. This is intended to be
// bound to keys like backspace, after
// [`joinBackward`](#commands.joinBackward) or other deleting
// commands, as a fall-back behavior when the schema doesn't allow
// deletion at the selected point.
export function selectNodeBackward<S extends Schema = any>(
    state: EditorState<S>,
    dispatch?: DispatchTransaction<S>,
    view?: EditorView<S>
) {
    const { $head, empty } = state.selection
    let $cut: ResolvedPos<S> | null | undefined = $head
    if (!empty) return false

    if ($head.parent.isTextblock) {
        if (view ? !view.endOfTextblock('backward', state) : $head.parentOffset > 0) return false
        $cut = findCutBefore($head)
    }
    const node = $cut && $cut.nodeBefore
    if (!$cut || !node || !NodeSelection.isSelectable(node)) return false
    if (dispatch)
        dispatch(state.tr.setSelection(NodeSelection.create(state.doc, $cut.pos - node.nodeSize)).scrollIntoView())
    return true
}

// If a block node is selected, create an empty paragraph before (if
// it is its parent's first child) or after it.
export function createParagraphNear<S extends Schema = any>(state: EditorState<S>, dispatch?: DispatchTransaction<S>) {
    const selection = state.selection
    const { $from, $to } = selection
    if (selection instanceof AllSelection || $from.parent.inlineContent || $to.parent.inlineContent) return false
    const type = defaultBlockAt($to.parent.contentMatchAt($to.indexAfter()))
    if (!type || !type.isTextblock) return false
    if (dispatch) {
        const side = (!$from.parentOffset && $to.index() < $to.parent.childCount ? $from : $to).pos
        const fill = type.createAndFill()
        if (!fill) {
            console.error('create and fill failed')
            return false
        }
        const tr = state.tr.insert(side, fill)
        tr.setSelection(TextSelection.create(tr.doc, side + 1))
        dispatch(tr.scrollIntoView())
    }
    return true
}

// If the cursor is in an empty textblock that can be lifted, lift the
// block.
export function liftEmptyBlock<S extends Schema = any>(state: EditorState<S>, dispatch?: DispatchTransaction<S>) {
    if (!(state.selection instanceof TextSelection)) return false
    const selection = state.selection as TextSelection<S>

    const { $cursor } = selection
    if (!$cursor || $cursor.parent.content.size > 0) return false
    if ($cursor.depth > 1 && $cursor.after() != $cursor.end(-1)) {
        const before = $cursor.before()
        if (canSplit(state.doc, before)) {
            if (dispatch) dispatch(state.tr.split(before).scrollIntoView())
            return true
        }
    }
    const range = $cursor.blockRange()
    if (!range) return false
    const target = range && liftTarget(range)
    if (target == undefined) return false
    if (dispatch) dispatch(state.tr.lift(range, target).scrollIntoView())
    return true
}

type SplitTypesAttr<S extends Schema = any> = { type: NodeType<S>; attrs?: { [key: string]: any } | null | undefined }
function deleteNodeTypeId<S extends Schema = any>(type: SplitTypesAttr<S>) {
    if ((type.type as any).defaultAttrs.id) {
        delete (type.type as any).defaultAttrs.id
    }
    if (type.attrs?.id) {
        delete type.attrs.id
    }
}
// Split the parent block of the selection. If the selection is a text
// selection, also delete its content.
export function splitBlock<S extends Schema = any>(state: EditorState<S>, dispatch?: DispatchTransaction<S>) {
    const { $from, $to } = state.selection
    // eslint-disable-next-line unicorn/consistent-destructuring
    if (state.selection instanceof NodeSelection && state.selection.node.isBlock) {
        // eslint-disable-next-line unicorn/consistent-destructuring
        const { type, attrs } = state.selection.node
        // we have to delete the id here so it's not copied into the new node
        deleteNodeTypeId({ type, attrs })
        if (!$from.parentOffset || !canSplit(state.doc, $from.pos)) return false
        if (dispatch) dispatch(state.tr.split($from.pos, undefined, [{ type, attrs }]).scrollIntoView())
        return true
    }

    if (!$from.parent.isBlock) return false

    if (dispatch) {
        const atEnd = $to.parentOffset == $to.parent.content.size
        const tr = state.tr
        if (state.selection instanceof TextSelection || state.selection instanceof AllSelection) tr.deleteSelection()
        const defaultNode =
            $from.depth == 0 ? undefined : defaultBlockAt($from.node(-1).contentMatchAt($from.indexAfter(-1)))
        let types: Array<SplitTypesAttr<S>> | undefined = /* atEnd && */ defaultNode
            ? [{ type: defaultNode }]
            : undefined
        types?.forEach((type) => {
            // we have to delete the id here so it's not copied into the new node
            deleteNodeTypeId(type)
        })
        // eslint-disable-next-line unicorn/no-array-callback-reference
        let can = canSplit(tr.doc, tr.mapping.map($from.pos), 1, types)
        if (
            defaultNode &&
            !types &&
            !can &&
            // eslint-disable-next-line unicorn/no-array-callback-reference
            canSplit(tr.doc, tr.mapping.map($from.pos), 1, defaultNode && [{ type: defaultNode }])
        ) {
            types = [{ type: defaultNode }]
            can = true
        }
        if (can) {
            // eslint-disable-next-line unicorn/no-array-callback-reference
            tr.split(tr.mapping.map($from.pos), 1, types)
            if (!atEnd && !$from.parentOffset && $from.parent.type != defaultNode) {
                const first = tr.mapping.map($from.before()),
                    $first = tr.doc.resolve(first)
                if (defaultNode && $from.node(-1).canReplaceWith($first.index(), $first.index() + 1, defaultNode))
                    tr.setNodeMarkup(tr.mapping.map($from.before()), defaultNode)
            }
        }
        dispatch(tr.scrollIntoView())
    }
    return true
}

function selectTextblockSide(side: number) {
    return <S extends Schema = any>(state: EditorState<S>, dispatch?: DispatchTransaction<S>) => {
        const sel = state.selection
        const $pos = side < 0 ? sel.$from : sel.$to
        let depth = $pos.depth
        while ($pos.node(depth).isInline) {
            if (!depth) return false
            depth--
        }
        if (!$pos.node(depth).isTextblock) return false
        if (dispatch)
            dispatch(
                state.tr.setSelection(TextSelection.create(state.doc, side < 0 ? $pos.start(depth) : $pos.end(depth)))
            )
        return true
    }
}

// Moves the cursor to the start of current text block.
export const selectTextblockStart = selectTextblockSide(-1)

// Moves the cursor to the end of current text block.
export const selectTextblockEnd = selectTextblockSide(1)

// Combine a number of command functions into a single function.
// Calls them one by one until one returns true.
export function chainCommands<S extends Schema = any>(...commands: Array<Command<S>>): Command<S> {
    return (state, dispatch, view) => {
        for (const command of commands) {
            if (command(state, dispatch, view)) return true
        }
        return false
    }
}

const enter = chainCommands(createParagraphNear, liftEmptyBlock, splitBlock)
const backspace = chainCommands(deleteSelection, joinBackward, selectNodeBackward)
const del = chainCommands(deleteSelection, joinForward, selectNodeForward)

const winKeymap = {
    Enter: enter,
    'Shift-Enter': enter,
    Backspace: backspace,
    'Mod-Backspace': backspace,
    'Shift-Backspace': backspace,
    Delete: del,
    'Mod-Delete': del,
    'Mod-a': selectAll,
}
const macKeymap = {
    ...winKeymap,
    'Ctrl-h': backspace,
    'Alt-Backspace': backspace,
    'Ctrl-d': del,
    'Ctrl-Alt-Backspace': del,
    'Alt-Delete': del,
    'Alt-d': del,
    'Ctrl-a': selectTextblockStart,
    'Ctrl-e': selectTextblockEnd,
}

export const editorKeymap = isMacOS ? macKeymap : winKeymap

export function normalizeKeyName(name: string): string {
    const parts = name.split(/-(?!$)/)
    let result = parts[parts.length - 1]
    if (result == 'Space') result = ' '
    let alt, ctrl, shift, meta
    for (let i = 0; i < parts.length - 1; i++) {
        const mod = parts[i]
        if (/^(cmd|meta|m)$/i.test(mod)) meta = true
        else if (/^a(lt)?$/i.test(mod)) alt = true
        else if (/^(c|ctrl|control)$/i.test(mod)) ctrl = true
        else if (/^s(hift)?$/i.test(mod)) shift = true
        else if (/^mod$/i.test(mod)) {
            if (isMacOS) meta = true
            else ctrl = true
        } else throw new Error('Unrecognized modifier name: ' + mod)
    }
    if (alt) result = 'Alt-' + result
    if (ctrl) result = 'Ctrl-' + result
    if (meta) result = 'Meta-' + result
    if (shift) result = 'Shift-' + result
    return result
}

function normalizeKeyMap(map: KeymapBindings): KeymapBindings {
    const copy = Object.create(null)
    for (const prop in map) copy[normalizeKeyName(prop)] = map[prop]
    return copy
}

function keyNameWithModifiers(name: string, event: KeyboardEvent, shift = true) {
    if (event.altKey) name = 'Alt-' + name
    if (event.ctrlKey) name = 'Ctrl-' + name
    if (event.metaKey) name = 'Meta-' + name
    if (shift !== false && event.shiftKey) name = 'Shift-' + name
    return name
}

export function keydownHandler(bindings: KeymapBindings) {
    const keyMap = normalizeKeyMap(bindings)
    return <S extends Schema = any>(view: EditorView<S>, event: KeyboardEvent) => {
        const isChar = event.key.length == 1 && event.key != ' '
        // check direct key mapping
        const direct = keyMap[keyNameWithModifiers(event.key, event, !isChar)]
        if (direct && direct(view.state, view.dispatch, view)) return true
        // check key mapping with shift prefix
        if (isChar && event.shiftKey) {
            const withShift = keyMap[keyNameWithModifiers(event.key, event, true)]
            if (withShift && withShift(view.state, view.dispatch, view)) return true
        }
        return false
    }
}
