import { Mark, Schema } from 'prosemirror-model'
import { EditorState, Transaction } from 'prosemirror-state'

import { schema } from './schema'
import { formattingToMark, originToMark, SectionMetaFormatting, SectionMetaOrigin } from './glue'
import { mergeTransactionSteps } from './util'

import { Document } from '../../data/document/document'
import { SectionType, SectionTypeTextMeta } from '../../data/document/section'
import { ChangeMap, HistoryStepType } from '../../data/document/history'

export const applyMetaMarks = (
    tr: Transaction,
    startPosition: number,
    endPosition: number,
    marks: Array<SectionTypeTextMeta>,
    mapper: (mark: number) => Mark<Schema<string, string>> | undefined
) => {
    for (const { data, length, position } of marks) {
        const mark = mapper(data)
        if (!mark) continue
        const from = startPosition + position
        const until = length ? startPosition + position + length : endPosition + 1
        tr.addMark(from, until, mark)
    }
}

export const updateOriginMarks = (tr: Transaction, state: EditorState) => {
    const mergedSteps = mergeTransactionSteps(tr, [])
    for (const step of mergedSteps) {
        let isEdit = false
        step.getMap().forEach((oldStart, oldEnd, newStart, newEnd) => {
            state.doc.nodesBetween(oldStart, oldEnd, (node, pos, parent) => {
                // check if replaced text is ai or edited text
                if (isEdit) return false
                if (
                    node.marks.some(
                        (mark) =>
                            mark.eq(schema.mark(schema.marks.ai_text)) || mark.eq(schema.mark(schema.marks.edit_text))
                    )
                ) {
                    isEdit = true
                }
                // only descend one level into the tree
                return parent.type.name === 'paragraph' ? false : true
            })
            let prevNodeMarks = [] as Mark[]
            if (oldStart == oldEnd && oldStart > 0) {
                // if adding, look at previous node
                state.doc.nodesBetween(oldStart - 1, oldEnd, (node, pos, parent) => {
                    prevNodeMarks = node.marks
                    return parent.type.name === 'paragraph' ? false : true
                })
            }
            let nextNodeMarks = [] as Mark[]
            if (oldStart == oldEnd && oldEnd < newEnd) {
                // if adding, look at next node
                state.doc.nodesBetween(oldStart, oldEnd + 1, (node, pos, parent) => {
                    nextNodeMarks = node.marks
                    return parent.type.name === 'paragraph' ? false : true
                })
            }
            if (isEdit) {
                tr.addMark(newStart, newEnd, schema.mark(schema.marks.edit_text))
            } else if (
                [...prevNodeMarks, ...nextNodeMarks].some((mark) => mark.eq(schema.mark(schema.marks.edit_text)))
            ) {
                tr.addMark(newStart, newEnd, schema.mark(schema.marks.edit_text))
            } else {
                tr.addMark(newStart, newEnd, schema.mark(schema.marks.user_text))
            }
        })
    }
}

export const reloadDocument = (tr: Transaction, document: Document) => {
    tr.delete(0, tr.doc.content.size)
    for (const { id, section } of document.getSections()) {
        if (section.type !== SectionType.text) {
            // skip non-text nodes for now
            continue
        }
        let startPosition = tr.doc.content.size
        const paragraph = schema.nodes.paragraph.create({ id }, section.text ? schema.text(section.text) : undefined)
        if (startPosition <= 2) {
            // if doc is empty replace the empty line
            tr.replaceWith(0, 1, paragraph)
            startPosition = 1
        } else {
            tr.insert(startPosition, paragraph)
            startPosition = startPosition + 1
        }
        console.log('applying', startPosition, section.text.length, section.meta)
        applyMetaMarks(
            tr,
            startPosition,
            startPosition + section.text.length,
            section.meta.get(SectionMetaOrigin) ?? [],
            originToMark
        )
        applyMetaMarks(
            tr,
            startPosition,
            startPosition + section.text.length,
            section.meta.get(SectionMetaFormatting) ?? [],
            formattingToMark
        )
    }
    tr.setMeta('internal', true)
}

export const undoChanges = (tr: Transaction, changes: ChangeMap, document: Document) => {
    const sortedChanges = [...changes.entries()].sort(([, a], [, b]) => a.type - b.type).reverse()
    for (const [id, changedNode] of sortedChanges) {
        console.log(changedNode.type, changedNode)
        switch (changedNode.type) {
            case HistoryStepType.remove: {
                const section = changedNode.previous
                if (section.type !== SectionType.text) {
                    // skip non-text nodes for now
                    continue
                }
                let startPosition = -1
                const paragraph = schema.nodes.paragraph.create(
                    { id: id },
                    section.text ? schema.text(section.text) : undefined
                )
                if (changedNode.after) {
                    tr.doc.descendants((node, pos) => {
                        if (node.attrs.id === changedNode.after) {
                            tr.insert(pos + node.nodeSize, paragraph)
                            startPosition = pos + node.nodeSize + 1
                        }
                        return false
                    })
                }
                if (!changedNode.after) {
                    // if not after another node, add it to the end
                    if (tr.doc.content.size <= 2) {
                        // if doc is empty replace the empty line
                        tr.replaceWith(0, 1, paragraph)
                        startPosition = 1
                    } else {
                        startPosition = tr.doc.content.size + 1
                        tr.insert(tr.doc.content.size, paragraph)
                    }
                }
                if (startPosition != -1) {
                    applyMetaMarks(
                        tr,
                        startPosition,
                        startPosition + section.text.length,
                        section.meta.get(SectionMetaOrigin) ?? [],
                        originToMark
                    )
                    applyMetaMarks(
                        tr,
                        startPosition,
                        startPosition + section.text.length,
                        section.meta.get(SectionMetaFormatting) ?? [],
                        formattingToMark
                    )
                }
                break
            }
            case HistoryStepType.create: {
                tr.doc.descendants((node, pos) => {
                    if (node.attrs.id === id) {
                        tr.delete(pos, pos + node.nodeSize)
                    }
                    return false
                })
                break
            }
            case HistoryStepType.update: {
                // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                const section = document.getSection(id)!
                if (section.type !== SectionType.text) {
                    // skip non-text nodes for now
                    continue
                }
                let startPosition = -1
                const paragraph = schema.nodes.paragraph.create(
                    { id: id },
                    section.text ? schema.text(section.text) : undefined
                )
                tr.doc.descendants((node, pos) => {
                    if (node.attrs.id === id) {
                        tr.replaceWith(pos, pos + node.nodeSize, paragraph)
                        startPosition = pos + 1
                    }
                    return false
                })
                if (startPosition != -1) {
                    applyMetaMarks(
                        tr,
                        startPosition,
                        startPosition + section.text.length,
                        section.meta.get(SectionMetaOrigin) ?? [],
                        originToMark
                    )
                    applyMetaMarks(
                        tr,
                        startPosition,
                        startPosition + section.text.length,
                        section.meta.get(SectionMetaFormatting) ?? [],
                        formattingToMark
                    )
                }
                break
            }
        }
    }
}

export const redoChanges = (tr: Transaction, changes: ChangeMap, document: Document) => {
    const sortedChanges = [...changes.entries()].sort(([, a], [, b]) => a.type - b.type)
    for (const [id, changedNode] of sortedChanges) {
        switch (changedNode.type) {
            case HistoryStepType.remove: {
                tr.doc.descendants((node, pos) => {
                    if (node.attrs.id === id) {
                        tr.delete(pos, pos + node.nodeSize)
                    }
                    return false
                })

                break
            }
            case HistoryStepType.update: {
                // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                const section = document.getSection(id)!
                if (section.type !== SectionType.text) {
                    // skip non-text nodes for now
                    continue
                }
                let startPosition = -1
                const paragraph = schema.nodes.paragraph.create(
                    { id: id },
                    section.text ? schema.text(section.text) : undefined
                )
                tr.doc.descendants((node, pos) => {
                    if (node.attrs.id === id) {
                        tr.replaceWith(pos, pos + node.nodeSize, paragraph)
                        startPosition = pos + 1
                    }
                    return false
                })
                if (startPosition != -1) {
                    applyMetaMarks(
                        tr,
                        startPosition,
                        startPosition + section.text.length,
                        section.meta.get(SectionMetaOrigin) ?? [],
                        originToMark
                    )
                    applyMetaMarks(
                        tr,
                        startPosition,
                        startPosition + section.text.length,
                        section.meta.get(SectionMetaFormatting) ?? [],
                        formattingToMark
                    )
                }
                break
            }
            case HistoryStepType.create: {
                const section = changedNode.section
                if (section.type !== SectionType.text) {
                    // skip non-text nodes for now
                    continue
                }
                let startPosition = -1
                const paragraph = schema.nodes.paragraph.create(
                    { id: id },
                    section.text ? schema.text(section.text) : undefined
                )
                if (changedNode.after) {
                    // trying to append after
                    tr.doc.descendants((node, pos) => {
                        if (node.attrs.id === changedNode.after) {
                            // appending after
                            tr.insert(pos + node.nodeSize, paragraph)
                            startPosition = pos + node.nodeSize + 1
                        }
                        return false
                    })
                }
                if (!changedNode.after) {
                    // if not after another node, add it to the end
                    if (tr.doc.content.size <= 2) {
                        // if doc is empty replace the empty line
                        tr.replaceWith(0, 1, paragraph)
                        startPosition = 1
                    } else {
                        startPosition = tr.doc.content.size + 1
                        tr.insert(tr.doc.content.size, paragraph)
                    }
                }
                if (startPosition != -1) {
                    applyMetaMarks(
                        tr,
                        startPosition,
                        startPosition + section.text.length,
                        section.meta.get(SectionMetaOrigin) ?? [],
                        originToMark
                    )
                    applyMetaMarks(
                        tr,
                        startPosition,
                        startPosition + section.text.length,
                        section.meta.get(SectionMetaFormatting) ?? [],
                        formattingToMark
                    )
                }
                break
            }
        }
    }
}
