import { Buffer } from 'buffer'
import { FormEvent, useEffect, useState } from 'react'
import { useRecoilValue } from 'recoil'
import { default as sodium } from 'libsodium-wrappers'
import styled from 'styled-components'
import Link from 'next/link'
import Head from 'next/head'

import { useRouter } from 'next/router'
import { getLoginRequest } from '../data/request/request'
import { User, UserInformation, UserPriority, UserSubscription } from '../data/user/user'
import { UserSettings } from '../data/user/settings'
import { GlobalUserContext } from '../globals/globals'
import { Session, SessionValue } from '../globals/state'
import {
    LoginError,
    Field,
    LoginBackground,
    LoginBox,
    LoginHeader,
    Submit,
    LoginTop,
    LoginContainer,
} from '../styles/components/login'
import Checkbox from '../components/controls/checkbox'
import { logError, logInfo, logWarning } from '../util/browser'
import { ArrowLeftIcon } from '../styles/ui/icons'
import { useUserSetup } from '../hooks/useUserSetup'
import { useAutoLogin } from '../hooks/useAutoLogin'
import { LoadingSpinner } from '../components/loading'
import { setLocalTrialState } from '../components/trialactions'
import { removeLocalStorage } from '../util/storage'
import { emailConfirmationRequiredDate } from '../util/util'

export default function Login(): JSX.Element {
    return (
        <LoginContainer>
            <Head>
                <title>Login - NovelAI</title>
            </Head>
            <LoginBackground />
            <LoginContent />
        </LoginContainer>
    )
}

interface ILoginData {
    email: undefined | string
    access_key: undefined | string
    encryption_key: undefined | string
    subscription: undefined | UserSubscription
    session: undefined | any
    settings: undefined | UserSettings
    priority: undefined | UserPriority
    information: undefined | UserInformation
}

function LoginContent(): JSX.Element {
    const authenticated = useRecoilValue(SessionValue('authenticated'))
    const session = useRecoilValue(Session)

    const [rawUsername, setUsername] = useState('')
    const [password, setPassword] = useState('')

    const [error, setError] = useState('')
    const [inputsEnabled, setInputsEnabled] = useState(true)

    const [remember, setRemember] = useState(true)
    const [accountSelection, setAccountSelection] = useState<Array<ILoginData>>([])

    const [loading, setLoading] = useState(true)

    const setupUser = useUserSetup()
    const router = useRouter()
    useAutoLogin(undefined, () => setLoading(false))

    // redirect when authenticated
    useEffect(() => {
        if (
            authenticated &&
            (session.information.emailVerified ||
                session.information.trialActivated ||
                session.information.accountCreatedAt <= emailConfirmationRequiredDate)
        )
            router.replace('/stories')
        else if (
            authenticated &&
            !(
                session.information.emailVerified ||
                session.information.trialActivated ||
                session.information.accountCreatedAt <= emailConfirmationRequiredDate
            )
        ) {
            logWarning('unauthorized session, not logging in')
            setInputsEnabled(true)
            setLoading(false)
        }
    }, [
        authenticated,
        router,
        session.information.emailVerified,
        session.information.emailVerificationLetterSent,
        session.information.trialActivated,
        session.information.accountCreatedAt,
    ])

    // process user data
    const process = async (user: User, email: string) => {
        if (
            !user.information.emailVerified &&
            !user.information.trialActivated &&
            user.information.accountCreatedAt > emailConfirmationRequiredDate
        ) {
            setLocalTrialState(-1)
            router.push('/confirm?email=' + encodeURIComponent(Buffer.from(email).toString('base64')))
            setInputsEnabled(true)
            return
        }
        try {
            setLocalTrialState(-1)
            await setupUser(user, remember)
            setInputsEnabled(true)
        } catch (error: any) {
            switch (error?.message) {
                case 'A mutation operation was attempted on a database that did not allow mutations.':
                    setError('Firefox private browsing is not supported as it does not support IndexedDB.')
                    break
                case 'Invalid Access Key':
                    setError('Invalid email or password.')
                    break
                default:
                    setError(error?.message ?? error)
                    break
            }
            setInputsEnabled(true)
            logError(error, false)
        }
    }

    // process login data
    const login = async (login_data: ILoginData) => {
        if (!login_data.access_key || !login_data.encryption_key) {
            throw new Error('Something went wrong, account data incomplete')
        }
        const user = new User(login_data.session.auth_token, login_data.encryption_key)

        user.subscription = login_data.subscription ?? new UserSubscription()
        user.settings = login_data.settings ?? new UserSettings()
        user.authenticated = login_data.session.authenticated
        user.priority = login_data.priority ?? new UserPriority()
        user.information = login_data.information ?? new UserInformation()

        GlobalUserContext.keystore = login_data.session.keystore

        await process(user, login_data.email || '')
    }

    // process form data
    const authenticate = async (event?: FormEvent) => {
        event?.preventDefault()

        setAccountSelection([])

        const username = rawUsername.toLowerCase()

        if (!username || !password) {
            setError('Email and Password cannot be empty')
            return
        }

        if (!username.includes('@') || username.length <= 3) {
            setError('Email must be in email format')
            return
        }
        if (password.length < 8) {
            setError('Password must be 8 characters or more')
            return
        }

        setInputsEnabled(false)
        setError('')

        await sodium.ready

        removeLocalStorage('session')

        try {
            const login_data = {
                email: undefined,
                access_key: undefined,
                encryption_key: undefined,
                subscription: undefined,
                session: undefined,
                settings: undefined,
                priority: undefined,
                information: undefined,
            } as ILoginData
            const legacy_login_data = {
                email: undefined,
                access_key: undefined,
                encryption_key: undefined,
                subscription: undefined,
                session: undefined,
                settings: undefined,
                priority: undefined,
                information: undefined,
            } as ILoginData

            let loginError

            try {
                login_data.access_key = sodium
                    .crypto_pwhash(
                        64,
                        new Uint8Array(Buffer.from(password)),
                        sodium.crypto_generichash(
                            sodium.crypto_pwhash_SALTBYTES,
                            password.slice(0, 6) + username + 'novelai_data_access_key'
                        ),
                        2,
                        2000000,
                        sodium.crypto_pwhash_ALG_ARGON2ID13,
                        'base64'
                    )
                    .slice(0, 64)
                login_data.encryption_key = sodium.crypto_pwhash(
                    128,
                    new Uint8Array(Buffer.from(password)),
                    sodium.crypto_generichash(
                        sodium.crypto_pwhash_SALTBYTES,
                        password.slice(0, 6) + username + 'novelai_data_encryption_key'
                    ),
                    2,
                    2000000,
                    sodium.crypto_pwhash_ALG_ARGON2ID13,
                    'base64'
                )

                const login = getLoginRequest(login_data.access_key, login_data.encryption_key)

                const result = await login.login()
                login_data.subscription = result.subscription
                login_data.session = result.session
                login_data.settings = result.settings
                login_data.priority = result.priority
                login_data.information = result.information
                login_data.email = username
            } catch (error: any) {
                logInfo(error, false, 'login failed:')
                loginError = error
            }

            // try non-lowercased username
            if (rawUsername !== username) {
                try {
                    legacy_login_data.access_key = sodium
                        .crypto_pwhash(
                            64,
                            new Uint8Array(Buffer.from(password)),
                            sodium.crypto_generichash(
                                sodium.crypto_pwhash_SALTBYTES,
                                password.slice(0, 6) + rawUsername + 'novelai_data_access_key'
                            ),
                            2,
                            2000000,
                            sodium.crypto_pwhash_ALG_ARGON2ID13,
                            'base64'
                        )
                        .slice(0, 64)
                    legacy_login_data.encryption_key = sodium.crypto_pwhash(
                        128,
                        new Uint8Array(Buffer.from(password)),
                        sodium.crypto_generichash(
                            sodium.crypto_pwhash_SALTBYTES,
                            password.slice(0, 6) + rawUsername + 'novelai_data_encryption_key'
                        ),
                        2,
                        2000000,
                        sodium.crypto_pwhash_ALG_ARGON2ID13,
                        'base64'
                    )

                    const login = getLoginRequest(
                        legacy_login_data.access_key,
                        legacy_login_data.encryption_key
                    )

                    const result = await login.login()
                    legacy_login_data.subscription = result.subscription
                    legacy_login_data.session = result.session
                    legacy_login_data.settings = result.settings
                    legacy_login_data.priority = result.priority
                    legacy_login_data.email = rawUsername
                } catch (error: any) {
                    logInfo(error, false, 'legacy login failed:')
                }
            }

            // try login with uppercase first character if both logins failed
            const legacyUsername = username.slice(0, 1).toUpperCase() + username.slice(1)
            if (
                !legacy_login_data.session &&
                !login_data.session &&
                legacyUsername !== rawUsername &&
                legacyUsername !== username
            ) {
                try {
                    if (legacyUsername !== username) {
                        legacy_login_data.access_key = sodium
                            .crypto_pwhash(
                                64,
                                new Uint8Array(Buffer.from(password)),
                                sodium.crypto_generichash(
                                    sodium.crypto_pwhash_SALTBYTES,
                                    password.slice(0, 6) + legacyUsername + 'novelai_data_access_key'
                                ),
                                2,
                                2000000,
                                sodium.crypto_pwhash_ALG_ARGON2ID13,
                                'base64'
                            )
                            .slice(0, 64)
                        legacy_login_data.encryption_key = sodium.crypto_pwhash(
                            128,
                            new Uint8Array(Buffer.from(password)),
                            sodium.crypto_generichash(
                                sodium.crypto_pwhash_SALTBYTES,
                                password.slice(0, 6) + legacyUsername + 'novelai_data_encryption_key'
                            ),
                            2,
                            2000000,
                            sodium.crypto_pwhash_ALG_ARGON2ID13,
                            'base64'
                        )

                        const login = getLoginRequest(
                            legacy_login_data.access_key,
                            legacy_login_data.encryption_key
                        )

                        const result = await login.login()
                        legacy_login_data.subscription = result.subscription
                        legacy_login_data.session = result.session
                        legacy_login_data.settings = result.settings
                        legacy_login_data.priority = result.priority
                        legacy_login_data.email = legacyUsername
                    }
                } catch (error: any) {
                    logInfo(error, false, 'legacy login failed:')
                }
            }

            if (!legacy_login_data.session && !login_data.session) {
                throw loginError
            } else if (legacy_login_data.session && !login_data.session) {
                /* alright, just one account with capitalization */
                login_data.subscription = legacy_login_data.subscription
                login_data.session = legacy_login_data.session
                login_data.settings = legacy_login_data.settings
                login_data.priority = legacy_login_data.priority
                login_data.access_key = legacy_login_data.access_key
                login_data.encryption_key = legacy_login_data.encryption_key
            } else if (!legacy_login_data.session && login_data.session) {
                /* nice, just one correct account */
            } else {
                /* oh no, two correct accounts */
                setAccountSelection([login_data, legacy_login_data])
                setInputsEnabled(true)
                return
            }
            login(login_data)
        } catch (error: any) {
            switch (error?.message) {
                case 'A mutation operation was attempted on a database that did not allow mutations.':
                    setError('Firefox private browsing is not supported as it does not support IndexedDB.')
                    break
                case 'Invalid Access Key':
                    setError('Invalid email or password.')
                    break
                default:
                    setError(error?.message ?? error)
                    break
            }
            setInputsEnabled(true)
            logError(error, false)
        }
    }

    const [rememberMeEnabled, setRememberMeEnabled] = useState(false)
    useEffect(() => setRememberMeEnabled(!!global.window?.localStorage), [])

    function handleKeyDown(e: any) {
        if (e.key === 'Enter' && rawUsername !== '' && password !== '') {
            authenticate()
        }
    }
    return (
        <LoginBox>
            <div>
                <LoginTop>
                    <Link href="/" passHref scroll={false}>
                        <a>
                            <ArrowLeftIcon />
                            Back
                        </a>
                    </Link>
                    <Link href="/register">Sign Up</Link>
                </LoginTop>
                <LoginHeader>
                    <div>Log In</div>
                    <div>Welcome back!</div>
                </LoginHeader>

                <div>
                    <label htmlFor="username">Email</label>
                </div>
                <Field
                    id="username"
                    value={rawUsername}
                    disabled={!inputsEnabled}
                    onChange={(e) => {
                        setAccountSelection([])
                        setUsername(e.target.value)
                    }}
                    onKeyDown={handleKeyDown}
                    type="email"
                    autoComplete="email"
                    required
                />

                <div>
                    <label htmlFor="password">Password</label>
                </div>
                <Field
                    id="password"
                    value={password}
                    disabled={!inputsEnabled}
                    onChange={(e) => {
                        setAccountSelection([])
                        setPassword(e.target.value)
                    }}
                    onKeyDown={handleKeyDown}
                    type="password"
                    autoComplete="current-password"
                    required
                />

                {rememberMeEnabled ? (
                    <Checkbox
                        style={{ flexDirection: 'column', gap: 0, alignItems: 'flex-start' }}
                        alternate={true}
                        label={'Remember me'}
                        value={remember}
                        setValue={setRemember}
                        checkedText={'Login is remembered for 30 days'}
                        uncheckedText={'Login will not be remembered'}
                    />
                ) : null}
                <div style={{ marginBottom: '20px' }}></div>

                {accountSelection.length > 0 ? (
                    <AccountSelection accounts={accountSelection} login={login} />
                ) : (
                    <Submit
                        type="submit"
                        value={inputsEnabled ? 'Sign In' : 'Loading...'}
                        disabled={!inputsEnabled}
                        onClick={authenticate}
                    />
                )}

                <LoginError>
                    {error ? `${error}` : ''}
                    <p>
                        <Link href={`/reset?email=${rawUsername}`}>Reset Password</Link>
                    </p>
                </LoginError>
            </div>

            <LoadingOverlay visible={loading}>
                <LoadingSpinner visible={true} />
            </LoadingOverlay>
        </LoginBox>
    )
}

export const LoadingOverlay = styled.div<{ visible: boolean }>`
    display: ${(props) => (props.visible ? 'flex' : 'none')};
    width: 100%;
    height: 100%;
    position: absolute;
    background: ${(props) => props.theme.colors.bg2};
    opacity: 0.5;
    top: 0;
    left: 0;
    justify-content: center;
    align-items: center;
`

const AccountSelectionBox = styled.div`
    display: flex;
    flex-direction: column;
    gap: 10px;
`
const AccountSelectionInfo = styled.label``
const AccountSelectionElement = styled(Submit)``

function AccountSelection(props: {
    accounts: Array<ILoginData>
    login: (account: ILoginData) => void
}): JSX.Element {
    return (
        <AccountSelectionBox>
            <AccountSelectionInfo>
                We found multiple accounts with your details, select the account you want to log in with
                below.
            </AccountSelectionInfo>
            {props.accounts.map((account) => {
                return (
                    <AccountSelectionElement
                        type="button"
                        key={account.email}
                        value={account.email}
                        onClick={() => props.login(account)}
                    ></AccountSelectionElement>
                )
            })}
        </AccountSelectionBox>
    )
}
