import { useEffect, useRef } from 'react'

import { useRecoilValue } from 'recoil'
import { getLoginRequest } from '../data/request/request'
import { User, UserInformation, UserPriority, UserSubscription } from '../data/user/user'
import { UserSettings } from '../data/user/settings'
import { GlobalUserContext } from '../globals/globals'
import { logError } from '../util/browser'
import { useUserSetup } from '../hooks/useUserSetup'
import { SessionValue } from '../globals/state'
import { getLocalTrialState } from '../components/trialactions'
import { getLocalStorage } from '../util/storage'
import { emailConfirmationRequiredDate } from '../util/util'

export function useAutoLogin(
    onSuccess?: () => void,
    onFail?: (error?: any) => void,
    noAccount?: boolean
): void {
    const setupUser = useUserSetup()
    const authenticated = useRecoilValue(SessionValue('authenticated'))
    const mounted = useRef(false)

    const restore = async (auth_token: string, encryption_key: string) => {
        try {
            const login_data = await getLoginRequest('', encryption_key, auth_token).login()

            const user = new User(auth_token, encryption_key)
            user.subscription = login_data.subscription ?? new UserSubscription()
            user.settings = login_data.settings ?? new UserSettings()
            user.authenticated = login_data.session.authenticated
            user.priority = login_data.priority ?? new UserPriority()
            user.information = login_data.information ?? new UserInformation()
            GlobalUserContext.keystore = login_data.session.keystore

            if (
                !user.information.emailVerified &&
                !user.information.trialActivated &&
                user.information.accountCreatedAt > emailConfirmationRequiredDate
            ) {
                onFail && onFail('email not verified')
                return
            }

            await setupUser(user, false)
            onSuccess && onSuccess()
        } catch (error: any) {
            logError(error, false)
            onFail && onFail(error)
        }
    }

    const noAccountLogin = async () => {
        const user = new User('', '', true)

        user.subscription = new UserSubscription()
        const localSettingsString = getLocalStorage('noAccountSettings')
        user.settings = new UserSettings()
        try {
            user.settings = {
                ...user.settings,
                ...(localSettingsString ? JSON.parse(localSettingsString) : {}),
            }
        } catch (error: any) {
            logError(error)
        }
        user.settings.remoteDefault = false
        user.authenticated = true
        user.priority = new UserPriority()
        await setupUser(user)
    }

    useEffect(() => {
        if (authenticated) {
            onSuccess && onSuccess()
            mounted.current = true
            return
        }
        const sessionString = getLocalStorage('session')
        if (!sessionString) {
            if (noAccount && !mounted.current && getLocalTrialState() > 0) {
                noAccountLogin()
                mounted.current = true
                return
            }
            onFail && onFail('no session')
            mounted.current = true
            return
        }
        const session = JSON.parse(sessionString)
        if (!session || !session.auth_token || !session.encryption_key) {
            onFail && onFail('invalid session')
            mounted.current = true
            return
        }
        restore(session.auth_token, session.encryption_key)
        mounted.current = true
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [authenticated])
}
