import React, { useMemo } from 'react'
import { Text, View } from 'react-native'
import { useRecoilValue } from 'recoil'
import { SelectorSection } from '../components/modals/selectormodal'
import { normalizeModel, TextGenerationModel } from '../../shared/data/request/model'
import { getModelPresets, getModelLoreGenPresets } from '../../shared/data/story/defaultpresets'
import { StoryPreset } from '../../shared/data/story/storysettings'
import { GlobalUserContext } from '../../shared/globals/globals'
import { SelectedStoryId, UserPresets } from '../../shared/globals/state'
import { modelsHaveSamePresets, modelName } from '../../shared/util/models'
import { BodyLarge600, BodyMedium400Faded, BodySmall400Faded } from '../styles/fonts'

function label(preset: StoryPreset, compatible: boolean = true) {
    const warn =
        (preset.parameters.repetition_penalty_frequency ?? 0) > 0 ||
        (preset.parameters.repetition_penalty_presence ?? 0) > 0
    return (
        <View style={{ padding: 10 }}>
            <BodyLarge600>{preset.name}</BodyLarge600>
            {!!preset.description && <BodyMedium400Faded>{preset.description}</BodyMedium400Faded>}
            {warn ? (
                <BodySmall400Faded>
                    Contains hidden experimental settings. Not recommended as a base for new presets.
                </BodySmall400Faded>
            ) : null}
            {!compatible ? (
                <BodySmall400Faded>
                    Preset was created for {modelName(normalizeModel(preset.model))} and might not work as
                    intended with this model.
                </BodySmall400Faded>
            ) : null}
        </View>
    )
}

export function filterPresets(model: TextGenerationModel): (preset: StoryPreset) => boolean {
    return (preset: StoryPreset) => modelsHaveSamePresets(preset.model, model)
}

export function usePresetOptions(
    model: TextGenerationModel,
    checkStory: boolean = false,
    loreGen: boolean = false
): SelectorSection[] {
    const userPresets = useRecoilValue(UserPresets)
    const selectedStory = useRecoilValue(SelectedStoryId)

    const currentStoryContent = GlobalUserContext.storyContentCache.get(selectedStory)
    const options = useMemo(() => {
        const defaultPresets = [...getModelPresets(model)]
        const loreGenPresets = [...getModelLoreGenPresets(model)]

        const options: SelectorSection[] = []
        if (loreGen) {
            options.push({
                title: 'Lore Generation',
                data: loreGenPresets.map((preset) => ({
                    value: preset.id,
                    description: preset.name + (preset.description ? `: ${preset.description}` : ''),
                    rawLabel: preset.name,
                    label: label(preset),
                })),
            })
        }

        options.push({
            title: 'User',
            data: userPresets.filter(filterPresets(model)).map((preset) => ({
                value: preset.id,
                description: preset.name + (preset.description ? `: ${preset.description}` : ''),
                rawLabel: preset.name,
                label: label(preset),
            })),
        })
        if (checkStory && currentStoryContent?.scenarioPreset) {
            options.push({
                title: 'Scenario',
                data: [currentStoryContent.scenarioPreset].map((preset) => ({
                    value: preset.id,
                    description: preset.name + (preset.description ? `: ${preset.description}` : ''),
                    rawLabel: preset.name,
                    label: label(
                        preset,
                        modelsHaveSamePresets(
                            currentStoryContent.settings.model,
                            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                            currentStoryContent.scenarioPreset!.model
                        )
                    ),
                })),
            })
        }
        options.push({
            title: 'Default',
            data: defaultPresets.map((preset) => ({
                value: preset.id,
                description: preset.name + (preset.description ? `: ${preset.description}` : ''),
                rawLabel: preset.name,
                label: label(preset),
            })),
        })
        return options
    }, [
        checkStory,
        currentStoryContent?.scenarioPreset,
        currentStoryContent?.settings.model,
        loreGen,
        model,
        userPresets,
    ])
    return options
}
