﻿import React, { useEffect } from 'react'
import { useRecoilState, useSetRecoilState } from 'recoil'
import { User } from '../../../shared/data/user/user'
import {
    IPLimitModal,
    Session,
    SiteTheme,
    SubscriptionDialogOpen,
    TrialUsageRemaining,
    TrialUsedModal,
} from '../../../shared/globals/state'
import { Dark } from '../../../shared/styles/themes/dark'
import { BodyMedium400, Bold } from '../../styles/fonts'
import { getLocalStorage, setLocalStorage } from '../../util/overrides'
import {
    AlternateSubmitButton,
    SubmitButtonText,
    SubtleButton,
    SubtleButtonText,
} from '../common/button.style'
import { HoverModal } from './common'
import { ModalBackgroundSplash } from './common.style'
import {
    BigInkwell,
    ModalHeading,
    StyledModalCloseButton,
    UsageBarInner,
    UsageBarOuter,
    UsageContainer,
    UsageText,
    UsageTitle,
    UsageTopRow,
} from './trialactions.style'

export function setLocalTrialState(value: number): void {
    setLocalStorage('localTrialState', Buffer.from(value.toString(), 'utf8').toString('base64'))
}

export function getLocalTrialState(): number {
    const number = Number.parseInt(
        Buffer.from(getLocalStorage('localTrialState') ?? '0', 'base64').toString('utf8')
    )
    return Number.isNaN(number) ? -1 : number
}

export default function TrialUsageDisplay(): JSX.Element {
    const [value, setValue] = useRecoilState(TrialUsageRemaining)
    const [trialModalOpen, setTrialModalOpen] = useRecoilState(TrialUsedModal)
    const [ipModalOpen, setIPModalOpen] = useRecoilState(IPLimitModal)
    const [session, setSession] = useRecoilState(Session)
    const setSubscriptionVisible = useSetRecoilState(SubscriptionDialogOpen)
    const setTheme = useSetRecoilState(SiteTheme)

    useEffect(() => {
        let remaining = 0
        remaining = session.noAccount ? getLocalTrialState() : session.information.trialActionsLeft
        if (remaining === 0) {
            setTrialModalOpen(true)
        }
        setValue(remaining)
    }, [session.information.trialActionsLeft, session.noAccount, setTrialModalOpen, setValue])

    useEffect(() => {
        if (session.noAccount) {
            setLocalTrialState(value)
        }
    }, [session.noAccount, value])

    return (
        <>
            <DryInkwellPrompt
                visible={trialModalOpen && session.noAccount}
                heading={'The inkwell’s run dry.'}
                onConfirm={() => {
                    setSession(new User('', ''))
                    setTheme(Dark)
                }}
                onClose={() => setTrialModalOpen(false)}
            >
                Create an account with us and we’ll give you another 50 actions for free to play around with.
            </DryInkwellPrompt>
            <DryInkwellPrompt
                visible={trialModalOpen && !session.noAccount}
                heading={'The inkwell’s run dry once more.'}
                confirmText={'Take me to the Pricing'}
                onConfirm={() => {
                    setSubscriptionVisible({ open: true, blocked: false })
                }}
                onClose={() => setTrialModalOpen(false)}
            >
                Your actions have run out again. No worries though! If you subscribe you can continue writing
                to your heart’s content.
            </DryInkwellPrompt>
            <DryInkwellPrompt
                visible={ipModalOpen}
                heading={'IP Limit Reached'}
                onConfirm={() => {
                    setSession(new User('', ''))
                    setTheme(Dark)
                }}
                onClose={() => setIPModalOpen(false)}
            >
                The limit of anonymous generations for your IP address has been reached. Create an account to
                continue with another 50 free actions.
            </DryInkwellPrompt>
            {session.information.trialActivated || session.noAccount ? (
                <UsageContainer>
                    <UsageTopRow>
                        <UsageTitle>Free Trial</UsageTitle>
                        <UsageText>
                            You have <Bold>{Math.max(value, 0)}/50 actions left</Bold>
                        </UsageText>
                    </UsageTopRow>
                    <UsageBarOuter>
                        <UsageBarInner width={(Math.max(value, 0) / 50) * 100}></UsageBarInner>
                    </UsageBarOuter>
                </UsageContainer>
            ) : (
                <UsageContainer>
                    <UsageText>
                        To activate the free trial and gain 50 free actions, please verify your email in
                        account settings.
                    </UsageText>
                </UsageContainer>
            )}
        </>
    )
}

function DryInkwellPrompt(props: {
    visible: boolean
    heading: string
    children: string
    confirmText?: string
    cancelText?: string
    onConfirm: () => void
    onClose: () => void
}): JSX.Element {
    return (
        <HoverModal
            visible={props.visible}
            onRequestClose={props.onClose}
            innerStyle={{ alignItems: 'center', width: '88%' }}
        >
            <StyledModalCloseButton onClose={props.onClose} />
            <ModalBackgroundSplash style={{ marginTop: -20 }} />
            <BigInkwell />
            <ModalHeading>{props.heading}</ModalHeading>
            <BodyMedium400 style={{ textAlign: 'center', padding: 10 }}>{props.children}</BodyMedium400>
            <AlternateSubmitButton
                style={{ marginHorizontal: 20 }}
                onPress={() => {
                    props.onConfirm()
                    props.onClose()
                }}
            >
                <SubmitButtonText>{props.confirmText ? props.confirmText : 'Sign Up'}</SubmitButtonText>
            </AlternateSubmitButton>
            <SubtleButton onPress={props.onClose}>
                <SubtleButtonText>{props.cancelText ? props.cancelText : 'No thanks.'}</SubtleButtonText>
            </SubtleButton>
        </HoverModal>
    )
}
