﻿import React, { useState, useMemo, useEffect } from 'react'
import { Modal, View } from 'react-native'
import { useRecoilValue, useSetRecoilState, useRecoilCallback } from 'recoil'
import { StoryId, StoryMetadata, StoryChildContent } from '../../../shared/data/story/storycontainer'
import { GlobalUserContext } from '../../../shared/globals/globals'
import {
    Session,
    StoryShelves,
    Stories,
    StoryUpdate,
    StorySort,
    SiteTheme,
} from '../../../shared/globals/state'
import { useDebounce } from '../../../shared/hooks/useDebounce'
import { getStorage } from '../../../shared/data/storage/storage'
import { SearchFilter, sortStoryMetadata } from '../../../shared/data/storage/search'
import { BodyMedium400 } from '../../styles/fonts'
import { Tags } from '../common/infotags'
import SelectorModal, { SelectorSection } from './selectormodal'
import { InputSelected } from '../../globals/state'
import { SectionTitle, TitleBox, DescriptionBox } from './storymetadatamodal.style'
import { StoryDeleteButton } from './deletemodal'
import { ModalTitleRow } from './common'
import { AlternateSubmitButton, SubmitButtonText } from '../common/button.style'
import { FullModalView } from './common.style'

const storyFilter = new SearchFilter()

const NoShelf = {
    value: '',
    description: 'No Shelf',
    rawLabel: 'No Shelf',
    label: 'No Shelf',
}

export default function StoryMetadataModal(props: {
    visible: boolean
    closeModal: () => void
    id: StoryId
}): JSX.Element {
    const currentStory = GlobalUserContext.stories.get(props.id)
    const session = useRecoilValue(Session)
    const shelves = useRecoilValue(StoryShelves)
    const setStories = useSetRecoilState(Stories)
    const updateStory = useRecoilCallback(({ set }) => (metadata: StoryMetadata) => {
        set(StoryUpdate(metadata.id), metadata.save())
    })

    const setInputSelected = useSetRecoilState(InputSelected)

    const [titleInput, setTitleInput, updateTitleInput] = useDebounce(
        currentStory?.title ?? '',
        (s: string) => {
            if (currentStory) {
                currentStory.title = s
                updateStory(currentStory)
            }
        }
    )

    const [descriptionInput, setDescriptionInput, updateDescriptionInput] = useDebounce(
        currentStory?.description ?? '',
        (d: string) => {
            if (currentStory) {
                currentStory.description = d
                updateStory(currentStory)
            }
        }
    )

    const shelf = useMemo(
        () =>
            [...GlobalUserContext.shelves.entries()].find(
                ([, shelf]) => !!shelf.children?.find((child) => child.id === props.id)
            )?.[1] ?? null,
        [props.id]
    )

    const [selectedShelf, setSelectedShelf] = useState(
        shelf
            ? {
                  value: shelf.id,
                  description: shelf.title,
                  rawLabel: shelf.title,
                  label: shelf.title,
              }
            : NoShelf
    )

    const updateShelf = () => {
        const oldShelf = shelf
        const newShelf = GlobalUserContext.shelves.get(selectedShelf.value)

        if (oldShelf) {
            oldShelf.children = oldShelf.children?.filter((child) => child.id !== props.id)
            getStorage(session).saveStoryShelf(oldShelf)
        }
        if (newShelf) {
            newShelf.children = [...(newShelf.children ?? []), { id: props.id, type: 'story' }]
            getStorage(session).saveStoryShelf(newShelf)
        }
        setStories((stories) => [...stories])
    }

    const updateSelectedShelf = (shelf?: StoryMetadata) => {
        setSelectedShelf(
            shelf
                ? {
                      value: shelf.id,
                      description: shelf.title,
                      rawLabel: shelf.title,
                      label: shelf.title,
                  }
                : NoShelf
        )
    }

    useEffect(() => {
        setDescriptionInput(currentStory?.description ?? '')
        setTitleInput(currentStory?.title ?? '')
    }, [currentStory?.description, currentStory?.title, setDescriptionInput, setTitleInput])

    const sortValue = useRecoilValue(StorySort)
    const shelfElements: StoryChildContent[] = []

    for (const shelf of shelves) {
        let found = false
        for (const shelfInner of shelves) {
            if (
                GlobalUserContext.shelves
                    .get(shelfInner)
                    ?.children?.filter((child) => child.type === 'shelf')
                    .map((child) => child.id)
                    .includes(shelf)
            ) {
                found = true
                break
            }
        }
        if (!found) {
            shelfElements.push({ type: 'shelf', id: shelf } as StoryChildContent)
        }
    }

    const results = storyFilter.metadataMatch(shelfElements, '')

    sortStoryMetadata(results, sortValue)

    const shelfOptions: SelectorSection[] = useMemo(
        () => [
            {
                title: '',
                data: [
                    NoShelf,
                    ...[...results].map(({ metadata }) => {
                        return {
                            value: metadata.id,
                            description: metadata.title ?? 'unknown',
                            rawLabel: metadata.title ?? 'unknown',
                            label: metadata.title ?? 'unknown',
                        }
                    }),
                ],
            },
        ],
        // eslint-disable-next-line react-hooks/exhaustive-deps
        [shelves]
    )
    const siteTheme = useRecoilValue(SiteTheme)

    const onClose = () => {
        currentStory && updateStory(currentStory)
        props.closeModal()
        updateShelf()
    }

    return (
        <Modal visible={props.visible} onRequestClose={onClose} animationType={'fade'}>
            <ModalTitleRow title={'Story Settings'} onClose={onClose} />
            <FullModalView>
                <View>
                    <SectionTitle>Story Title</SectionTitle>
                    <TitleBox
                        aria-label="Story Title"
                        value={titleInput}
                        onChangeText={(text: string) => updateTitleInput(text)}
                    />
                    <SectionTitle>Description</SectionTitle>
                    <DescriptionBox
                        value={descriptionInput}
                        onChangeText={(text: string) => updateDescriptionInput(text)}
                        onFocus={() => setInputSelected(true)}
                        multiline={true}
                        blurOnSubmit={true}
                    />
                    <SectionTitle>Shelf</SectionTitle>
                    <SelectorModal
                        aria-label="Select a shelf for this story"
                        data={shelfOptions}
                        selectedID={selectedShelf.value}
                        setID={(id) => updateSelectedShelf(GlobalUserContext.shelves.get(id))}
                    />
                    <SectionTitle style={{ paddingBottom: 0 }}>Search Tags</SectionTitle>
                    <BodyMedium400>Type in the box below and submit to save.</BodyMedium400>
                    <Tags />
                </View>
                <View style={{ marginBottom: 20 }}>
                    <AlternateSubmitButton onPress={onClose}>
                        <SubmitButtonText>Save and Close</SubmitButtonText>
                    </AlternateSubmitButton>
                    <StoryDeleteButton id={props.id} />
                </View>
            </FullModalView>
        </Modal>
    )
}
