﻿import React, { useRef, useState } from 'react'
import { Modal, View } from 'react-native'
import { useRecoilValue, useRecoilCallback, useSetRecoilState } from 'recoil'
import { getStorage } from '../../../shared/data/storage/storage'
import { StoryId } from '../../../shared/data/story/storycontainer'
import { GlobalUserContext } from '../../../shared/globals/globals'
import { SelectedShelf, SelectedStoryId, Session, StoryShelves } from '../../../shared/globals/state'
import { FocusedMenubarElement, InputSelected } from '../../globals/state'
import { AlternateSubmitButton, SubmitButtonText, WarningButton } from '../common/button.style'
import { ModalTitleRow } from './common'
import { FullModalView } from './common.style'
import DeleteModal from './deletemodal'
import { DescriptionBox, SectionTitle, TitleBox } from './storymetadatamodal.style'

export function ShelfMetadataModal(props: {
    id: StoryId
    visible: boolean
    closeModal: () => void
}): JSX.Element {
    const currentShelf = GlobalUserContext.shelves.get(props.id)
    const session = useRecoilValue(Session)

    const oldTitle = useRef(currentShelf?.title ?? '')
    const oldDescription = useRef(currentShelf?.description ?? '')

    const [titleInput, updateTitleInput] = useState(currentShelf?.title ?? '')
    const [descriptionInput, updateDescriptionInput] = useState(currentShelf?.description ?? '')

    const selectedStoryId = useRecoilValue(SelectedStoryId)
    const setFocusedElement = useSetRecoilState(FocusedMenubarElement)
    const setInputSelected = useSetRecoilState(InputSelected)

    const updateShelf = useRecoilCallback(({ set }) => () => {
        if (oldTitle.current === titleInput && oldDescription.current === descriptionInput) {
            return
        }
        set(StoryShelves, (shelves) => [...shelves])
        if (currentShelf) {
            getStorage(session).saveStoryShelf(currentShelf)
        }
    })

    const onClose = () => {
        updateShelf()
        props.closeModal()
    }

    const deleteShelf = useRecoilCallback(({ set }) => () => {
        if (currentShelf) getStorage(session).deleteStoryShelf(currentShelf)
        set(StoryShelves, (shelves) => [...shelves].filter((shelf) => shelf !== props.id))
        set(SelectedShelf, '')
        setFocusedElement(selectedStoryId)
        props.closeModal()
    })

    const [deleteOpen, setDeleteOpen] = useState(false)

    return (
        <Modal visible={props.visible} onRequestClose={onClose} animationType={'fade'}>
            <ModalTitleRow title={'Shelf Settings'} onClose={onClose} />
            <FullModalView>
                <View>
                    <SectionTitle>Shelf Name</SectionTitle>
                    <TitleBox
                        aria-label="Shelf Name"
                        value={titleInput}
                        onChangeText={(text: string) => {
                            updateTitleInput(text)
                            if (currentShelf) {
                                currentShelf.title = text
                            }
                        }}
                    />
                    <SectionTitle>Description</SectionTitle>
                    <DescriptionBox
                        value={descriptionInput}
                        onChangeText={(text: string) => {
                            updateDescriptionInput(text)
                            if (currentShelf) {
                                currentShelf.description = text
                            }
                        }}
                        onFocus={() => setInputSelected(true)}
                        multiline={true}
                        blurOnSubmit={true}
                    />
                </View>
                <View style={{ marginBottom: 20 }}>
                    <AlternateSubmitButton onPress={onClose}>
                        <SubmitButtonText>Save and Close</SubmitButtonText>
                    </AlternateSubmitButton>
                    <WarningButton onPress={() => setDeleteOpen(true)}>
                        <SubmitButtonText>Delete Shelf</SubmitButtonText>
                    </WarningButton>
                </View>
                <DeleteModal
                    label={'Delete Shelf?'}
                    description={`Really delete the Shelf ${currentShelf?.title}?\nContained stories will be moved out of the shelf.`}
                    confirmText={'Delete Shelf'}
                    visible={deleteOpen}
                    closeModal={() => setDeleteOpen(false)}
                    onConfirm={deleteShelf}
                />
            </FullModalView>
        </Modal>
    )
}
