﻿import React, { ReactElement, useRef, useState } from 'react'

import {
    Tabs as StyledTabs,
    TabHeaderList,
    TabContent,
    Tab as StyledTab,
    TabHeader as StyledTabHeader,
    TabHeaderContent,
} from './tabs.styled'
import { BodyLarge700 } from '../../styles/fonts'
import { useRecoilValue } from 'recoil'
import { SiteTheme } from '../../../shared/globals/state'
import { SvgImageData } from '../common/icons'
import { Animated, LayoutChangeEvent } from 'react-native'

interface TabProps {
    title: string
    icon?: SvgImageData
    children: ReactElement[] | ReactElement
}

export default function Tabs(props: {
    children: ReactElement[]
    selected?: number
    padded?: boolean
    setSelected?: (i: number) => void
}): JSX.Element {
    const [tabIndex, setTabIndex] = useState(props.selected ?? 0)

    return (
        <StyledTabs>
            <TabHeaderList padded={props.padded}>
                {props.children.map((tab, index) => {
                    return (
                        <TabHeader
                            key={index}
                            title={tab.props.title}
                            icon={tab.props.icon}
                            onPress={() => {
                                setTabIndex(index)
                                props.setSelected && props.setSelected(index)
                            }}
                            selected={index === tabIndex}
                        />
                    )
                })}
            </TabHeaderList>
            <TabContent>
                {props.children.map((tab, index) => {
                    return (
                        <StyledTab key={index} visible={index === tabIndex}>
                            {tab}
                        </StyledTab>
                    )
                })}
            </TabContent>
        </StyledTabs>
    )
}

export function Tab(props: TabProps): JSX.Element {
    return <>{props.children}</>
}

export function TabHeader(props: {
    title: string
    icon?: SvgImageData
    onPress: () => void
    selected?: boolean
    animationData?: { index: number; position: Animated.Value | Animated.AnimatedInterpolation }
}): JSX.Element {
    const siteTheme = useRecoilValue(SiteTheme)

    return (
        <StyledTabHeader
            onPress={() => props.onPress()}
            narrow={!!props.icon}
            selected={props.selected ?? false}
        >
            <Animated.View
                style={
                    props.animationData
                        ? {
                              opacity: props.animationData.position.interpolate({
                                  inputRange: [
                                      props.animationData.index - 2,
                                      props.animationData.index - 1,
                                      props.animationData.index,
                                      props.animationData.index + 1,
                                      props.animationData.index + 2,
                                  ],
                                  outputRange: [0.3, 0.3, 1, 0.3, 0.3],
                              }),
                          }
                        : { opacity: props.selected ? 1 : 0.3 }
                }
            >
                {!!props.icon ? (
                    <props.icon primary={siteTheme.colors.textMain} height={20} width={20} />
                ) : (
                    <BodyLarge700>{props.title}</BodyLarge700>
                )}
            </Animated.View>
            {props.animationData ? (
                <Animated.View
                    style={{
                        opacity: props.animationData.position.interpolate({
                            inputRange: [
                                props.animationData.index - 1,
                                props.animationData.index,
                                props.animationData.index + 1,
                            ],
                            outputRange: [0, 1, 0],
                        }),
                        position: 'absolute',
                        backgroundColor: siteTheme.colors.bg2,
                        width: '100%',
                        height: '100%',
                        zIndex: -1,
                        top: 0,
                        left: 0,
                    }}
                />
            ) : (
                <></>
            )}
        </StyledTabHeader>
    )
}
