﻿import React, { useEffect, useMemo, useState } from 'react'
import { Pressable, View } from 'react-native'
import Collapsible from 'react-native-collapsible'
import Icon from 'react-native-vector-icons/MaterialIcons'
import { useRecoilValue, useSetRecoilState } from 'recoil'
import { setMaxLength, SetterPackage, updateStory } from '../../../shared/component-logic/optionslogic'
import { modelRepPenMax, modelRepPenStepSize } from '../../../shared/data/ai/model'
import { DefaultModel } from '../../../shared/data/request/model'
import { getModelPresets } from '../../../shared/data/story/defaultpresets'
import { LogitWarper, StorySettings, TextGenerationSettings } from '../../../shared/data/story/storysettings'
import { GlobalUserContext } from '../../../shared/globals/globals'
import {
    SelectedStoryLoaded,
    Session,
    SiteTheme,
    StoryStateValue,
    StoryUpdate,
} from '../../../shared/globals/state'
import { useReload } from '../../../shared/hooks/useReload'
import { BodyLarge600, BodyMedium600, Heading } from '../../styles/fonts'
import { getLocalStorage } from '../../util/overrides'
import Checkbox from '../common/checkbox'
import EditorCard, { MainSettingSliderCard, MinorSettingSliderCard } from '../common/editorcard'
import { EditorCardDescription } from '../common/editorcard.style'
import { useStoryPresetSelect } from '../presetselect'
import { SidebarPlaceholder } from './infobar'
import { SidebarElementsContainer } from './infobar.style'

export function CollapsableMenu(props: {
    name: string
    children: React.ReactNode
    open: boolean
    setOpen: (open: boolean) => void
}): JSX.Element {
    const siteTheme = useRecoilValue(SiteTheme)
    useEffect(() => console.log(props.open), [props.open])
    return (
        <>
            <Pressable
                onPress={() => props.setOpen(!props.open)}
                style={{ flexDirection: 'row', paddingTop: 17, paddingLeft: 12 }}
            >
                <BodyLarge600>
                    <Heading>{props.name}</Heading>
                </BodyLarge600>
                {props.open ? (
                    <Icon
                        name={'arrow-drop-up'}
                        size={34}
                        style={{ color: siteTheme.colors.textHeadings, bottom: 5 }}
                    />
                ) : (
                    <Icon
                        name={'arrow-drop-down'}
                        size={34}
                        style={{ color: siteTheme.colors.textHeadings, bottom: 5 }}
                    />
                )}
            </Pressable>
            <Collapsible collapsed={!props.open} align={'bottom'}>
                {props.children}
            </Collapsible>
        </>
    )
}

export default function SlidersTab(props: { selectedStory: string }): JSX.Element {
    const session = useRecoilValue(Session)
    useRecoilValue(SelectedStoryLoaded)

    const [showAdvancedSettings, setShowAdvancedSettings] = useState(false)
    const [showRegularSettings, setShowRegularSettings] = useState(true)

    useEffect(() => {
        getLocalStorage('advancedSettingsShown').then((value) => setShowAdvancedSettings(value === 'true'))
        getLocalStorage('regularSettingsShown').then((value) => setShowRegularSettings(value !== 'false'))
    }, [])

    const currentStoryContent = GlobalUserContext.storyContentCache.get(props.selectedStory)

    const settings = currentStoryContent?.settings

    const { presetSelect, currentPreset, setUserPresets, setPreset, userPresets, defaultPresets } =
        useStoryPresetSelect(
            () => {},
            () => {}
        )

    let combinedPresets = [...defaultPresets, ...userPresets]

    //preset manipulation functions go here

    const defaultSettings =
        combinedPresets.find((preset) => preset.id === currentPreset.id)?.parameters ??
        getModelPresets(currentStoryContent?.settings.model ?? DefaultModel)[0].parameters

    if (!settings) return <SidebarPlaceholder status={''} />

    return (
        <SidebarElementsContainer>
            <View style={{ paddingVertical: 20 }}>
                <View style={{ paddingHorizontal: 10 }}>{presetSelect}</View>
                <CollapsableMenu
                    name={'Generation Options'}
                    open={showRegularSettings}
                    setOpen={setShowRegularSettings}
                >
                    <RegularSettings selectedStory={props.selectedStory} defaultSettings={defaultSettings} />
                </CollapsableMenu>
                <CollapsableMenu
                    name={'Advanced Options'}
                    open={showAdvancedSettings}
                    setOpen={setShowAdvancedSettings}
                >
                    <AdvancedSettings selectedStory={props.selectedStory} defaultSettings={defaultSettings} />
                </CollapsableMenu>
            </View>
        </SidebarElementsContainer>
    )
}

//Exact same as it is on web. Could be shared.
function RegularSettings(props: { selectedStory: string; defaultSettings: TextGenerationSettings }) {
    const defaultSettings = props.defaultSettings
    const currentStory = GlobalUserContext.stories.get(props.selectedStory)
    const currentStoryContent = GlobalUserContext.storyContentCache.get(props.selectedStory)
    const session = useRecoilValue(Session)
    const settings = currentStoryContent?.settings
    const model = settings?.model

    const genSettings = useMemo(
        () => settings?.parameters ?? new TextGenerationSettings(),
        [settings?.parameters]
    )
    const setStoryUpdate = useSetRecoilState(StoryUpdate(''))

    const update = useReload()
    const setterPackage: SetterPackage = useMemo(
        () => ({
            currentStory: currentStory,
            currentStoryContent: currentStoryContent,
            genSettings: genSettings,
            updateState: (
                valOrUpdater: StoryStateValue | ((currVal: StoryStateValue) => StoryStateValue)
            ) => {
                setStoryUpdate(valOrUpdater)
                update()
            },
        }),
        [currentStory, currentStoryContent, genSettings, setStoryUpdate, update]
    )

    const randomness = useMemo(() => {
        return (
            <MainSettingSliderCard
                title={'Randomness'}
                hint={'Default: ' + defaultSettings.temperature}
                tooltip={'The higher the value, the more random the output!'}
                onHintClick={() =>
                    updateStory(() => (genSettings.temperature = defaultSettings.temperature), setterPackage)
                }
                min="0.1"
                max="2.5"
                step="0.01"
                value={genSettings.temperature}
                onChange={(e) => updateStory(() => (genSettings.temperature = e), setterPackage)}
            ></MainSettingSliderCard>
        )
    }, [defaultSettings.temperature, genSettings, setterPackage])

    const outputLength = useMemo(() => {
        return (
            <MainSettingSliderCard
                title={'Output Length'}
                hint={'Default: ' + defaultSettings.max_length * 4}
                tooltip={'Increase the length of the generated responses'}
                onHintClick={() => setMaxLength(defaultSettings.max_length, setterPackage)}
                min={4}
                max={session.subscription.tier >= 3 ? 600 : 400}
                step={4}
                value={genSettings.max_length * 4}
                suffix={() => `Characters`}
                prefix={() => `~`}
                onChange={(e) => setMaxLength(Math.ceil(e / 4), setterPackage)}
                preventDecimal={true}
                forceStep={true}
            ></MainSettingSliderCard>
        )
    }, [defaultSettings.max_length, genSettings.max_length, session.subscription.tier, setterPackage])

    const repetitionPenalty = useMemo(() => {
        return (
            <MainSettingSliderCard
                title={'Repetition Penalty'}
                hint={'Default: ' + defaultSettings.repetition_penalty}
                tooltip={'Higher values make the output less repetitive.'}
                onHintClick={() =>
                    updateStory(() => {
                        genSettings.repetition_penalty = defaultSettings.repetition_penalty
                    }, setterPackage)
                }
                min="1.0"
                max={modelRepPenMax(model)}
                step={modelRepPenStepSize(model)}
                value={genSettings.repetition_penalty}
                onChange={(e) => updateStory(() => (genSettings.repetition_penalty = e), setterPackage)}
                uncapMin={true}
                uncapMax={true}
            ></MainSettingSliderCard>
        )
    }, [defaultSettings.repetition_penalty, genSettings, model, setterPackage])

    if (settings === undefined || currentStory === undefined || genSettings === undefined) {
        return <></>
    }

    return (
        <>
            {randomness}
            {outputLength}
            {repetitionPenalty}
        </>
    )
}

//Only differs from web in its return statement.
function AdvancedSettings(props: { selectedStory: string; defaultSettings: TextGenerationSettings }) {
    const defaultSettings = props.defaultSettings
    const currentStory = GlobalUserContext.stories.get(props.selectedStory)
    const currentStoryContent = GlobalUserContext.storyContentCache.get(props.selectedStory)
    const settings = useMemo(
        () => currentStoryContent?.settings ?? new StorySettings(),
        [currentStoryContent?.settings]
    )
    const genSettings = useMemo(
        () => settings?.parameters ?? new TextGenerationSettings(),
        [settings?.parameters]
    )
    const setStoryUpdate = useSetRecoilState(StoryUpdate(''))

    const update = useReload()
    const setterPackage: SetterPackage = useMemo(
        () => ({
            currentStory: currentStory,
            currentStoryContent: currentStoryContent,
            genSettings: genSettings,
            updateState: (
                valOrUpdater: StoryStateValue | ((currVal: StoryStateValue) => StoryStateValue)
            ) => {
                setStoryUpdate(valOrUpdater)
                update()
            },
        }),
        [currentStory, currentStoryContent, genSettings, setStoryUpdate, update]
    )

    const topk = useMemo(() => {
        return (
            <MinorSettingSliderCard
                key="Top K"
                title={`Top K`}
                hint={'Default: ' + (defaultSettings.top_k > 0 ? defaultSettings.top_k : '0 (off)')}
                tooltip={`Increases consistency by only selecting the most likely\
                            tokens and redistributing the probabilities. \nLower settings\
                            create a smaller pool of tokens, trading creativity for consistency.`}
                onHintClick={() =>
                    updateStory(() => (genSettings.top_k = defaultSettings.top_k), setterPackage)
                }
                suffix={(v) => (v > 0 ? '' : '(off)')}
                min="0"
                max="150"
                step="1"
                value={genSettings.top_k}
                onChange={(e) => updateStory(() => (genSettings.top_k = e), setterPackage)}
                preventDecimal={true}
                uncapMax={true}
            />
        )
    }, [defaultSettings.top_k, genSettings, setterPackage])

    const nucleus = useMemo(() => {
        return (
            <MinorSettingSliderCard
                key="Nucleus"
                title={`Nucleus`}
                hint={'Default: ' + (defaultSettings.top_p < 1 ? defaultSettings.top_p : '1 (off)')}
                tooltip={`Increases consistency by taking tokens from the top and\
                adding up their probabilities until it reaches the percentage\
                set. \nLower settings create a smaller pool of tokens, trading\
                creativity for consistency.`}
                onHintClick={() =>
                    updateStory(() => (genSettings.top_p = defaultSettings.top_p), setterPackage)
                }
                suffix={(v) => (v < 1 ? '' : '(off)')}
                min="0.05"
                max="1.0"
                step="0.025"
                value={genSettings.top_p}
                onChange={(e) => updateStory(() => (genSettings.top_p = e), setterPackage)}
            />
        )
    }, [defaultSettings.top_p, genSettings, setterPackage])

    const tailFree = useMemo(() => {
        return (
            <MinorSettingSliderCard
                key="Tail-Free"
                title={`Tail-Free`}
                hint={
                    'Default: ' +
                    (defaultSettings.tail_free_sampling < 1 ? defaultSettings.tail_free_sampling : '1 (off)')
                }
                tooltip={`Increases the consistency of the output by working from the\
                bottom and trimming the 'worst' possible tokens. \nGenerally\
                has a smaller effect on creativity than other sampling types.\
                Experiment!`}
                onHintClick={() =>
                    updateStory(
                        () => (genSettings.tail_free_sampling = defaultSettings.tail_free_sampling),
                        setterPackage
                    )
                }
                suffix={(v) => (v < 1 ? '' : '(off)')}
                min="0.05"
                max="1.00"
                step="0.001"
                value={genSettings.tail_free_sampling}
                onChange={(e) => updateStory(() => (genSettings.tail_free_sampling = e), setterPackage)}
                roundDigits={3}
            />
        )
    }, [defaultSettings.tail_free_sampling, genSettings, setterPackage])

    const topA = useMemo(() => {
        return (
            <MinorSettingSliderCard
                key="Top A"
                title={`Top A`}
                hint={'Default: ' + (defaultSettings.top_a < 1 ? defaultSettings.top_a : '1 (off)')}
                tooltip={`Increases the consistency of the output by removing unlikely tokens\
                    based on the highest token probability.`}
                onHintClick={() =>
                    updateStory(() => (genSettings.top_a = defaultSettings.top_a), setterPackage)
                }
                suffix={(v) => (v < 1 ? '' : '(off)')}
                min="0.05"
                max="1.00"
                step="0.025"
                value={genSettings.top_a}
                onChange={(e) => updateStory(() => (genSettings.top_a = e), setterPackage)}
                roundDigits={3}
            />
        )
    }, [defaultSettings.top_a, genSettings, setterPackage])

    const typicalP = useMemo(() => {
        return (
            <MinorSettingSliderCard
                key="Typical"
                title={`Typical`}
                hint={'Default: ' + (defaultSettings.typical_p < 1 ? defaultSettings.typical_p : '1 (off)')}
                tooltip={`Selects tokens according to the expected amount of information they contribute.`}
                onHintClick={() =>
                    updateStory(() => (genSettings.typical_p = defaultSettings.typical_p), setterPackage)
                }
                suffix={(v) => (v < 1 ? '' : '(off)')}
                min="0.05"
                max="1.00"
                step="0.025"
                value={genSettings.typical_p}
                onChange={(e) => updateStory(() => (genSettings.typical_p = e), setterPackage)}
                roundDigits={3}
            />
        )
    }, [defaultSettings.typical_p, genSettings, setterPackage])

    const [drprInput, setDrprInput] = useState(false)
    useEffect(() => setDrprInput(settings?.dynamicPenaltyRange ?? false), [settings?.dynamicPenaltyRange])
    const repPenRange = useMemo(() => {
        return (
            <MinorSettingSliderCard
                key="Range"
                title={`Range`}
                hint={
                    'Default: ' +
                    (defaultSettings.repetition_penalty_range != 0
                        ? defaultSettings.repetition_penalty_range
                        : '0 (off)')
                }
                tooltip="How many tokens, starting from the last generated one, will
                be considered repeated if they appear in the next output."
                onHintClick={() => {
                    if (defaultSettings.repetition_penalty_range)
                        updateStory(
                            () =>
                                (genSettings.repetition_penalty_range =
                                    defaultSettings.repetition_penalty_range),
                            setterPackage
                        )
                }}
                disabled={drprInput === true}
                suffix={(v) => (v != 0 ? '' : '(off)')}
                min="0"
                max="2048"
                step="16"
                value={genSettings.repetition_penalty_range}
                onChange={(e) => updateStory(() => (genSettings.repetition_penalty_range = e), setterPackage)}
                preventDecimal={true}
            />
        )
    }, [defaultSettings.repetition_penalty_range, drprInput, genSettings, setterPackage])

    const drpr = useMemo(() => {
        return (
            <EditorCard style={{ marginTop: 0, marginBottom: '-15px' }}>
                <Checkbox
                    label={'Dynamic Range'}
                    alternate={true}
                    value={drprInput ?? false}
                    setValue={(e) => {
                        setDrprInput(e)
                        updateStory(() => (settings.dynamicPenaltyRange = e), setterPackage)
                    }}
                    checkedText={'The repetition penalty range is dynamic, only apply to "Story" text'}
                    uncheckedText={'The repetition penalty range is a fixed number of tokens'}
                />
            </EditorCard>
        )
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [setterPackage, settings, settings.dynamicPenaltyRange])

    const repPenSlope = useMemo(() => {
        return (
            <MinorSettingSliderCard
                key="Slope"
                title={`Slope`}
                hint={
                    'Default: ' +
                    (defaultSettings.repetition_penalty_slope != 0
                        ? defaultSettings.repetition_penalty_slope
                        : '0 (off)')
                }
                tooltip={`Affects the ramping of the penalty's harshness, starting\
                from the final token. \nHigher values penalize the final\
                tokens more harshly, but are softer on the earlier tokens.\
                Lower values cause a smoother reduction of probability across all tokens.`}
                onHintClick={() => {
                    if (defaultSettings.repetition_penalty_slope)
                        updateStory(
                            () =>
                                (genSettings.repetition_penalty_slope =
                                    defaultSettings.repetition_penalty_slope),
                            setterPackage
                        )
                }}
                suffix={(v) => (v != 0 ? '' : '(off)')}
                min="0.00"
                max="10.00"
                step="0.09"
                value={genSettings.repetition_penalty_slope}
                onChange={(e) => updateStory(() => (genSettings.repetition_penalty_slope = e), setterPackage)}
            />
        )
    }, [defaultSettings.repetition_penalty_slope, genSettings, setterPackage])

    const repPenPresence = useMemo(() => {
        return (
            <MinorSettingSliderCard
                key="Presence"
                title={`Presence`}
                hint={
                    'Default: ' +
                    (defaultSettings.repetition_penalty_presence != 0
                        ? defaultSettings.repetition_penalty_presence
                        : '0 (off)')
                }
                tooltip={`Applies a static penalty to the generation of tokens that appear within\
                the Repetition Penalty Range.
                This penalty is applied separately from the Repetition Penalty setting and is not affected by\
                Repetition Penalty Slope.`}
                onHintClick={() => {
                    updateStory(
                        () =>
                            (genSettings.repetition_penalty_presence =
                                defaultSettings.repetition_penalty_presence ?? 0),
                        setterPackage
                    )
                }}
                suffix={(v) => (v != 0 ? '' : '(off)')}
                min="0.00"
                max="5.00"
                step="0.05"
                value={genSettings.repetition_penalty_presence}
                onChange={(e) =>
                    updateStory(() => (genSettings.repetition_penalty_presence = e), setterPackage)
                }
            />
        )
    }, [defaultSettings.repetition_penalty_presence, genSettings, setterPackage])

    const repPenFrequency = useMemo(() => {
        return (
            <MinorSettingSliderCard
                key="Frequency"
                title={`Frequency`}
                hint={
                    'Default: ' +
                    (defaultSettings.repetition_penalty_frequency != 0
                        ? defaultSettings.repetition_penalty_frequency
                        : '0 (off)')
                }
                tooltip={`Applies a penalty to the generation of tokens based on the number of occurrences of that token within\
                the Repetition Penalty Range.
                This penalty is applied separately from the Repetition Penalty setting and is not affected by\
                Repetition Penalty Slope.`}
                onHintClick={() => {
                    updateStory(
                        () =>
                            (genSettings.repetition_penalty_frequency =
                                defaultSettings.repetition_penalty_frequency ?? 0),
                        setterPackage
                    )
                }}
                suffix={(v) => (v != 0 ? '' : '(off)')}
                min="0.00"
                max="1.00"
                step="0.01"
                value={genSettings.repetition_penalty_frequency}
                onChange={(e) =>
                    updateStory(() => (genSettings.repetition_penalty_frequency = e), setterPackage)
                }
            />
        )
    }, [defaultSettings.repetition_penalty_frequency, genSettings, setterPackage])

    const [orderModalOpen, setOrderModalOpen] = useState(false)
    const siteTheme = useRecoilValue(SiteTheme)

    if (settings === undefined || currentStory === undefined || genSettings === undefined) {
        return <></>
    }
    const hiddenSettingsCount = genSettings.order.reduce((a, b) => (b.enabled ? a : a + 1), 0)

    //Nothing modified until this line

    return (
        <>
            {genSettings.order.filter((setting) => setting.enabled).length > 1 && (
                <EditorCardDescription style={{ paddingLeft: 10, paddingTop: 20, paddingBottom: 0 }}>
                    <BodyMedium600>Sampling</BodyMedium600>
                </EditorCardDescription>
            )}
            {genSettings.order
                .filter((setting) => setting.enabled)
                .map((setting) => {
                    switch (setting.id) {
                        case LogitWarper.TopK:
                            return topk
                        case LogitWarper.TopP:
                            return nucleus
                        case LogitWarper.TFS:
                            return tailFree
                        case LogitWarper.TopA:
                            return topA
                        case LogitWarper.TypicalP:
                            return typicalP
                    }
                })}
        </>
    )
}
