import React, { lazy, Suspense, useMemo } from 'react'
import { useRecoilValue } from 'recoil'
import { Animated, Dimensions } from 'react-native'
import { SafeAreaProvider } from 'react-native-safe-area-context'
import LinearGradient from 'react-native-linear-gradient'
import {
    Infobar,
    InnerHoverBar,
    OuterHoverBar,
    SidebarElementsContainer,
    SidebarPlaceholder as StyledSidebarPlaceholder,
} from './infobar.style'
import { SelectedStoryId, SiteTheme } from '../../../shared/globals/state'
import { BodyLarge700 } from '../../styles/fonts'
import SlidersIcon from '../../assets/images/icons/sliders.svg'
import { TabHeader } from './tabs'
import { NavigationHelpers, ParamListBase, TabActions } from '@react-navigation/native'
import { StoryAppScreenParams } from '../storyapp'
import { TabHeaderList } from './tabs.styled'
import { SvgImageData } from '../common/icons'
import { hexOpacity } from '../../util/colors'
import { MaterialTopTabNavigationEventMap } from '@react-navigation/material-top-tabs/lib/typescript/src/types'
import { GlobalUserContext } from '../../../shared/globals/globals'

const Story = lazy(() => import('./storytab'))
const Advanced = lazy(() => import('./advancedtab'))
const Sliders = lazy(() => import('./sliderstab'))

export function SidebarPlaceholder(props: { status: string }): JSX.Element {
    return (
        <SidebarElementsContainer>
            <StyledSidebarPlaceholder>
                <BodyLarge700>{props.status}</BodyLarge700>
            </StyledSidebarPlaceholder>
        </SidebarElementsContainer>
    )
}

const tabs: { title: string; icon?: SvgImageData }[] = [
    { title: 'Story' },
    { title: 'Advanced' },
    { title: 'Sliders', icon: SlidersIcon },
]

export default function InfoBar(props: {
    name: keyof StoryAppScreenParams
    element: React.LazyExoticComponent<(props: { selectedStory: string }) => JSX.Element>
    bordered?: boolean
}): JSX.Element {
    //should use a recoil state for story loaded instead of this
    const selectedStoryId = useRecoilValue(SelectedStoryId)
    const currentStoryContent = GlobalUserContext.storyContentCache.get(selectedStoryId)
    const settings = currentStoryContent?.settings

    const siteTheme = useRecoilValue(SiteTheme)
    const FadeSeparator = useMemo(
        () => (
            <LinearGradient
                start={{ x: 0, y: 1 }}
                end={{ x: 0, y: 0 }}
                locations={[0, 0.5, 0.7, 0.8, 1]}
                colors={[
                    hexOpacity(siteTheme.colors.bg2, 0),
                    hexOpacity(siteTheme.colors.bg2, 0.75),
                    hexOpacity(siteTheme.colors.bg2, 0.9),
                    hexOpacity(siteTheme.colors.bg2, 0.95),
                    siteTheme.colors.bg2,
                ]}
                style={{ position: 'absolute', top: 60, left: 0, width: '100%', height: 20, zIndex: 10 }}
            />
        ),
        [siteTheme]
    )

    return (
        <SafeAreaProvider style={{ flex: 1 }}>
            <Infobar bordered={props.bordered ?? !!settings}>
                {FadeSeparator}
                <Suspense fallback={<SidebarPlaceholder status={'Loading...'} />}>
                    <props.element selectedStory={selectedStoryId} />
                </Suspense>
            </Infobar>
        </SafeAreaProvider>
    )
}

export function StoryTab(): JSX.Element {
    return <InfoBar name={'infobar_story'} element={Story} bordered={true} />
}

export function AdvancedTab(): JSX.Element {
    return <InfoBar name={'infobar_advanced'} element={Advanced} />
}

export function SlidersTab(): JSX.Element {
    return <InfoBar name={'infobar_sliders'} element={Sliders} />
}

export function HoverBar(props: {
    navigationIndex: number
    position: Animated.Value | Animated.AnimatedInterpolation
    navigation: NavigationHelpers<ParamListBase, MaterialTopTabNavigationEventMap>
}): JSX.Element {
    const width = Dimensions.get('window').width
    const selectedStoryId = useRecoilValue(SelectedStoryId)
    const currentStoryContent = GlobalUserContext.storyContentCache.get(selectedStoryId)
    const settings = currentStoryContent?.settings

    return (
        <Animated.View
            style={[
                props.position
                    ? {
                          transform: [
                              {
                                  translateX: props.position.interpolate({
                                      inputRange: [1, 2, 4],
                                      outputRange: [width, 0, 0],
                                  }),
                              },
                          ],
                      }
                    : undefined,
                {
                    position: 'absolute',
                    width: width - 2,
                    top: 0,
                    left: 1,
                },
            ]}
        >
            <OuterHoverBar>
                <InnerHoverBar>
                    <TabHeaderList padded={false}>
                        {tabs.map((tab, index) => {
                            return (
                                <TabHeader
                                    key={index}
                                    title={tab.title}
                                    icon={tab.icon}
                                    animationData={{
                                        index: index,
                                        position: props.position.interpolate({
                                            inputRange: [2, 4],
                                            outputRange: [0, 2],
                                        }),
                                    }}
                                    onPress={() => {
                                        props.navigation.dispatch(
                                            TabActions.jumpTo(
                                                (('infobar_' +
                                                    tab.title.toLowerCase()) as keyof StoryAppScreenParams) ??
                                                    'infobar_story'
                                            )
                                        )
                                    }}
                                />
                            )
                        })}
                    </TabHeaderList>
                </InnerHoverBar>
                {!settings && <SidebarPlaceholder status={'No Story selected.'} />}
            </OuterHoverBar>
        </Animated.View>
    )
}
