import React, { useMemo, useState } from 'react'
import { View } from 'react-native'
import { useRecoilCallback, useRecoilState, useRecoilValue } from 'recoil'
import { setMinLength, SetterPackage, updateStory } from '../../../shared/component-logic/optionslogic'
import { modelSupportsPhraseBias } from '../../../shared/data/ai/model'
import { DefaultModel } from '../../../shared/data/request/model'
import { Scenario } from '../../../shared/data/story/scenario'
import { GlobalUserContext } from '../../../shared/globals/globals'
import {
    CustomModules,
    SelectedStoryLoaded,
    Session,
    StoryStateValue,
    StoryUpdate,
} from '../../../shared/globals/state'
import { useReload } from '../../../shared/hooks/useReload'
import { getModelEncoderType } from '../../tokenizer/encoder'
import { downloadTextFile } from '../../util/browser'
import { LightColorButton, RegularButtonText } from '../common/button.style'
import Checkbox from '../common/checkbox'
import { FlexRow } from '../common/common.style'
import EditorCard, { MinorSettingSliderCard } from '../common/editorcard'
import { scenarioStarter } from '../modals/storyimporter'
import { SidebarPlaceholder } from './infobar'
import { SidebarElementsContainer } from './infobar.style'
import { ContextViewerButtons } from './items/contextviewer'
import { EphemeralContextDisplay } from './items/ephemeralcontext'
import { TokenInput } from './items/tokeninput'

export function PhraseBiasCard(props: { selectedStory: string }): JSX.Element {
    const [, setStoryUpdate] = useRecoilState(StoryUpdate(props.selectedStory))

    const currentStory = GlobalUserContext.stories.get(props.selectedStory)
    const currentStoryContent = GlobalUserContext.storyContentCache.get(props.selectedStory)

    const settings = currentStoryContent?.settings
    const genSettings = settings?.parameters

    const [selectedGroup, setSelectedGroup] = useState(
        currentStoryContent?.phraseBiasGroups?.length ?? -1 > 0 ? 0 : -1
    )

    const setterPackage: SetterPackage = {
        currentStory: currentStory,
        currentStoryContent: currentStoryContent,
        genSettings: genSettings,
        updateState: setStoryUpdate,
    }

    if (settings === undefined || currentStory === undefined || genSettings === undefined) {
        return <></>
    }

    const downloadLogitBias = () => {
        if (!currentStoryContent?.phraseBiasGroups) return
        const json = {
            phraseBiasGroups: currentStoryContent?.phraseBiasGroups,
        }
        for (const group of json.phraseBiasGroups) {
            for (const phrase of group.phrases) {
                phrase.tokens = undefined
            }
        }
        downloadTextFile(
            JSON.stringify(json, undefined, '  '),
            `${currentStory.title.slice(0, 40)} (${new Date().toISOString()}).bias`
        )
    }

    return (
        <EditorCard
            title="Phrase Bias"
            hint="Export"
            description="Weigh the AI’s chance of generating certain words or phrases."
            tooltip={`Set a bias on words or phrases to increase or decrease their chance of being generated.\
                      Surround with {curly braces} to input exact text.\
                      Surround with [square brackets] to input token ids (tokenizer specific)`}
            onHintClick={currentStoryContent?.phraseBiasGroups ? downloadLogitBias : undefined}
        >
            <></>
        </EditorCard>
    )
}

export function BannedTokensCard(props: { selectedStory: string }): JSX.Element {
    const currentStory = GlobalUserContext.stories.get(props.selectedStory)
    const currentStoryContent = GlobalUserContext.storyContentCache.get(props.selectedStory)
    const settings = currentStoryContent?.settings
    const genSettings = settings?.parameters
    const [, setStoryUpdate] = useRecoilState(StoryUpdate(props.selectedStory))

    const setterPackage: SetterPackage = {
        currentStory: currentStory,
        currentStoryContent: currentStoryContent,
        genSettings: genSettings,
        updateState: setStoryUpdate,
    }

    const [selectedGroup, setSelectedGroup] = useState(
        currentStoryContent?.bannedSequenceGroups?.length ?? -1 > 0 ? 0 : -1
    )

    if (settings === undefined || currentStory === undefined || genSettings === undefined) {
        return <></>
    }

    const downloadBannedTokens = () => {
        if (!currentStoryContent?.bannedSequenceGroups) return
        const json = {
            bannedSequenceGroups: currentStoryContent?.bannedSequenceGroups,
        }

        for (const group of json.bannedSequenceGroups ?? []) {
            for (const sequence of group.sequences) {
                sequence.tokens = undefined
            }
        }

        downloadTextFile(
            JSON.stringify(json, undefined, '  '),
            `${currentStory.title.slice(0, 40)} (${new Date().toISOString()}).badwords`
        )
    }

    return (
        <>
            <EditorCard
                title="Banned Tokens"
                hint="Export"
                description="Prevents certain token sequences from being generated."
                tooltip={`Ban sequences of tokens from being generated.\
                          Input regularly to automatically ban variants.\
                          Surround with {curly braces} to input exact text.\
                          Surround with [square brackets] to input token ids (tokenizer specific)`}
                onHintClick={currentStoryContent?.bannedSequenceGroups ? downloadBannedTokens : undefined}
            >
                <></>
            </EditorCard>
            <EditorCard>
                <Checkbox
                    label={'Ban Bracket Generation'}
                    alternate={true}
                    value={settings.banBrackets}
                    setValue={(newValue) =>
                        updateStory(() => (settings.banBrackets = newValue), setterPackage)
                    }
                    checkedText={"Tokens containing brackets won't be generated"}
                    uncheckedText={'Tokens containing brackets can be generated'}
                />
            </EditorCard>
        </>
    )
}

export function ImportAndStart(props: { selectedStory: string }): JSX.Element {
    const currentStory = GlobalUserContext.stories.get(props.selectedStory)
    const [importScenarioModalVisible, setImportScenarioModalVisible] = useState(false)
    const [importedScenario, setImportedScenario] = useState(new Scenario())

    const startAsScenario = useRecoilCallback(({ snapshot }) => async () => {
        if (!currentStory) return
        const customModules = await snapshot.getPromise(CustomModules)

        scenarioStarter(currentStory, customModules, setImportedScenario, setImportScenarioModalVisible)
    })

    return (
        <EditorCard
            title="Duplicate and Start as Scenario"
            description={'Imports current story as a scenario with placeholders'}
        >
            <FlexRow>
                <LightColorButton onPress={startAsScenario} style={{ flexGrow: 1 }}>
                    <RegularButtonText>Duplicate and Start as Scenario</RegularButtonText>
                </LightColorButton>
                {/*<ImportScenarioModal
                    visible={importScenarioModalVisible}
                    setVisible={setImportScenarioModalVisible}
                    importedScenario={importedScenario}
                />*/}
            </FlexRow>
        </EditorCard>
    )
}

export function EndOfSampling(props: { selectedStory: string }): JSX.Element {
    const [, setStoryUpdate] = useRecoilState(StoryUpdate(props.selectedStory))
    const currentStory = GlobalUserContext.stories.get(props.selectedStory)
    const session = useRecoilValue(Session)

    const currentStoryContent = GlobalUserContext.storyContentCache.get(props.selectedStory)
    const settings = currentStoryContent?.settings
    const update = useReload()

    const genSettings = settings?.parameters
    // eslint-disable-next-line react-hooks/exhaustive-deps
    const setterPackage: SetterPackage = {
        currentStory: currentStory,
        currentStoryContent: currentStoryContent,
        genSettings: genSettings,
        updateState: (valOrUpdater: StoryStateValue | ((currVal: StoryStateValue) => StoryStateValue)) => {
            setStoryUpdate(valOrUpdater)
            update()
        },
    }

    const minOutput = useMemo(() => {
        return (
            <MinorSettingSliderCard
                title={'Min Output Length'}
                hint={'Apply Default: 1'}
                tooltip={`The minimum generated output length in tokens,\
                            which are 4-5 characters long on average.\nOnly has an effect if\
                            at least one Stop Sequence is set.`}
                onHintClick={() => setMinLength(1, setterPackage)}
                min={1}
                max={session.subscription.tier >= 3 ? 150 : 100}
                step={1}
                value={genSettings?.min_length ?? 1}
                suffix={(value: number) => (value === 1 ? 'Token' : 'Tokens')}
                onChange={(value) => setMinLength(value, setterPackage)}
                disabled={
                    currentStoryContent?.eosSequences === undefined ||
                    currentStoryContent?.eosSequences.length < 1
                }
                preventDecimal={true}
            />
        )
    }, [currentStoryContent?.eosSequences, genSettings?.min_length, session.subscription.tier, setterPackage])

    const encoderType = getModelEncoderType(currentStoryContent?.settings.model ?? DefaultModel)
    const eos = useMemo(() => {
        return (
            <EditorCard
                title={'Stop Sequences: '}
                tooltip={`Cuts generation short upon reaching a specified token sequence.
        Note: if under minimum output length (${
            genSettings?.min_length ?? 1
        }), generation will not be interrupted.
        Surround with [square brackets] to input token ids (tokenizer specific)
       `}
            >
                <TokenInput
                    encoderType={encoderType}
                    placeholder={'Type your stop sequence here'}
                    onTokenSubmit={(token) => {
                        if (!currentStoryContent) return
                        updateStory(() => (currentStoryContent.eosSequences = token), setterPackage)
                    }}
                    eosSequences={currentStoryContent?.eosSequences ?? []}
                />
            </EditorCard>
        )
    }, [genSettings?.min_length, encoderType, currentStoryContent, setterPackage])

    return (
        <>
            {eos}
            {minOutput}
        </>
    )
}

export default function AdvancedTab(props: { selectedStory: string }): JSX.Element {
    useRecoilValue(SelectedStoryLoaded)

    const currentStoryContent = GlobalUserContext.storyContentCache.get(props.selectedStory)
    const settings = currentStoryContent?.settings

    if (!settings) return <SidebarPlaceholder status={''} />

    return (
        <SidebarElementsContainer>
            <EditorCard title="Context" description="Get a full view of what’s sent to the AI.">
                <>
                    <ContextViewerButtons />
                    <EphemeralContextDisplay />
                </>
            </EditorCard>
            {modelSupportsPhraseBias(currentStoryContent?.settings.model) ? (
                <PhraseBiasCard selectedStory={props.selectedStory} />
            ) : (
                <></>
            )}
            <View>
                <BannedTokensCard selectedStory={props.selectedStory} />
            </View>
            <EndOfSampling selectedStory={props.selectedStory} />
            <ImportAndStart selectedStory={props.selectedStory} />
            <View style={{ flexGrow: 1, flexShrink: 1, flexBasis: 0, minHeight: 20 }} />
        </SidebarElementsContainer>
    )
}
