import React, { MutableRefObject, useEffect, useRef, useState, useLayoutEffect } from 'react'
import { Animated, Text, View } from 'react-native'
import { useRecoilState, useRecoilValue, useSetRecoilState } from 'recoil'
import {
    GenerationRequestActive,
    GenerationRequestError,
    InputModes,
    SelectedInputMode,
    LorebookOpen,
    SelectedStory,
    Session,
    SiteTheme,
    GenerationRequestCancelled,
} from '../../../shared/globals/state'
import {
    ConversationControlsContent,
    ConversationControls as StyledConversationControls,
    ConversationInput,
    RedoContainer,
    RequestErrorInfo,
    RedoMenu,
    RedoOption,
    ControlButton,
    ConversationControlsGroup,
    RedoCounterView,
    RedoCounterText,
    UndoMenuArrowUpIcon,
    UndoMenuArrowDownIcon,
    ControlText,
    ConversationInputContainer,
    ConversationStoryControls,
} from './conversationcontrols.style'
import { MinusIcon, PlusIcon } from '../common/icons'
import Book from '../../assets/images/book.svg'
import Send from '../../assets/images/send.svg'
//import CrossRounded from '../assets/images/cross-rounded.svg'
import Reload from '../../assets/images/reload.svg'
import Undo from '../../assets/images/undo.svg'
import Redo from '../../assets/images/redo.svg'
import { DataOrigin, StoryMode } from '../../../shared/data/story/story'
import { InputSelected } from '../../globals/state'
import { SvgProps } from 'react-native-svg'
import { BodyMedium600 } from '../../styles/fonts'

function formatRequestError(error: string): string {
    if (
        `${error}`.toLowerCase() === 'error: timeout' ||
        `${error}`.toLowerCase() === 'timeout' ||
        `${error}`.toLowerCase().includes('timeout fetching')
    ) {
        return 'Error: Timeout - Unable to reach NovelAI servers. Please wait for a moment and try again'
    }
    if (`${error}`.toLowerCase().includes('worker timeout')) {
        return `${error} - Please try again, or if this issue persists, \
            try restarting your browser or clearing your browser cache (after backing up your local stories)`
    }
    return error
}

export function ConversationControls(props: {
    onUndoRequest: () => void
    onRedoRequest: (index: number | undefined) => void
    handleCancelRequest: () => void
    onInputReceived: (input: string, origin?: DataOrigin) => Promise<void>
    onRetryRequest: () => void
    onRetryHover: (hovered: boolean) => void
    onUndoHover: (hovered: boolean) => void

    visible: boolean
    redoOptions: string[][][]
    canUndo: boolean
    canRetry: boolean
    mode: StoryMode
}): JSX.Element {
    const [text, setText] = useState('')
    const [error, setError] = useRecoilState(GenerationRequestError)
    const generationRequestActive = useRecoilValue(GenerationRequestActive)
    const generationRequestCancelled = useRecoilValue(GenerationRequestCancelled)
    const setLorebookVisible = useSetRecoilState(LorebookOpen)
    const [inputSelected, setInputSelected] = useRecoilState(InputSelected)
    const slideAnim = useRef(new Animated.Value(1)).current

    const contentRef = useRef<any>(null)
    const [margin, setMargin] = useState(0)

    const [redoMenuVisible, setRedoMenuVisible] = useState(false)
    const session = useRecoilValue(Session)
    const selectedStory = useRecoilValue(SelectedStory)
    const inputWindow = useRef<any>(null)
    const redoMenu = useRef<any>(null)
    const inputModes = useRecoilValue(InputModes)
    const [inputMode, setInputMode] = useRecoilState(SelectedInputMode)
    const theme = useRecoilValue(SiteTheme)

    const showRedoMenu = () => {
        setRedoMenuVisible(true)
    }

    const hideRedoMenu = () => {
        setRedoMenuVisible(false)
    }

    const toggleRedoMenu = () => {
        if (generationRequestActive || props.redoOptions.length === 0 || !props.visible) {
            return false
        }

        if (!redoMenuVisible) {
            showRedoMenu()
            if (!props.canUndo) {
                props.onUndoHover(false)
            }
        } else {
            hideRedoMenu()
        }

        return true
    }

    const clickCancel = () => {
        if (!generationRequestActive) {
            return
        }
        props.handleCancelRequest()
    }
    const clickSend = () => {
        setError('')
        if (generationRequestActive) {
            return
        }
        props.onInputReceived(text, DataOrigin.user).catch((error: any) => {
            setError(`${error}`)
        })
        setText('')
    }
    useEffect(() => {
        if (generationRequestActive) {
            setError('')
        }
    }, [generationRequestActive, setError])

    React.useEffect(() => {
        !inputSelected
            ? Animated.timing(slideAnim, {
                  toValue: 1,
                  duration: 50,
                  useNativeDriver: true,
              }).start()
            : slideAnim.setValue(0)
    }, [inputSelected])

    return (
        <Animated.View
            style={{
                transform: !inputSelected
                    ? [{ translateY: slideAnim.interpolate({ inputRange: [0, 1], outputRange: [30, 0] }) }]
                    : undefined,
            }}
        >
            <ConversationControlsContent
                ref={contentRef}
                //style={{ marginBottom: margin }}
                visible={props.visible}
                reversed={props.mode === StoryMode.adventure}
            >
                <RequestErrorInfo visible={error.length > 0}>{formatRequestError(error)}</RequestErrorInfo>
                <StyledConversationControls>
                    <ConversationControlsGroup side={'left'}>
                        <IconControlButton
                            iconUrl={Undo}
                            swap={true}
                            text="Undo"
                            style={{ marginLeft: 0 }}
                            onPress={() => {
                                props.onUndoRequest()
                            }}
                            disabled={generationRequestActive || !props.canUndo || !props.visible}
                        ></IconControlButton>
                        <RedoContainer>
                            <IconControlButton
                                iconUrl={Redo}
                                swap={true}
                                text="Redo"
                                onPress={() => {
                                    props.onRedoRequest(void 0)
                                    if (!props.canUndo) {
                                        props.onUndoHover(false)
                                    }
                                }}
                                disabled={
                                    generationRequestActive ||
                                    props.redoOptions.length === 0 ||
                                    !props.visible
                                }
                            ></IconControlButton>

                            <RedoCounterView
                                onPress={(e) => {
                                    e.stopPropagation()
                                    toggleRedoMenu()
                                }}
                                disabled={
                                    generationRequestActive ||
                                    props.redoOptions.length === 0 ||
                                    !props.visible
                                }
                            >
                                {props.redoOptions.length === 0 ? (
                                    <UndoMenuArrowDownIcon />
                                ) : (
                                    <UndoMenuArrowUpIcon />
                                )}
                                <RedoCounterText displayToggle={props.redoOptions.length === 0}>
                                    {props.redoOptions.length}
                                </RedoCounterText>
                            </RedoCounterView>

                            {redoMenuVisible ? (
                                <RedoMenu ref={redoMenu}>
                                    {props.redoOptions.map((text, index) => {
                                        return (
                                            <RedoOption
                                                key={index}
                                                onPress={() => props.onRedoRequest(index)}
                                            >
                                                {text[0].length === 1 && text[0][0].trim() === '' ? (
                                                    <></>
                                                ) : (
                                                    <>
                                                        <View>
                                                            <MinusIcon></MinusIcon>
                                                            <Text>{text[0].join('... ')}</Text>
                                                        </View>
                                                    </>
                                                )}
                                                {text[1].length === 1 && text[1][0].trim() === '' ? (
                                                    <></>
                                                ) : (
                                                    <>
                                                        <View>
                                                            <PlusIcon></PlusIcon>
                                                            <Text>{text[1].join('... ')}</Text>
                                                        </View>
                                                    </>
                                                )}
                                            </RedoOption>
                                        )
                                    })}
                                </RedoMenu>
                            ) : (
                                <></>
                            )}
                        </RedoContainer>
                        <IconControlButton
                            text="Lorebook"
                            iconUrl={Book}
                            showText={false}
                            onPress={() => setLorebookVisible(true)}
                            aria-label="Lorebook"
                        ></IconControlButton>
                    </ConversationControlsGroup>
                    <ConversationControlsGroup side={'right'}>
                        <IconControlButton
                            text="Retry"
                            iconUrl={Reload}
                            onPress={() => {
                                props.onRetryRequest()
                                if (!props.canUndo) {
                                    props.onRetryHover(false)
                                }
                            }}
                            disabled={generationRequestActive || !props.canRetry || !props.visible}
                        ></IconControlButton>
                        <IconControlButton
                            text="Send"
                            showText={true}
                            style={{ paddingLeft: 16, paddingRight: 16, paddingTop: 10, paddingBottom: 10 }}
                            iconStyle={{ maxHeight: 12, marginTop: 5 }}
                            iconUrl={Send}
                            onPress={() => clickSend()}
                            disabled={generationRequestActive || !props.visible}
                            highlight={true}
                            fill={theme.colors.textHeadings}
                        ></IconControlButton>
                    </ConversationControlsGroup>
                </StyledConversationControls>
                {session.settings.showInputBox || props.mode === StoryMode.adventure ? (
                    <ConversationInputContainer mode={props.mode} inputModeIcon={inputMode.icon}>
                        <ConversationInput
                            ref={inputWindow}
                            editable={props.visible}
                            value={text}
                            multiline={true}
                            blurOnSubmit={true}
                            onFocus={() => setInputSelected(true)}
                            placeholder={
                                props.mode === StoryMode.adventure
                                    ? inputMode.placeholderText
                                    : 'Write your input here'
                            }
                            onChange={(e) => setText(e.nativeEvent.text)}
                        />
                        {props.mode === StoryMode.adventure ? (
                            <ConversationStoryControls>
                                {inputModes.map((mode, index) => {
                                    return (
                                        <ControlButton
                                            key={index}
                                            aria-label={mode.name}
                                            onPress={() => {
                                                setInputMode(mode)
                                            }}
                                            style={{
                                                backgroundColor:
                                                    inputMode === mode ? theme.colors.bg3 : theme.colors.bg0,
                                                borderColor:
                                                    inputMode === mode
                                                        ? theme.colors.textHeadings
                                                        : theme.colors.textMain,
                                            }}
                                        >
                                            <BodyMedium600>{mode.name}</BodyMedium600>
                                        </ControlButton>
                                    )
                                })}
                            </ConversationStoryControls>
                        ) : (
                            <></>
                        )}
                    </ConversationInputContainer>
                ) : (
                    <></>
                )}
            </ConversationControlsContent>
        </Animated.View>
    )
}

function IconControlButton(props: {
    highlight?: boolean
    showText?: boolean
    text: string
    iconUrl: React.FC<SvgProps>
    onPress: () => void
    disabled?: boolean
    style?: any
    iconStyle?: any
    swap?: boolean
    fill?: string
}): JSX.Element {
    const theme = useRecoilValue(SiteTheme)
    return (
        <ControlButton
            aria-label={props.text}
            style={props.style}
            disabled={props.disabled}
            onPress={props.onPress}
        >
            {props.showText ?? false ? (
                <ControlText highlight={props.highlight}>{props.text}</ControlText>
            ) : (
                <></>
            )}
            <props.iconUrl
                style={[props.swap ?? false ? { width: '0.9rem' } : {}, { marginTop: 3 }, props.iconStyle]}
                height={14}
                width={20}
                fill={props.fill ?? theme.colors.textMain}
            />
        </ControlButton>
    )
}
