import React, { MutableRefObject, useEffect, useRef, useState } from 'react'
import { Gesture, GestureDetector } from 'react-native-gesture-handler'
import Animated, {
    Easing,
    Extrapolate,
    interpolate,
    runOnJS,
    useAnimatedStyle,
    useSharedValue,
    withTiming,
} from 'react-native-reanimated'
import styled, { css } from 'styled-components/native'

const SliderThumb = styled(Animated.View)<{ fat: boolean; hidden?: boolean }>`
    ${(props) => props.hidden && 'display: none;'};
    ${(props) =>
        props.fat
            ? css`
                  height: 26px;
                  width: 14px;
              `
            : css`
                  height: 18px;
                  width: 12px;
              `}
    z-index: 100;
`

const SliderThumbVisual = styled(Animated.View)<{ focused: boolean }>`
    background-color: ${(props) =>
        props.focused ? props.theme.colors.textHeadings : props.theme.colors.bg3};
    height: 100%;
    width: 100%;
    position: absolute;
`

const SliderRail = styled(Animated.View)<{ fat: boolean }>`
    background-color: ${(props) => props.theme.colors.bg0};
    ${(props) =>
        props.fat
            ? css`
                  height: 26px;
                  top: 8px;
              `
            : css`
                  height: 12px;
                  top: 11px;
              `}
    width: 100%;
    position: absolute;
    left: 0;
`

const SliderBody = styled(Animated.View)`
    padding-vertical: 8px
    width: 100%;
`

function AnimatedThumb(props: {
    setThumbStateRef: MutableRefObject<
        React.Dispatch<
            React.SetStateAction<{
                focused: boolean
                backupPosition: number
            }>
        >
    >
    position: Animated.SharedValue<number>
    fat?: boolean
    onLayoutChange: () => void
    sliderWidth: MutableRefObject<number>
    thumbWidth: MutableRefObject<number>
}): JSX.Element {
    const [state, setState] = useState({ focused: false, backupPosition: 0 })
    props.setThumbStateRef.current = setState

    //if interpolation is not done here, the refs will be incorrect
    const positionStyle = useAnimatedStyle(() => ({
        transform: [
            {
                translateX: interpolate(
                    state.focused ? props.position.value : state.backupPosition,
                    [0, props.sliderWidth.current - props.thumbWidth.current],
                    [0, props.sliderWidth.current - props.thumbWidth.current],
                    Extrapolate.CLAMP
                ),
            },
        ],
    }))

    const focusedOpacity = useSharedValue(0)

    useEffect(() => {
        focusedOpacity.value = withTiming(state.focused ? 1 : 0, {
            duration: 100,
            easing: Easing.inOut(Easing.exp),
        })
    }, [state.focused])

    const focusedStyle = useAnimatedStyle(() => ({
        opacity: focusedOpacity.value,
    }))

    const blurredStyle = useAnimatedStyle(() => ({
        opacity: interpolate(focusedOpacity.value, [0, 1], [1, 0]),
    }))

    //console.log('thumb rerendered:', props.position, state.focused, state.backupPosition)

    return (
        <SliderThumb
            onLayout={(e) => {
                if (props.thumbWidth.current !== e.nativeEvent.layout.width) {
                    props.thumbWidth.current = e.nativeEvent.layout.width
                    props.onLayoutChange()
                }
            }}
            fat={props.fat ?? false}
            style={positionStyle}
            hidden={props.sliderWidth.current === 0}
        >
            <SliderThumbVisual style={focusedStyle} focused={true} />
            <SliderThumbVisual style={blurredStyle} focused={false} />
        </SliderThumb>
    )
}

export default function Slider(props: {
    min: number
    max: number
    step: number
    value: number
    disabled?: boolean
    fat?: boolean
    onChange: (value: number) => void
    setFocusedRef: MutableRefObject<(focused: boolean) => void>
    onTickRef: MutableRefObject<(value: number) => void>
}): JSX.Element {
    const width = useRef(0)
    const thumbWidth = useRef(0)

    const setThumbStateRef: MutableRefObject<
        React.Dispatch<
            React.SetStateAction<{
                focused: boolean
                backupPosition: number
            }>
        >
    > = useRef(() => null)

    useEffect(() => {
        if (props.step <= 0 || props.step > props.max) console.warn('Slider step is out of bounds.')
    }, [props.step, props.max])

    const convertValue = (value: number, realRange: number) => {
        return ((value - props.min) / (props.max - props.min)) * realRange
    }

    //This fires on every tick for the counter. Should trim it down if possible.
    const convertPosition = (value: number) => {
        const realRange = width.current - thumbWidth.current //get range for slider component
        if (props.step <= 0 || realRange <= 0) return 0 //prevent invalid values
        value = Math.max(Math.min(value, realRange), 0) //clamp
        value = (value / realRange) * (props.max - props.min) + props.min //convert to new range
        value = Math.round(value / props.step) * props.step //apply step
        value = Math.round(value * 10000) / 10000 //prevent rounding errors
        return Math.max(Math.min(value, props.max), props.min) //clamp again
    }

    const position = useSharedValue(0)

    const updatePositions = () => {
        const realRange = width.current - thumbWidth.current
        if (realRange > 0) {
            const newPosition = convertValue(props.value, realRange)
            position.value = newPosition
            setThumbStateRef.current((oldState) => ({ ...oldState, ...{ backupPosition: newPosition } }))
        }
        props.onTickRef?.current(props.value)
    }

    useEffect(updatePositions, [props.min, props.max, props.value, width.current, thumbWidth.current])

    //these wrappers allow the gestures to properly access the refs
    const setTickCallback = (value: number) => props.onTickRef.current(convertPosition(value))
    const setFocusedCallback = (focused: boolean) => props.setFocusedRef.current(focused)
    const setThumbFocusedCallback = (focused: boolean) =>
        setThumbStateRef.current((oldState) => ({ ...oldState, ...{ focused: focused } }))
    const onChangeCallback = () => {
        props.onChange(convertPosition(position.value))
        setThumbStateRef.current({ backupPosition: position.value, focused: false })
    }

    const isLongPressed = useSharedValue(false)

    const longPress = Gesture.LongPress()
        .minDuration(200)
        .onStart((event) => {
            position.value = event.x
            runOnJS(setThumbFocusedCallback)(true)
            runOnJS(setFocusedCallback)(true)
            isLongPressed.value = true
        })

    const panGesture = Gesture.Pan()
        .manualActivation(true)
        .onTouchesMove((event, stateManager) => {
            if (isLongPressed.value) {
                stateManager.activate()
            } else {
                stateManager.fail()
            }
        })
        .onUpdate((event) => {
            position.value = event.x
            runOnJS(setTickCallback)(event.x)
        })
        .onTouchesUp(() => {
            runOnJS(onChangeCallback)()
            runOnJS(setFocusedCallback)(false)
            isLongPressed.value = false
        })

    const composedGesture = Gesture.Simultaneous(longPress, panGesture)

    //console.log('Slider rerendered', position)

    return (
        <GestureDetector gesture={composedGesture}>
            <SliderBody>
                <SliderRail
                    fat={props.fat ?? false}
                    onLayout={(e) => {
                        if (width.current !== e.nativeEvent.layout.width) {
                            width.current = e.nativeEvent.layout.width
                            updatePositions()
                        }
                    }}
                />
                <AnimatedThumb
                    setThumbStateRef={setThumbStateRef}
                    position={position}
                    fat={props.fat}
                    onLayoutChange={updatePositions}
                    thumbWidth={thumbWidth}
                    sliderWidth={width}
                />
            </SliderBody>
        </GestureDetector>
    )
}
