﻿import React, {
    MutableRefObject,
    ReactElement,
    useCallback,
    useEffect,
    useMemo,
    useRef,
    useState,
} from 'react'
import { StyleProp, Text, View, ViewStyle } from 'react-native'
import {
    EditorCardContainer,
    EditorCardHeader,
    EditorCardTitle,
    EditorCardHint,
    EditorCardHintPressable,
    EditorCard as StyledEditorCard,
    EditorCardDescription,
    EditorCardTooltipIcon,
    SliderCounterText,
} from './editorcard.style'
import { BodyMedium400, BodyMedium600, BodyXL600, HeadingSmall600 } from '../../styles/fonts'
import Tooltip from '../shared/tooltip'
import { processNumericInput, EXCLUDE_REGEX, REGEX, round } from './editorcardutil'
import Slider from './slider'
import { FlexRow } from './common.style'

export default function EditorCard(props: {
    title?: string
    hint?: string | JSX.Element
    children: ReactElement
    tooltip?: string
    description?: string | JSX.Element
    onHintClick?: () => void
    style?: StyleProp<ViewStyle>
    small?: boolean
}): JSX.Element {
    return (
        <StyledEditorCard style={props.style} small={props.small}>
            {props.title ? (
                <EditorCardHeader data-tip={props.tooltip}>
                    <EditorCardTitle small={props.small}>
                        <HeadingSmall600>{props.title}</HeadingSmall600>
                        {props.tooltip ? (
                            <Tooltip delay={1} tooltip={props.tooltip}>
                                <EditorCardTooltipIcon />
                            </Tooltip>
                        ) : null}
                    </EditorCardTitle>
                    {props.hint ? (
                        props.onHintClick ? (
                            <EditorCardHintPressable onPress={props.onHintClick}>
                                <BodyMedium600>{props.hint}</BodyMedium600>
                            </EditorCardHintPressable>
                        ) : (
                            <EditorCardHint>
                                <BodyMedium600>{props.hint}</BodyMedium600>
                            </EditorCardHint>
                        )
                    ) : null}
                </EditorCardHeader>
            ) : (
                <></>
            )}
            {props.description ? (
                <EditorCardDescription>
                    <BodyMedium600>{props.description}</BodyMedium600>
                </EditorCardDescription>
            ) : null}
            <>{props.children}</>
        </StyledEditorCard>
    )
}

export interface MultiActionHint {
    hint: string
    onHintClick: () => void
}

export function MainSettingSliderCard(props: {
    title: string | JSX.Element
    hint?: string
    tooltip?: string
    onHintClick?: () => void
    value: number
    onChange: (e: number) => void
    min: string | number
    max: string | number
    step: string | number
    prefix?: (value: number) => string
    suffix?: (value: number) => string
    changeDelay?: number
    preventDecimal?: boolean
    forceStep?: boolean
    uncapMin?: boolean
    uncapMax?: boolean
    compact?: boolean
}): JSX.Element {
    const setCounterRef: MutableRefObject<(value: number) => void> = useRef(() => null)

    const maxVal = useMemo(
        () => (typeof props.max === 'number' ? props.max : Number.parseFloat(props.max)),
        [props.max]
    )
    const minVal = useMemo(
        () => (typeof props.min === 'number' ? props.min : Number.parseFloat(props.min)),
        [props.min]
    )
    const step = useMemo(
        () => (typeof props.step === 'number' ? props.step : Number.parseFloat(props.step)),
        [props.step]
    )

    const setFocusedRef: MutableRefObject<(focused: boolean) => void> = useRef(() => null)
    const [value, setValue] = useState(props.value)

    const changeValue = (value: number) => {
        setValue(value)
        props.onChange(value)
    }

    useEffect(() => {
        setValue(props.value)
    }, [props.value])

    return (
        <StyledEditorCard small={props.compact}>
            <HeadingSmall600>{props.title}</HeadingSmall600>
            {props.tooltip ? (
                <EditorCardDescription>
                    <BodyMedium600>{props.tooltip}</BodyMedium600>
                </EditorCardDescription>
            ) : (
                <></>
            )}
            <FlexRow style={{ paddingHorizontal: 2 }}>
                <BodyXL600>
                    {props.prefix ? props.prefix(value) : ''}
                    <SliderCounter
                        setFocusedRef={setFocusedRef}
                        setCounterRef={setCounterRef}
                        suffix={props.suffix}
                    />
                </BodyXL600>
                {props.hint ? (
                    props.onHintClick ? (
                        <EditorCardHintPressable onPress={props.onHintClick}>
                            <BodyMedium400>{props.hint}</BodyMedium400>
                        </EditorCardHintPressable>
                    ) : (
                        <EditorCardHint>
                            <BodyMedium400>{props.hint}</BodyMedium400>
                        </EditorCardHint>
                    )
                ) : null}
            </FlexRow>
            <Slider
                fat={true}
                min={minVal}
                max={maxVal}
                step={step}
                value={value}
                onChange={changeValue}
                setFocusedRef={setFocusedRef}
                onTickRef={setCounterRef}
            />
        </StyledEditorCard>
    )
}

function SliderCounter(props: {
    setFocusedRef: MutableRefObject<(focused: boolean) => void>
    setCounterRef: MutableRefObject<(value: number) => void>
    //rounding?: number
    suffix?: (value: number) => string
}): JSX.Element {
    const [focused, setFocused] = useState(false)
    const [counter, setCounter] = useState(0)

    props.setFocusedRef.current = setFocused
    props.setCounterRef.current = (value: number) => {
        //if (props.rounding) value = round(value, props.rounding)
        if (value !== counter) setCounter(value)
    }

    return (
        <Text>
            <SliderCounterText focused={focused}>{counter}</SliderCounterText>{' '}
            {props.suffix ? props.suffix(counter ?? 0) : ''}
        </Text>
    )
}

export function MinorSettingSliderCard(props: {
    title: string
    hint?: string
    tooltip?: string
    onHintClick?: () => void
    value?: number
    onChange: (e: number) => void
    min: string | number
    max: string | number
    step: string | number
    suffix?: (value: number) => string
    disabled?: boolean
    changeDelay?: number
    logarithmic?: boolean
    preventDecimal?: boolean //unused, no manual input on mobile slider cards
    uncapMax?: boolean //unused, no manual input on mobile slider cards
    roundDigits?: number //unused, no manual input on mobile slider cards
}): JSX.Element {
    const setCounterRef: MutableRefObject<(value: number) => void> = useRef(() => null)

    const maxVal = useMemo(
        () => (typeof props.max === 'number' ? props.max : Number.parseFloat(props.max)),
        [props.max]
    )
    const minVal = useMemo(
        () => (typeof props.min === 'number' ? props.min : Number.parseFloat(props.min)),
        [props.min]
    )
    const step = useMemo(
        () => (typeof props.step === 'number' ? props.step : Number.parseFloat(props.step)),
        [props.step]
    )

    const logreverse = useCallback(
        (value: number) => {
            const negative = value < 0 ? -1 : 1

            // position will be between 0 and 100
            const minp = 1
            const maxp = 201

            // The result should be between min and max
            const min = 1
            const max = maxVal + 1

            const minv = Math.log(min)
            const maxv = Math.log(max)
            // calculate adjustment factor
            const scale = (maxv - minv) / (maxp - minp)
            const result = (Math.log(value * negative + 1) - minv) / scale + minp
            return (result - 1) * negative
        },
        [maxVal]
    )

    const logslider = useCallback(
        (position: number) => {
            const negative = position < 0 ? -1 : 1

            // position will be between 0 and 100
            const minp = 1
            const maxp = 201
            // The result should be between min and max
            const min = 1
            const max = maxVal + 1

            const minv = Math.log(min)
            const maxv = Math.log(max)
            // calculate adjustment factor
            const scale = (maxv - minv) / (maxp - minp)
            const result = Math.exp(minv + scale * (position * negative + 1 - minp))
            return (result - 1) * negative
        },
        [maxVal]
    )

    const transformToSlider = useCallback(
        (n: number) => {
            if (props.logarithmic) {
                return logreverse(n)
            }
            return n
        },
        [logreverse, props.logarithmic]
    )

    const transformFromSlider = useCallback(
        (n: number, roundValue: boolean = true) => {
            if (props.logarithmic) {
                if (roundValue) {
                    return round(logslider(n))
                }
                return logslider(n)
            }
            return n
        },
        [logslider, props.logarithmic]
    )

    const setFocusedRef: MutableRefObject<(focused: boolean) => void> = useRef(() => null)
    const [value, setValue] = useState(props.value || 0)

    const changeValue = (e: number) => {
        setValue(e)
        setCounterRef.current(e)
        props.onChange(e)
    }

    useEffect(() => {
        if (props.value !== undefined) {
            setValue(props.value)
        }
    }, [logreverse, props.logarithmic, props.value])

    return (
        <StyledEditorCard>
            <FlexRow style={{ paddingHorizontal: 1 }}>
                <BodyMedium600>
                    {props.title}:{' '}
                    <SliderCounter
                        setFocusedRef={setFocusedRef}
                        setCounterRef={setCounterRef}
                        suffix={props.suffix}
                    />
                </BodyMedium600>
                {props.hint ? (
                    props.onHintClick ? (
                        <EditorCardHintPressable onPress={props.onHintClick}>
                            <BodyMedium400>{props.hint}</BodyMedium400>
                        </EditorCardHintPressable>
                    ) : (
                        <EditorCardHint>{props.hint}</EditorCardHint>
                    )
                ) : null}
            </FlexRow>
            <Slider
                min={props.logarithmic ? -200 : minVal}
                max={props.logarithmic ? 200 : maxVal}
                step={props.logarithmic ? 1 : step}
                value={transformToSlider(value)}
                onChange={(value) => {
                    changeValue(transformFromSlider(value))
                }}
                disabled={!!props.disabled}
                setFocusedRef={setFocusedRef}
                onTickRef={setCounterRef}
            />
        </StyledEditorCard>
    )
}

//only used for presets - sort out later
export function MultiActionEditorCard(props: {
    title: string
    hint?: string
    children: ReactElement
    tooltip?: string
    onHintClick?: () => void
}): JSX.Element {
    return (
        <EditorCardContainer>
            <EditorCardHeader data-tip={props.tooltip}>
                <EditorCardTitle>
                    <HeadingSmall600>{props.title}</HeadingSmall600>
                    {props.tooltip ? (
                        <Tooltip delay={1} tooltip={props.tooltip}>
                            <EditorCardTooltipIcon />
                        </Tooltip>
                    ) : null}
                </EditorCardTitle>
                {props.onHintClick ? (
                    <EditorCardHintPressable onPress={props.onHintClick}>
                        {props.hint}
                    </EditorCardHintPressable>
                ) : (
                    <EditorCardHint>{props.hint}</EditorCardHint>
                )}
            </EditorCardHeader>
            <View>{props.children}</View>
        </EditorCardContainer>
    )
}
