import { ExecutionContext, HttpException, HttpStatus } from "@nestjs/common";
import { ThrottlerGuard } from "@nestjs/throttler";
import { getAccessTokenDataFromRequest } from "./require-access-token.guard";

export class UserBasedThrottlerGuard extends ThrottlerGuard {
  async handleRequest(
    context: ExecutionContext,
    limit: number,
    ttl: number,
  ): Promise<boolean> {
    // Skip user-based throttling by request
    if (process.env.NODE_ENV !== "production") return true;

    return super.handleRequest(context, limit, ttl);
  }

  getTracker(req: Record<string, any>): string {
    if (req.headers["authorization"]) {
      const credentialsData = getAccessTokenDataFromRequest(req);

      if (credentialsData.type != "bearer") return credentialsData.token;

      const accessTokenSplit = credentialsData.token.split(".");
      if (accessTokenSplit.length != 3)
        throw new HttpException("Invalid accessToken.", HttpStatus.BAD_REQUEST);

      try {
        return JSON.parse(
          Buffer.from(accessTokenSplit[1], "base64").toString("utf-8"),
        ).id;
      } catch (ex) {
        console.error(ex);
        throw new HttpException(
          "Invalid accessToken.",
          HttpStatus.INTERNAL_SERVER_ERROR,
        );
      }
    }

    return req.headers["cf-connecting-ip"] || req.ip;
  }
}
