import { NestFactory } from "@nestjs/core";
import {
  FastifyAdapter,
  NestFastifyApplication,
} from "@nestjs/platform-fastify";
import { AppModule } from "./app.module";
import * as dotenv from "dotenv";
import { HttpException, HttpStatus, ValidationPipe } from "@nestjs/common";
import { DocumentBuilder, SwaggerModule } from "@nestjs/swagger";
import { HttpExceptionFilter } from "./shared/filters/http-exception.filter";
import { WINSTON_MODULE_NEST_PROVIDER } from "nest-winston";
import { WsAdapter } from "@nestjs/platform-ws";
// import fastifyCookie from "fastify-cookie";

const CORS_DOMAIN_WHITELIST = [
  /^http[s]{0,1}:\/\/novelai.net/,
  /^http[s]{0,1}:\/\/.*\.novelai.net/,
  /^http[s]{0,1}:\/\/localhost/,
  /^http[s]{0,1}:\/\/novelai-web\.pages\.dev/,
  /^http[s]{0,1}:\/\/[a-zA-Z0-9\-]*\.novelai-web\.pages\.dev/,
];

async function bootstrap() {
  dotenv.config();

  const app = await NestFactory.create<NestFastifyApplication>(
    AppModule,
    new FastifyAdapter({
      trustProxy: true,
      bodyLimit: 50 * 1048576,
      requestTimeout: 120 * 1000,
    }),
  );

  // Register cookie support
  // app.register(fastifyCookie);

  const logger = app.get(WINSTON_MODULE_NEST_PROVIDER);

  app.useLogger(logger);
  app.useGlobalFilters(new HttpExceptionFilter(logger));

  app.useGlobalPipes(
    new ValidationPipe({
      whitelist: true,
      exceptionFactory: (errors) => {
        throw new HttpException(
          {
            statusCode: 400,
            message: errors
              .map((error) => {
                if (error.children.length > 0)
                  return Object.values(error.children)
                    .map((error) => Object.values(error.constraints).join(", "))
                    .join(", ");

                return Object.values(error.constraints).join(", ");
              })
              .join("; "),
          },
          HttpStatus.BAD_REQUEST,
        );
      },
    }),
  );

  app.useWebSocketAdapter(new WsAdapter(app));

  const config = new DocumentBuilder()
    .setTitle("NovelAI Backend")
    .setDescription("")
    .setVersion("1.0")
    .addBearerAuth()
    //.addCookieAuth("at")
    .build();

  const document = SwaggerModule.createDocument(app, config);

  delete document.paths["/Q2oloyzi9LZvqXAQAcT4pzDuORktuZYA"]; // Hide Prometheus metrics API endpoint

  SwaggerModule.setup("docs", app, document);

  app.enableCors({
    origin: (origin, cb) => {
      if (origin && !CORS_DOMAIN_WHITELIST.some((entry) => entry.test(origin)))
        return cb(new Error(`Origin \"${origin}\" not allowed by CORS rules.`));

      cb(null, origin || "*");
    },
    allowedHeaders:
      "X-Requested-With, X-HTTP-Method-Override, Content-Type, Accept, Observe, Authorization",
    methods: "GET,PUT,POST,DELETE,PATCH,OPTIONS",
    credentials: true,
  });

  await app.listen(3000, process.env.LISTEN_ADDRESS || "127.0.0.1");
}
bootstrap();
