import {
  Body,
  Controller,
  Delete,
  Get,
  Header,
  HttpCode,
  HttpException,
  HttpStatus,
  Param,
  Post,
  Req,
} from "@nestjs/common";
import {
  ApiBadRequestResponse,
  ApiBearerAuth,
  ApiConflictResponse,
  ApiCreatedResponse,
  ApiInternalServerErrorResponse,
  ApiNotFoundResponse,
  ApiOkResponse,
  ApiResponse,
  ApiTags,
  ApiUnauthorizedResponse,
} from "@nestjs/swagger";
import { Throttle } from "@nestjs/throttler";
import { RequireAccessToken } from "../shared/guards/require-access-token.guard";
import { ApiError } from "../shared/dto/api-error.dto";
import { AIModuleService } from "./ai-module.service";
import {
  AiModuleDto,
  AiModuleTrainRequest,
  BuyTrainingStepsRequest,
} from "./dto/ai-modules.dto";
import { User } from "../user/user.entity";
import { CounterMetric, PromService } from "@digikare/nestjs-prom";
import { GenerationModelAccessRightsData } from "../shared/generation-models";
import { UserService } from "src/user/user.service";

const MAX_SIMULTANEOUS_TRAINING_MODULES = 1;

@ApiTags("/ai/module/")
@Controller("ai/module")
@ApiInternalServerErrorResponse({
  description: "An unknown error occured.",
  type: ApiError,
})
@Throttle(6, 2)
export class AIModuleController {
  private readonly _step_purchase_counter: CounterMetric;

  constructor(
    private readonly aiModuleService: AIModuleService, // private readonly taskPriorityService: TaskPriorityService,
    private readonly userService: UserService,
    private readonly promService: PromService,
  ) {
    this._step_purchase_counter = this.promService.getCounter({
      name: "ai_module_step_purchase_count",
      help: "AI Module Step Purchase Count",
    });
  }

  @Post("train")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiResponse({
    status: HttpStatus.PAYMENT_REQUIRED,
    description: "An active subscription required to access this endpoint.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict error occured.",
    type: ApiError,
  })
  @ApiCreatedResponse({
    description: "The training request has been successfully sent.",
    type: AiModuleDto,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  async trainModule(
    @Req() req,
    @Body() aiTrainingRequest: AiModuleTrainRequest,
  ): Promise<AiModuleDto> {
    const user: User = req.user;

    if (!aiTrainingRequest)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    if (
      !GenerationModelAccessRightsData[aiTrainingRequest.model].canTrainModules
    )
      throw new HttpException(
        "Specified model does not support module training.",
        HttpStatus.BAD_REQUEST,
      );

    /*if (!user.hasSubscription())
      throw new HttpException(
        "Incorrect subscription.",
        HttpStatus.PAYMENT_REQUIRED,
      );*/

    if (aiTrainingRequest.steps < 50)
      throw new HttpException(
        "Training steps amount is too low.",
        HttpStatus.CONFLICT,
      );

    if (
      user.availableModuleTrainingSteps + user.purchasedModuleTrainingSteps <
      aiTrainingRequest.steps
    )
      throw new HttpException(
        "You have an insufficent amount of available training steps.",
        HttpStatus.CONFLICT,
      );

    if (process.env.NODE_ENV === "production") {
      const modules = (
        await this.aiModuleService.getAllUserModules(user)
      ).filter((item) => item.status == "pending" || item.status == "training");

      if (modules.length >= MAX_SIMULTANEOUS_TRAINING_MODULES)
        throw new HttpException(
          "You have reached the limit of concurrently training modules.",
          HttpStatus.CONFLICT,
        );
    }

    return await this.aiModuleService.trainModule(user, aiTrainingRequest);
  }

  @Get("all")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  allModules(@Req() req): Promise<AiModuleDto[]> {
    const user: User = req.user;

    return this.aiModuleService.getAllUserModules(user);
  }

  @Get(":id")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiNotFoundResponse({
    description: "Module not found",
    type: ApiError,
  })
  async getModule(@Req() req, @Param("id") id: string): Promise<AiModuleDto> {
    const user: User = req.user;

    const module = await this.aiModuleService.getUserModule(user, id);
    if (!module)
      throw new HttpException("Module not found.", HttpStatus.NOT_FOUND);

    return this.aiModuleService.moduleToModuleDto(module);
  }

  @Delete(":id")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiNotFoundResponse({
    description: "Module not found",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict error occured.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Module deleted successfully.",
    type: AiModuleDto,
  })
  async deleteModule(
    @Req() req,
    @Param("id") id: string,
  ): Promise<AiModuleDto> {
    const user: User = req.user;

    const module = await this.aiModuleService.getUserModule(user, id);
    if (!module)
      throw new HttpException("Module not found.", HttpStatus.NOT_FOUND);

    if (module.status == "training")
      throw new HttpException(
        "Module is currently training, deletion is not possible.",
        HttpStatus.CONFLICT,
      );

    await this.aiModuleService.deleteModule(module);

    return this.aiModuleService.moduleToModuleDto(module);
  }

  @Post("buy-training-steps")
  @ApiBearerAuth()
  @RequireAccessToken()
  @Throttle(2, 10)
  @HttpCode(HttpStatus.OK)
  @ApiOkResponse({
    description: "Steps have been purchased properly.",
  })
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict occured while buying training steps.",
    type: ApiError,
  })
  async buyTrainingSteps(@Req() req, @Body() body: BuyTrainingStepsRequest) {
    if (!body) throw new HttpException("Invalid body.", HttpStatus.BAD_REQUEST);

    const user: User = req.user;
    const amount = body.amount;

    if (amount <= 0)
      throw new HttpException("Invalid steps amount.", HttpStatus.BAD_REQUEST);

    try {
      await this.userService.loadSubscriptionData(user);
      const result = await this.aiModuleService.tryBuyTrainingSteps(
        user,
        amount,
      );
      this._step_purchase_counter.inc(amount);
      return result;
    } catch (e) {
      return e;
    }
  }
}
