package main

import (
	"log"
	"nai-gobot/gobot"
	"os"
	"os/signal"

	"github.com/bwmarrin/discordgo"
)

var (
	// Global variables for discord session
	session    *discordgo.Session
	guild      string
	configFile string
)

var conf = new(gobot.Config)
var guildMembers = make([]*discordgo.User, 0)

func discordHandler(s *discordgo.Session, i *discordgo.InteractionCreate) {
	if i.Type == discordgo.InteractionMessageComponent {
		genReq, api, err := gobot.StrToImageGenerationRequest(i.Message.Content, i.Message.Attachments[0].URL)
		if err != nil {
			gobot.ErrorHandler(err, true, s, i)
			return
		}
		err = gobot.MessageComponentHandler(conf, i, genReq)
		if err != nil {
			log.Printf("Error handling message component: %v", err)
			return
		}
		gobot.RespondCommand(s, i)

		err = gobot.SendImage(conf, s, i, genReq, api, false)
		if err != nil {
			gobot.ErrorHandler(err, true, s, i)
			return
		}
		return
	}

	// Command handling part

	data := i.ApplicationCommandData()
	err := gobot.CommandHandler(conf, s, i, &data, &guildMembers)
	if err != nil {
		gobot.ErrorHandler(err, true, s, i)
		return
	}
}

func init() {
	var err error
	guild = os.Getenv("DISCORD_GUILD")
	discordToken := os.Getenv("DISCORD_TOKEN")
	if discordToken == "" {
		log.Fatal("DISCORD_TOKEN not set")
	}
	if os.Getenv("GOBOT_MODE") == "devel" {
		configFile = "config/devel.yaml"
	} else if os.Getenv("GOBOT_MODE") == "prod" {
		configFile = "config/prod.yaml"
	} else {
		log.Fatal("GOBOT_MODE not set to `devel` or `prod`")
	}

	session, err = discordgo.New("Bot " + discordToken)
	if err != nil {
		log.Fatal(err)
	}
}

func main() {
	// This is temporary until the backend returns the module list

	session.AddHandler(discordHandler)
	err := session.Open()
	if err != nil {
		log.Fatal("error opening connection ", err)
	}
	log.Printf("Adding commands for guild %s", guild)
	conf = gobot.MakeConfig(configFile)
	commands := conf.Commands
	registeredCommands := make([]*discordgo.ApplicationCommand, 0)
	for _, command := range commands {
		if command.AllowDMs && conf.AllowDM {
			globalCmd, err := session.ApplicationCommandCreate(session.State.User.ID, "", command.Command)
			if err != nil {
				log.Printf("Error creating global command: %v", err)
			} else {
				registeredCommands = append(registeredCommands, globalCmd)
			}
		}
		cmd, err := session.ApplicationCommandCreate(session.State.User.ID, guild, command.Command)
		if err != nil {
			log.Panicf("Error creating command %s: %s", command.Name, err.Error())
		}
		registeredCommands = append(registeredCommands, cmd)
		log.Printf("Registered command %s", cmd.Name)
	}
	guildMembersList, err := session.GuildMembers(guild, "", 40000)
	for _, member := range guildMembersList {
		guildMembers = append(guildMembers, member.User)
	}

	log.Println("Bot is now running.  Press CTRL-C to exit.")
	stop := make(chan os.Signal, 1)
	signal.Notify(stop, os.Interrupt)
	<-stop

	defer func(session *discordgo.Session) {
		err := session.Close()
		if err != nil {
			log.Fatal(err)
		}
	}(session)
}
