import { AppProps } from 'next/app'
import '../styles/globals.css'
import { Router, useRouter } from 'next/router'
import styled from 'styled-components'
import { Block, Container, Fill, MaskImage, MOBILE_SMALL, Row, RowFull, Space, TABLET_LARGE, MOBILE_LARGE, DESKTOP_SMALL } from '../components/common'
import LogoMark from '../assets/icons/logomark.svg';
import Link from 'next/link'
import LightCircleBackground from '../assets/backgrounds/circles-light.svg';
import { motion, MotionValue, useAnimation, useMotionValue, useTime, useTransform } from 'framer-motion'
import bezier from 'bezier-easing'
import { useEffect, useRef, useState } from 'react'
import { useWindowSize } from '../hooks/useWindowSize'

const Header = styled(Block)`
  position: absolute;
  height: 190px;
  z-index: 3;
`

const Footer = styled(Block)`
  background-color: #1A1A1A;
  min-height: 252px;
  background-image: url(${LightCircleBackground.src});
  background-repeat: no-repeat;
  background-position: right -1470px top -1330px;
  @media (max-width: ${TABLET_LARGE}px) {
    background-position: right -1693px top -1256px;
  }
  @media (max-width: ${MOBILE_SMALL}px) {
    background-size: 1855px;
    min-height: 340px;
    background-position: right -1053px  top -542px;
  }
`


let easing = bezier(0, -0.01, 1, .02);


const Circle = (props: { duration: number, timeOffset: number, color: string, time: MotionValue, maxSize: number, offsetX: number, offsetY: number }) => {
  let r = useTransform(props.time, latest => {
    let seconds = latest / 1000
    let progressPercent = (seconds % props.duration) / props.duration
    let adjustedPercent = (progressPercent + props.timeOffset) % 1

    return Math.max(0, easing(adjustedPercent) * props.maxSize)
  })

  return <motion.circle
    cx={props.offsetX}
    cy={props.offsetY}
    r={r}
    transition={{
      duration: props.duration,
      repeat: Infinity
    }}
    fill={'transparent'}
    stroke="#5200FF"
    stroke-opacity="0.1"
    stroke-width="2" />

}

const CircleBackground = (props: { offsetX: number, offsetY: number, children: JSX.Element | JSX.Element[], circles: number, duration: number }): JSX.Element => {
  const maxSize = 3200

  const container = {
    hidden: { opacity: 1 },
    show: {
      opacity: 1,
      transition: {
        staggerChildren: 6,
        delay: -5
      }
    }
  }

  const divRef = useRef<HTMLDivElement>(null)
  const router = useRouter()

  let time = useTime()

  const [height, setHeight] = useState(0)

  // watch and set height to window height
  useEffect(() => {
    setHeight(divRef.current?.clientHeight || 0)
    let resize =  () => {
      console.log('resize')
      setHeight(divRef.current?.clientHeight || 0)
    }
    router.events.on('routeChangeComplete', resize)
    window.addEventListener('resize', resize)
    
    return () => {
      window.removeEventListener('resize', resize)
      router.events.off('routeChangeComplete', resize)
    }

  }, [router.events])

  return <div>
    <div style={{ position: 'absolute', height: divRef.current?.clientHeight ?? '100%', width: '100%', overflow: 'hidden', backgroundColor: '#E4E4F5' }}>

      <motion.svg
        width={maxSize}
        height={`${height}px`}
        variants={container}
        initial="hidden"
        animate="show"


      >
        {Array.from({ length: props.circles }).map((_, i) => {
          let offset = (i / props.circles) + 0.05
          return <Circle key={i} duration={props.duration} timeOffset={offset} color={'#E4E4F5'} time={time} maxSize={maxSize} offsetX={props.offsetX} offsetY={props.offsetY} />
        })}
      </motion.svg>
    </div>
    <div ref={divRef} style={{ position: 'relative' }}>
      {props.children}
    </div>
  </div>
}

const App = ({ Component, pageProps }: AppProps): JSX.Element => {
  let window = useWindowSize()
  const xOffset = () => {
    switch (true) {
      case window.width < MOBILE_SMALL:
        return 70
      case window.width < MOBILE_LARGE:
        return 105
      case window.width < TABLET_LARGE:
        return 155
      case window.width < DESKTOP_SMALL:
        return 205
      default:
        return 305
    }
  }
  const circles = () => {
    switch (true) {
      case window.width < MOBILE_SMALL:
        return 10
      default:
        return 7
    }
  }
  return <CircleBackground offsetX={xOffset()} offsetY={60} circles={circles()} duration={180}>
    <Container>
      <div style={{ position: 'relative' }}>
        <Header>
          <Space height={50} />
          <RowFull>
            <Row>
              <Link href='/'>
                <a>
                  <MaskImage style={{ height: 19, width: 150 }} src={LogoMark.src} color='#5200FF' />
                </a>
              </Link>
            </Row>
            <Row>
              <Link href="/about">
                <a>About</a>
              </Link>
              <Space width={40} />
              <a href='mailto:contact@anlatan.ai'>Contact</a>
              <Space width={40} />
              {
                /*
              <Link href="/jobs">
                <a>Jobs</a>
              </Link>
                */
              }
            </Row>
          </RowFull>
        </Header>

      </div>
      <Component {...pageProps} />
      <Fill />
      <Footer>
        <Space height={50} />
        <MaskImage style={{ height: 19 }} src={LogoMark.src} color='#E4E4F5' />
        <Space height={30} />
        <Row style={{ color: '#E4E4F5' }}>
          <Link href="/about">
            <a>About</a>
          </Link>
          <Space width={40} />
          <a href='mailto:contact@anlatan.ai'>Contact</a>
        </Row>
      </Footer>
    </Container>
  </CircleBackground>

}

export default App
