/* eslint-disable @next/next/no-img-element */

import { MutableRefObject, useRef, useCallback, useEffect } from 'react'
import styled, { css } from 'styled-components'
import Typewriter, { Options } from 'typewriter-effect'
import Link from 'next/link'
import Head from 'next/head'
import { Router } from 'next/router'
import { LazyLoadImage } from 'react-lazy-load-image-component'
import Book from '../assets/images/book-open.svg'
import zork from '../assets/images/prefixes/textadventure.webp'
import vanilla from '../assets/images/prefixes/vanilla.webp'
import postapoc from '../assets/images/prefixes/postapoc.webp'
import merchant from '../assets/images/prefixes/merchant.webp'
import sherlock from '../assets/images/prefixes/sherlock.webp'
import poe from '../assets/images/prefixes/poe.webp'
import lovecraft from '../assets/images/prefixes/lovecraft.webp'
import darkfantasy from '../assets/images/prefixes/darkfantasy.webp'
import dragons from '../assets/images/prefixes/dragons.webp'
import mars from '../assets/images/prefixes/mars.webp'
import GooseGreen from '../assets/images/landing/testimonials/goose_green.svg'
import GoosePurple from '../assets/images/landing/testimonials/goose_purple.svg'
import GoosePink from '../assets/images/landing/testimonials/goose_pink.svg'
import GooseBlue from '../assets/images/landing/testimonials/goose_blue.svg'
import GooseOrange from '../assets/images/landing/testimonials/goose_orange.svg'
import GooseYellow from '../assets/images/landing/testimonials/goose_yellow.svg'
import ImageGenBackground from '../assets/images/landing/imagegen.webp'
import {
    ConversationInput,
    ControlButton,
    ConversationInputContainer,
    ConversationStoryControls,
} from '../styles/components/conversationcontrols'

import ContentBackground from '../assets/images/landing/landing_content_bg.svg'
import PenTip from '../../public/icons/pen-tip-light.svg'
import ArrowImage from '../assets/images/arrow.svg'
import {
    CrossIcon,
    ArrowRightIcon,
    Icon,
    ArrowDownIcon,
    BookIcon,
    ImportIcon,
    ExportIcon,
    SearchIcon,
    PlusIcon,
    ArrowUpIcon,
    EaselIcon,
    LeftSparkleIcon,
    RightSparkleIcon,
} from '../styles/ui/icons'
import { darken, transparentize } from '../util/colour'
import { SubtleButton } from '../styles/ui/button'
import Say from '../assets/images/speech-bubble.svg'
import Do from '../assets/images/walking.svg'
import Swords from '../assets/images/swords.svg'
import SmallArrow from '../assets/images/small_arrow.svg'
import Lock from '../assets/images/lock.svg'
import Text from '../assets/images/text.svg'
import Theme from '../assets/images/theme.svg'
import Pencil from '../assets/images/pencil.svg'
import Mind from '../assets/images/mind.svg'
import Sliders from '../assets/images/sliders.svg'
import ArrowCircle from '../assets/images/arrow_circle.svg'
import World from '../assets/images/world.svg'

import { DefaultInputModes } from '../data/story/defaultinputmodes'
import { Dark } from '../styles/themes/dark'
import useVersionCheck from '../hooks/useVersionCheck'
import useRestoreScrollPosition from '../hooks/useRestoreScrollPosition'
import {
    OpusTierContent,
    PaperTierContent,
    ScrollTierContent,
    TabletTierContent,
    TierFinePrint,
} from '../components/subscription'
import { mobileSize, smallerMobileSize, smallMobileSize } from '../globals/constants'
import Tooltip from '../components/tooltip'
import { randomizeArray } from '../util/util'
import { FlexColSpacer, FlexRow } from '../styles/ui/layout'
import { backgroundImages, backgroundIndex } from '../util/build'

const typewriterStrings = [
    'a castle across the bay',
    'a city powered by steam',
    'a city reclaimed by nature',
    'childhood memories',
    'a fractured past',
    'a journey through the mountains',
    'a kingdom in the desert',
    'a land of magic',
    'a misty wood',
    'a moonlit curse',
    'a nation at war',
    'the age of man',
    'a new case',
    'a quiet life in the country',
    'a riverside stroll',
    'a somber evening',
    'a twisted reality',
    'a vibrant coastline',
    'a walk through the gardens',
    'a warm inn',
    'research best forgotten',
    'an armed insurrection',
    'an enchanted library',
    'an ephemeral dreamscape',
    'an expedition into the unknown',
    'ancient prophecies',
    'bustling markets',
    'a calm night at sea',
    'exploring crystalline caverns',
    'losing your way',
    'egg?',
    'a picnic by the lake',
    'far-off mountains',
    'foreboding hallways',
    'heroes and villains',
    'hidden temples',
    'inspiring tales',
    'forgotten places',
    'love and lust',
    'otherworldly beings',
    'rough waters',
    'rumblings of war',
    'rural society',
    'hunting sky whales',
    'strange occurrences',
    'the beginning of a new day',
    'the city of Alchemy',
    'the dead of night',
    'the end of an era',
    'the gardens of Heaven',
    'the gardens of Hell',
    'the Home of the Gods',
    'the Imperial Court',
    'the palace menagerie',
    'the path to the underworld',
    'a mountaintop institute',
    'the struggles of an artist',
    "the sun's blessings",
    'a secluded workshop',
    'things beyond comprehension',
    'travelling through frozen lands',
    'unexpected romance',
    'unseen depths',
    'Victorian times',
]

const unshuffledBackgrounds = backgroundImages.map((s: any, i: number) => {
    return { url: s.default.src, text: typewriterStrings[i % typewriterStrings.length] }
})

const saved = unshuffledBackgrounds[backgroundIndex]
const toShuffle = [
    ...unshuffledBackgrounds.slice(0, backgroundIndex),
    ...unshuffledBackgrounds.slice(backgroundIndex + 1),
]

const backgrounds: { url: string; text: string }[] = [...randomizeArray(toShuffle), saved]

const sortedStrings = backgrounds.map((o) => o.text)

const images: HTMLImageElement[] = []
function preload(url: string) {
    if (typeof window !== 'undefined') {
        images.shift()
        const image = new window.Image()
        image.src = url
        images.push(image)
    }
}

preload(backgrounds[0].url)
preload(backgrounds[1].url)
preload(backgrounds[2].url)

export class AdventureNode {
    name: string
    icon: string
    regex?: RegExp

    constructor(name: string, icon: string, regex?: RegExp) {
        this.name = name
        this.icon = icon
        this.regex = regex
    }
}
export interface SchemaNodes {
    adventure: AdventureNode[]
}
export enum StoryMode {
    normal = 0,
    adventure = 1,
}

export default function Landing(): JSX.Element {
    const introRef: MutableRefObject<null | HTMLDivElement> = useRef(null)
    const featuresRef: MutableRefObject<null | HTMLDivElement> = useRef(null)
    const pricingRef: MutableRefObject<null | HTMLDivElement> = useRef(null)
    const scrollRef: MutableRefObject<null | HTMLDivElement> = useRef(null)

    const backgroundRef: MutableRefObject<HTMLDivElement | null> = useRef(null)
    const backgroundBackgroundRef: MutableRefObject<HTMLDivElement | null> = useRef(null)

    const backgroundRefs = [backgroundRef, backgroundBackgroundRef]
    const currentBackgroundRef = useRef(0)

    const characterCount = useRef(sortedStrings[sortedStrings.length - 1].length * 2)
    const currentBg = useRef(backgrounds.length - 1)

    const storeScroll = useRestoreScrollPosition(
        scrollRef,
        'landingScrollPos',
        !global.window?.document?.referrer || !global.window.document.referrer.includes('//novelai.net')
    )
    useEffect(() => {
        const store = () => storeScroll()
        Router.events.on('beforeHistoryChange', store)
        return () => Router.events.off('beforeHistoryChange', store)
    }, [storeScroll])

    // reload page when update is available
    const update = useVersionCheck()
    const updateRef = useRef(0)
    useEffect(() => {
        if (update.available) {
            clearTimeout(updateRef.current)
            updateRef.current = setTimeout(() => {
                // preload html to ensure cache busting
                fetch('/', {
                    cache: 'no-cache',
                })
                    .then(() => {
                        window.location.reload()
                    })
                    .catch(() => {
                        // ignore
                    })
            }, 30000) as unknown as number
        }
        return () => clearTimeout(updateRef.current)
    }, [update])

    const scrollIntro = useCallback(
        () =>
            introRef.current?.scrollIntoView({
                block: 'start',
                behavior: 'smooth',
            }),
        []
    )
    const scrollPricing = useCallback(
        () =>
            pricingRef.current?.scrollIntoView({
                block: 'start',
                behavior: 'smooth',
            }),
        []
    )
    const scrollFeatures = useCallback(
        () =>
            featuresRef.current?.scrollIntoView({
                block: 'start',
                behavior: 'smooth',
            }),
        []
    )

    return (
        <>
            <Head>
                <title>NovelAI - The GPT-powered AI Storyteller</title>
            </Head>

            <TopBackground src={backgrounds[backgrounds.length - 1].url} suppressHydrationWarning={true}>
                <div ref={backgroundBackgroundRef} />
                <div ref={backgroundRef}></div>
                <div></div>
            </TopBackground>

            <LandingPage ref={scrollRef}>
                <LandingTop>
                    <Container>
                        <Header>
                            <Logo />
                            <Title>NovelAI</Title>
                            <Placeholder />
                            <div>
                                <LoginButton onClick={scrollIntro}>About</LoginButton>
                                <LoginButton onClick={scrollPricing}>Pricing</LoginButton>
                                <LoginLink target="_blank" href="https://blog.novelai.net/">
                                    Blog
                                </LoginLink>
                                <LoginLink target="_blank" href="https://discord.com/invite/novelai">
                                    Discord
                                </LoginLink>
                            </div>
                            <div>
                                <Link href="/login" passHref>
                                    <LoginLink>Login</LoginLink>
                                </Link>
                            </div>
                        </Header>
                    </Container>
                    <LandingRow>
                        <TagLine>Write about</TagLine>
                    </LandingRow>
                    <LandingRow>
                        <TagLineOther>
                            <Typewriter
                                onInit={() => {
                                    {
                                        if (!backgroundRef.current) return
                                        backgroundRef.current.style.backgroundImage = `url(${
                                            backgrounds[currentBg.current].url
                                        })`
                                    }
                                }}
                                options={
                                    {
                                        strings: sortedStrings,
                                        autoStart: true,
                                        loop: true,
                                        delay: 100,
                                        pauseFor: 8000,
                                        initialText: sortedStrings[sortedStrings.length - 1],
                                        initialTextDelay: 3000,
                                        onCreateTextNode: (_character, textNode) => {
                                            return textNode
                                        },
                                        onRemoveNode: ({ node }) => {
                                            characterCount.current--
                                            if (characterCount.current === 0) {
                                                const value = (currentBg.current + 1) % sortedStrings.length
                                                const preloadValue = (value + 1) % sortedStrings.length

                                                currentBg.current = value

                                                if (backgroundRefs.every((ref) => ref.current)) {
                                                    const current =
                                                        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                                                        backgroundRefs[currentBackgroundRef.current].current!
                                                    const next =
                                                        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                                                        backgroundRefs[
                                                            (currentBackgroundRef.current + 1) %
                                                                backgroundRefs.length
                                                        ].current!

                                                    current.style.opacity = '0'
                                                    next.style.backgroundImage = `url(${backgrounds[value].url})`
                                                    next.style.opacity = '1'

                                                    currentBackgroundRef.current =
                                                        (currentBackgroundRef.current + 1) %
                                                        backgroundRefs.length

                                                    preload(backgrounds[preloadValue].url)
                                                }

                                                characterCount.current =
                                                    sortedStrings[currentBg.current].length * 2
                                            }

                                            return node
                                        },
                                    } as Partial<Options>
                                }
                            />
                        </TagLineOther>
                    </LandingRow>
                    <LandingRow>
                        <TagSubline>
                            Driven by AI, painlessly construct unique stories, thrilling tales, seductive
                            romances, or just fool around. Anything goes!{' '}
                        </TagSubline>
                    </LandingRow>
                    <MainButtons>
                        <div style={{ display: 'flex', flex: '0 0 auto' }}>
                            <LearnBox onClick={scrollIntro}>
                                <span>LEARN MORE</span> <SmallArrowIcon />
                            </LearnBox>
                            <Link href="/trial" passHref>
                                <Start>START FOR FREE_</Start>
                            </Link>
                        </div>
                    </MainButtons>
                    <PreviewImage>
                        <video
                            autoPlay={true}
                            muted={true}
                            playsInline={true}
                            loop={true}
                            style={{ maxWidth: '100%' }}
                        >
                            <source src={'./videos/start.mp4'} type="video/mp4" />
                        </video>
                    </PreviewImage>
                    <MobilePreviewImage>
                        <video
                            autoPlay={true}
                            muted={true}
                            playsInline={true}
                            loop={true}
                            style={{ maxWidth: '100%' }}
                        >
                            <source src={'./videos/mobile_start.mp4'} type="video/mp4" />
                        </video>
                    </MobilePreviewImage>
                </LandingTop>
                <LandingContent>
                    <LandingCardFirst>
                        <Container>
                            <IntroCard ref={introRef}>
                                <IntroTextSection>
                                    <IntroLine>What even is NovelAI?</IntroLine>
                                    <IntroSubLine>
                                        NovelAI is a monthly subscription service for AI-assisted authorship,
                                        storytelling, virtual companionship, or simply a GPT powered sandbox
                                        for your imagination.
                                    </IntroSubLine>
                                    <IntroText>
                                        Our Artificial Intelligence algorithms create human-like writing based
                                        on your own, enabling anyone, regardless of ability, to produce
                                        quality literature. We offer unprecedented levels of freedom with our
                                        Natural Language Processing playground by using our own AI models,
                                        trained on real literature. The AI seamlessly adapts to your input,
                                        maintaining your perspective and style.
                                    </IntroText>
                                    <IntroButtons>
                                        <SkipButton onClick={scrollPricing}>
                                            Take me to the Pricing
                                        </SkipButton>
                                        <SubtleButton onClick={scrollFeatures}>Our Features</SubtleButton>
                                    </IntroButtons>
                                    <div>
                                        <Arrow onClick={scrollFeatures} />
                                    </div>
                                </IntroTextSection>
                                <PrefixImages>
                                    <LazyLoadImage
                                        threshold={420}
                                        delayTime={50}
                                        effect="blur"
                                        // eslint-disable-next-line max-len
                                        placeholderSrc="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAFcUlEQVRYR3WXiXLbRhBEsSBA+Scky4cskT6SSsqVVK7//6ZIINKvZ2YBKmWx1kvKJPtNz7GrdvvT3Trop/FowzDq+ajnrEPuYxv1mge/z2crr/QBPrGyx8+qb1v1z2XQWi/a9ViXYdGKV9rzOe9qPwKwuIiQC5DtEXg8EiIBVu2I87gYAvFcEg6IAFn1nPddAYx2oITlgCOv/RogpPFLH8JDxO2AvhbxHwKEI4YoAAz01zni2BE3wG7fI/ToDRAQZX8HyBQsdkEOZDpCHBfkwN23rAHE98JjABzGwxXMVgmJbfuxXtslIr8CkHgAYD75r7qIVLW7r3crxdcMgGCKSniq5+0wjAlkgEwNkbsQif6ys98gEsp9A6h6yATgwFsBUL1d/KCIEWc/TIKI1yx3Q4qHdOU/7Zfgmi5cLgAQ/bo5sIs+KkAA919uBaDoK1oJT5OEEdcyREKVC1sXUHVYn9EXABCIA+EUCILXroFsUeRRvv98qxQkAGIWn4dpToiECRfSAblQ0bcsPudfInZgQTwhANBz0gBIzIfYYW/vzglQls/zME8A5A6AVk8DEGm/x88+egk770SeEI68xJ2OciDMa+93AJOFj8Ms8b4ngNOBuAFAyB8BED2RryW8bCkwgB2IyBk/+ZGogQ8n1YAmkHMNwFEALEDYDTAPh2lfiFmCMXlCXBF3+3HCDqR4FaNZEyDLxwC01QEhbD/ehHCuSkUB4EJ1QPPkoQUlTtQpbHGnYweh9xG93mYHzE4KPgJAB+CARQVwA0SByJUqRNVJpCCOIRegopeSHSgAiydE2Y9w5D8ALip8AzycNIhIQdlv8TfD8SZgpmMU45UD+jDirMh/CAKxvKjZunh1Q0S+lDiTM0dIezgzCeWA7ZfozZtwQOuo1xSjWzI7Ye+AG7ny76gl/pL2F4RnAMXntzrySEFAtE+fN4DJ9hP9Pg3RknTBeIjzgbnh6DuAsguAxIFYCqJ3QNgeDmB9OaDvefzy1g6MVYA7B+yEW/IVgGsgo+8OEH05kGkgco/mzLs+ZQcy/9yA2uPX+yhC10BGvodAfN+Gqhc/nP9ovVVRr0QvgGWfBrdfWl+Fh3gtvufpWwCMHYAayDqgBljkvw+iOoTymzP3BRAp2BzYR9+FpecjmC54+vldT8E0XztgcYowAVyA2MYnywGq34KVghDv50GMirBdqV65dtVuB355n4NIrcb0U/RTzoE9wPhqBvhb3XqvxLMO4jiOvnfRSRgADRI91x6XkKGdfv2gURyD6OC+jxUzAAcE5g7YhpC9w9uMdH15yQJMF4DqtyNubACE+NWC4fT9Y6QAm93zEpYD3gVAcTKmR19IuLrnBMnpF+2XAM+5+1zg7M8be0au0+wKwO18+u0h7gO6AwBgCKIXAMPJCwcE4A/4FlQHENVfuZf4KwCPCv+xodlB9HKxGUK7r+AA/A6AHECkACweMI7eAHEXjAtIdABtFwAlfu0AABaRINE30qhT1a8Npf87//FJNcI5LxHstnABzCGeAP4bgJXDJyo/xR19AHgu+JoMQETfCARxIMoJHD3/+SgArmQBYBemsN+vWbY/KzdPwMj9BrA8P/cUrBcNXQNIQJEzwguAWisAp/T8FwBQYg8Fd0wI7Ygrev7PAHn/r9OvAJZ/Q3zRoiMCIDsNAApc0Y861ArAUOie/37irWlNiJKGERDE/wfAdSvy79Fr4QJ4DgC54+7nDx0sn3WbcpcFADA4swFgRaXAtlf0cwfw4Mg/PguAkWsAORAQLEWPAwIYfdWTuIT7ci3EyRoA/5QD/LLyHgBNrymapr+MSJMnsILzVYsjV9F2gISIFMShWwDcJwIAdzMd/t42/Ae0H5vP+zl5zgAAAABJRU5ErkJggg=="
                                        src={zork.src}
                                        alt="Preview"
                                    />
                                    <LazyLoadImage
                                        threshold={420}
                                        delayTime={50}
                                        effect="blur"
                                        // eslint-disable-next-line max-len
                                        placeholderSrc="data:image/png;base64,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"
                                        src={vanilla.src}
                                        alt="Preview"
                                    />
                                    <LazyLoadImage
                                        threshold={420}
                                        delayTime={50}
                                        effect="blur"
                                        // eslint-disable-next-line max-len
                                        placeholderSrc="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAEKklEQVRYR41X22pUQRA8c35CRBB8EHwQfBB8EARBEEFERPGWTzDxQlACgaCEhP0Yr4i5md1P2+N0d1V3z2xEl5xLdud0V1dVz8wpn1+tTaUM9qnXMsqp/smX8r0eZZgwZKo3y3qalnId6nUalvWQ/+UYZAAGa4h60pDjMIz1xu6LhLTjy/rLOjySKZh6kgdsBJBJbMRXAAIEiR2ADBBUUU9NaLGYeKz3AioAbAgAVC/ZMKBnQZOzQK08MUEGfAAI7RkQNnoAX18nAKweVwHj7PQAKgVL0K4MQA6nH48qA1IXqF8B8O3tCzBgA/vKhS7XX4KqByB1kmGCLEZTZkCSR+UGIMnwfRMAIHkG4EaMmFYpDmVA72HClFwJYOUOwIzo+guQH++fT+4z9xzZQHuwS2BwBwAvZFBau9iavhYQ0J3d4CYUkD+3niUTJgnICDsEBtT4mQXcS1L5PndAdrt7gO1HgL+2nzZ9QzaUfsPjJybIAPxeRiYAvQTeiioLGJbQBztP9DGyrHOAVw9G8GsAqMbs5JAIDQOp17X65Af3gIQ/+vi4lSAlTxwEt0hkAKxqa38vQVE3RktVEwxn33K8+6hhwCVASuMjfWqinFASswIqoM9A42y8nJwsl5O9hx32kMNRJn2jalYvAAgyrqsMsP/bDimn+w/cOrn6PuTgMO0XqzzuY8aEcHm+d/275CLWfHa/BaAK5poQEELpRcFACiyZ9kQkb7sgm7Bd4yqAe2dKsEoqeqXRXGQgXFgdmV1jXwsqRAzx3wTyYnbXy02hlOYiM1pvxpSQVUf1GJ3pR1c1BlSD4qnF7E7XBSvhSKxBAf0pBENBAnRAmk9Y+VgrylO0FrmY3e7tFRshqqpMBBfWCfQBr7p2m8B5HsBXMRGhQJK12L8VDGRqoh6XwwRBwgbAGCyglXKlwQB4M5KscRZ7N1MXtPRHN7B+APinD9LeouYJ/bMEyDXfvdFI4MQSpdcMznLySTeOXn3fitEJlti2mQFCXF7mn64384CHA4DobnCW9W/AhCd8JWUMbb8JTHQeON251nYBfIDd+WobAoCEsQWo6t9IEoZtfQAWpGrqL9z93r5qALIBM3J2glnRDibEVQGk2dEheBxWn0wIIOVk64pCcn8TiNKWfQ8PeOWW1A//3rqRILLmPg9kFo4/XObK3kwSYRqGCwa49jcAnAUCAPjEgs0FJoVFq/dHm5d8UxquxUCVBi0oP7JfOuptWwYZ+m0ZnG+teEYbHr27GF0A2n1wN6OZ/rYTiurlLQlbNAH4PwCov+Q7fHPBu+Cvmwi6VK6aJBLay2r9X98RegDRcrEOcFXEb4cb52NP2DAAE6qGmNm4J0iU2ysZJcDWHCyY1Kn/XYK0NB+sn4s2pPMrFavLJ8LBByoBEscbkuZrZFCj+URERsKIfwACPAgjfb4m/gAAAABJRU5ErkJggg=="
                                        src={postapoc.src}
                                        alt="Preview"
                                    />
                                    <LazyLoadImage
                                        threshold={420}
                                        delayTime={50}
                                        effect="blur"
                                        // eslint-disable-next-line max-len
                                        placeholderSrc="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAFV0lEQVRYR22WvYpkRRiGq0737PUoIqJmxsZegH8osoiI4vykXoAXYLK7LruuYLCJgQYGCgYigiiiRiYGIho6M93H9++rc7rXHmqqu2fOeZ/vfb+qU/3013/m1mb84DXvW9c7zvs2tV3b9F3b4v0J5pNp3250jGnG+1nzdtPatve2mXrr09TwC3eY2r5v2vU8tau2aZeY/503GJw5Or7vbYe5n/7yt7QNMUv4AACftxSHMMcNQmAYAOIaAQCIACC+Mz6EJgCsxQMA8R3UAPBXlS8ID1ePOnALuhAHAlDVcz4BAKvfQHw4AAheeY07XEH8UhDlQpcLVwIA7+nPfyoCm7DYPyKIA9uVA4yixA8B4kBFcACwuHDJCIYDP/0xIiBARcAaPNADEFcfTIyC1huA9hugtQkOTOmBvRyY4ID74HL0gZ0gwHX1wNkPvy8AdCIQawc26z6gOCAMYAhGMHGkBwzgPmATXgmixAsApUGun33/2wDoisAudKyIcoFNSAfkxBEAqxeAeqCasBywCwbILPv5PVcBAM6//fHAAUJwKS4AbsIBIAhUDhc4rwFAgTvi6pUDasSIlxPXjACDi72ff/PdaELtARVBHKD9rIcAApGwZ4tj1St/iw8AQHAZEsD7ATufldcKAAD3gYuvvj5qQlbvGCZBQEiJGqAgKLwBBPtOI+J0wRuRFzIjUC9EnDDVgHLg4ssv7ADzKAcAwKoJ4nayCxQUSInHBWipeP0CjfCzEgqinODmswMEZ255/eLzh0sEI3+7MMUJAwRGLtD2jIgTwvanB1YA3hUzhnhF8Nk9RyAx3INzhmpJDHbCECXelT/+hxceARBEu0g1ZIS9v2Igfwr3i4cfrpahYzAAYdgHFE0kK4AeBzhTnBDzWAUWrxiWLU3BQtwxCeD80w/A4hfvJSckurhRzZj2ghirXob/2cMRVAwBSdXxL9X7gn7+yfuJIPdQ5eVA+iC9Yf4Iq2vx2aQQdQwLQOHy6WjbBTCqD8DZ/dM8De2A+4D3NoT7oJZmNikI84+yPwP3jQMrCF69Eq8wZ+XPC3DJ6d2b/+PAuO9KnPevAwvFDWFKOhAXRgzJOQcULuACUKfJMkTw3p0XDwDk7CMjlVfrUDwRUNwR2D7ed90H9nAZ+lx2EeDd2y8IwA2ILyoCfV7BjL5N9nFgXjkhF+Ts0ogGcPXYuCNud+TAO7eeHw4YBF9KmKVUT9SiySzRisDv5+EEb6JWVf4SLQCcjgzCbk0PvH3ruQC4erlw5IYf0aoLv9EHA6BAMKcPain6v1eVB6QNCOfV37r1bDaisjzWVARZ4oZI8xECT8SGc4KdIIAd8aZyXP0Wl8cJzOs+6G/efnIVQapXGbZ/xFDrnmtfonic1CzxioSF2WY7sPWYOSjuPlAUdODmncfTvlqWrvdIXNVz3a/s7wTAGZFz01wd6x2JR3MKGuBkAAiKINmQ+hsfPZYILDxe41tn4a3X1UscYzoCMESqq8pT/X4GRMQFVgCv36UDR6/xTcST/RqgxD2zJ8o2nopodVl/ovMUI9gnBm3HBfDavSd0FF1XP/bmsfarcj6KWTkFMQ8X+Hc6RHE+GgNAQdhPYQ+cLASxAnj1fpqQDCFZutK5Vva2vSAKAJ9RfQF0RYCujwMWLICtTkOMoB5K/ZWPnzowYDaFLOEup7Wf7FUx3tN2DZ6OCCQA7j/o6rH5RJjH2QLY8z2fiATwZtVffvD0AgDxql7vtCM9an9Zb/E6IRFgOgRYiyeC3d7V6/HMZfjSg2cOWk6V54fifuy681V9cqe4AOSIj+aLA8zeS3DJH/ZLPABpwv8AuNkjVHEtQeUAAAAASUVORK5CYII="
                                        src={merchant.src}
                                        alt="Preview"
                                    />
                                </PrefixImages>
                            </IntroCard>
                        </Container>
                    </LandingCardFirst>
                    <LandingCard
                        ref={featuresRef}
                        style={{
                            justifyContent: 'flex-end',
                        }}
                    >
                        <div
                            style={{
                                position: 'relative',
                                bottom: -20,
                                padding: '0 30px',
                            }}
                        >
                            <div
                                style={{
                                    display: 'flex',
                                    flexDirection: 'column',
                                    alignItems: 'center',
                                    width: '100%',
                                }}
                            >
                                <div
                                    style={{
                                        display: 'flex',
                                        justifyContent: 'center',
                                    }}
                                >
                                    <LeftSparkleIcon
                                        highlight
                                        style={{
                                            height: 60,
                                            width: 60,
                                            marginTop: 20,
                                        }}
                                    />
                                    <div
                                        style={{
                                            display: 'flex',
                                            flexDirection: 'column',
                                            alignItems: 'center',
                                        }}
                                    >
                                        <HeaderEasel
                                            style={{
                                                height: 26,
                                                width: 34,
                                                marginBottom: 30,
                                            }}
                                        />
                                        <TagLineMed
                                            style={{
                                                maxWidth: 500,
                                                textAlign: 'center',
                                            }}
                                        >
                                            Visualize your favorite characters with Image Generation.
                                        </TagLineMed>
                                    </div>
                                    <RightSparkleIcon
                                        highlight
                                        style={{
                                            height: 60,
                                            width: 60,
                                            marginTop: 20,
                                        }}
                                    />
                                </div>
                                <TagSublineSmall>
                                    Use powerful image models to depict characters and moment from your
                                    stories.
                                </TagSublineSmall>
                                <FlexColSpacer min={25} max={25} />
                                <SkipButton onClick={scrollPricing}>
                                    <div>
                                        <span>Get started</span>
                                        <SmallArrowIcon />
                                    </div>
                                </SkipButton>
                            </div>
                        </div>
                        <div
                            style={{
                                height: 489,
                                width: '100vw',
                                backgroundImage: `url(${ImageGenBackground.src})`,
                                backgroundRepeat: 'no-repeat',
                                backgroundSize: '3000px 489px',
                                backgroundPosition: 'center',
                            }}
                        ></div>
                    </LandingCard>

                    <LandingCard ref={featuresRef}>
                        <Container>
                            <InfoCard>
                                <div>
                                    <AdventureSample
                                        nodeTypes={{
                                            adventure: [
                                                new AdventureNode('adventureSay', Say.src, /^> You \S* "/),
                                            ],
                                        }}
                                    >
                                        <div className="ProseMirror">
                                            <p className="adventureStory adventureStoryEnd">
                                                With a glazed stare, you watch and ponder what you see in the
                                                orb: random images from all around the world. You gaze into
                                                the globe and contemplate what these images mean to you. A
                                                forest, an ocean, a desert, a mountain range... you spend
                                                countless hours staring at the image, pondering its meaning.
                                            </p>
                                            <p className="adventureInput adventureSay">
                                                &gt; You sigh and rest your weary eyes for a moment, {'"'}
                                                Mayhaps a break is in order.{'"'}
                                            </p>
                                            <p className="adventureStory adventureStoryEnd">
                                                The images in this orb are so strange and foreign that you can
                                                {"'"}t help but feel a sense of awe at their beauty. Their
                                                colors and shapes speak to your soul and leave you feeling
                                                peaceful and refreshed. Your thoughts wander. What wonders
                                                await you when you look deeper?
                                            </p>
                                        </div>
                                        <div>
                                            <ConversationInputContainer
                                                mode={StoryMode.adventure}
                                                inputModeIcon={Do.src}
                                            >
                                                <ConversationInput
                                                    disabled={true}
                                                    placeholder={'What would you like to do?'}
                                                />
                                                <ConversationStoryControls>
                                                    {DefaultInputModes.map((mode, index) => {
                                                        return (
                                                            <ControlButton
                                                                key={index}
                                                                aria-label={mode.name}
                                                                style={{
                                                                    background:
                                                                        index === 0
                                                                            ? Dark.colors.bg3
                                                                            : Dark.colors.bg0,
                                                                    color:
                                                                        index === 0
                                                                            ? Dark.colors.textHeadings
                                                                            : Dark.colors.textMain,
                                                                }}
                                                            >
                                                                {mode.name}
                                                            </ControlButton>
                                                        )
                                                    })}
                                                </ConversationStoryControls>
                                            </ConversationInputContainer>
                                        </div>
                                    </AdventureSample>
                                </div>
                                <InfoBlock>
                                    <SwordsIcon />

                                    <TagLineMed>Begin your (text) adventure here.</TagLineMed>
                                    <TagSublineSmall>
                                        Open-ended not quite your style? Enable&nbsp;the Text Adventure
                                        Module.
                                    </TagSublineSmall>
                                    <SkipButton onClick={scrollPricing}>
                                        <div>
                                            <span>Start your quest</span>
                                            <SmallArrowIcon />
                                        </div>
                                    </SkipButton>
                                </InfoBlock>
                            </InfoCard>
                        </Container>
                    </LandingCard>
                    <LandingCard>
                        <Container>
                            <MobilePaddedInfoBlock>
                                <ThemeIcon />
                                <TagLineSmall>Customize the editor how&nbsp;you&nbsp;like.</TagLineSmall>
                                <InfoSeparator />
                                <TagSublineSmall>
                                    Design this space for yourself with our theme editor. Freely edit fonts,
                                    sizes, and&nbsp;the color scheme at any time.
                                </TagSublineSmall>
                            </MobilePaddedInfoBlock>
                            <HiddenInfoBlock>
                                <TextIcon />
                                <TagLineSmall>Take your writing anywhere.</TagLineSmall>
                                <InfoSeparator />
                                <TagSublineSmall>
                                    Whether you’re at home or on the go, NovelAI is always accessible to write
                                    something new.
                                </TagSublineSmall>
                            </HiddenInfoBlock>
                            <InfoBlock>
                                <LockIcon />
                                <TagLineSmall>
                                    Secure your
                                    <br />
                                    writing.
                                </TagLineSmall>
                                <InfoSeparator />
                                <TagSublineSmall>
                                    No one except you will be able to read your stories. Every story stored on
                                    our servers is encrypted with{' '}
                                    <LandingLink
                                        target="_blank"
                                        href="https://libsodium.gitbook.io/doc/advanced/stream_ciphers/xsalsa20"
                                    >
                                        XSalsa20
                                    </LandingLink>
                                    .
                                </TagSublineSmall>
                            </InfoBlock>
                        </Container>
                    </LandingCard>
                    <LandingCard>
                        <Container>
                            <InfoCardReverse>
                                <InfoBlock>
                                    <PencilIcon />
                                    <TagLineMed>Steer the AI down&nbsp;a&nbsp;new&nbsp;path.</TagLineMed>
                                    <TagSublineSmall>
                                        Using AI Modules, you can draw upon specific themes, replicate famous
                                        <br />
                                        writers, or even train one with your own data.
                                    </TagSublineSmall>
                                    <SkipButton onClick={scrollPricing}>
                                        <div>
                                            <span>Begin writing</span>
                                            <SmallArrowIcon />
                                        </div>
                                    </SkipButton>
                                </InfoBlock>
                                <div>
                                    <FakeModal>
                                        <div>
                                            <FakeModalTitle>AI Modules</FakeModalTitle>
                                            <ModalClose />
                                        </div>
                                        <ModuleCategory>
                                            <div>
                                                <div>Style</div>
                                                <div>Emulate your favorite writers.</div>
                                            </div>
                                            <ArrowDownIcon />
                                        </ModuleCategory>
                                        <div>
                                            <Module>
                                                <LazyLoadImage
                                                    threshold={420}
                                                    delayTime={50}
                                                    effect="blur"
                                                    // eslint-disable-next-line max-len
                                                    placeholderSrc="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAFbUlEQVRYR3WXa4scRRSGu6d7Nv5gFf2wRFBXNERQoohiEAWjiJCgiCYfoquERVRYDAlIRPRX6M50t+/lnKrqcc2kpqZ7euo85z2Xqu1//+ts6ZauW5alm+e5WzDmacKIOe7xezzV9X3f9RuOTbcZOIYy817fc/R4tMeamOYO62LtiQPrx8x7fKB/8icAsPCCGwuetuE0PuEH+MxV8DARsLQAaNwQBAiIzSAAvPlJ/ICGBKE1DcFZ9jD63/540ADwgbX3vKYqNl8BenoPg1UFfKYCGPGUAFIBex4AYVwAj598HyGg/A6DIWKsQuC1pUDxvqogRQQAELwUNIpHgw2A7VAeAPz6+L5cU+wpMwxPTQgUN0IoBE0OhALD2ISBIaAKfK3yoAI4xzLkWO+X87sGYPyVLA7BNLcqOD8yBjUBbXzIPGBIDhLRYagxlxJxrSQ8+/FL5zcMSBoqAONFBQI1CjDBMglb48MwOh8IEZVAJVxhMJAQ9J5qqgoQzu9OPw+AGv/0fspwCICJmDngWG9Gez+Mo2cqESpsCCoAQ6iKIu4uqijDb+5+jM+mUgJi2DDHXkoQyDnw/wAjFBgIVAAAWCDgafQaKpF9RwrcuXNzBZDG9wnBcASYHFEVuAml9yMUSADNUQ0MhVPSAKlEmXn/k1tv031VgLxX/Kduv993hNjH9RRVsGQOwFN6LOMaW0DENb8jIAHwfIFQtgdM9pUP3r/uHIA0Mg6IPaSn8V0LwSrJVggPM+Yyvt12W8w5qMLIZ2IkRLYoRzKS8J0bJyUENE7vdwDwgAI5CwBiSYEMAY3C+NbzUYJAiS3VYGISAr/hQP9yjyAA3zi/ce2qxcW7va8AF1AgQXYAcBjY6Fz/9HQLo0fwnsY1CCI1AgLPWg0CBIRAYlx76XmtyQ2nBbiA5wTQzEF1mCeHADJ+1F2BcQ9cC2LQkBIBUCHYKQPglavPKhiUlzlABYpRAPwD4xwX/I7PSDfuBU44Sn+FA8afEshoEIIRQBBQgUkJFRQKbucJcHL8tBuVPHQICgAM/73fAQD3AMAw7JWIzAPH+GhIzwngIQgpg8F8wH5BgBKKBOA6J8fPNArM3Y6GUvbw3ioQYJEKszYbLkrvaMwqpPGcSxhCBZXmYBV0qKEvr77w3DoJCYCRKqRxKcAkzTxAdbPrMcYVYjxQIPKA/SLDQAApEGF4/eXjCkCJaSQBCkiEQI3KiYh9TJsOZSWAIRh3zjTMEFwOoH2EOUCIG9df9L6Al/qAIBgGeswKsCIeASBkV7QhqhKMuYxnAmqOHFAIuJFlOWKF9956LcowK4EyuwWrEaVxfXYpUgXlIjYGesGWSwjlBGGi/EoFBIByQPsEKyEU+Ojmm3EgCYBQgGrQawJ4sEKskA4u+pU7G7ccQnBxxpoQjjlLNUowjDMJXYaRA5/dencFoIbU5IJ6Q+QFZ29Y2r9id7Mn3HDcdhPEG5JKL7O/eN8AfHH7w0sBlA+SnM3Ju2QCWIEK4bbqfHCjqRuRmk/GfQUQ+t37+lM5ozM8X9qaw3huz6FAeq+DS3OsylAYIodjXWPuax3nQn41tNNvbwcAy6tuSisIqKAzYqjSAmQodOAIAIUkhg0eGGcC6nyP/2cPvgoB4lgWCiRABakQCkEeVNUUmqpQp84Yp7eN524AZUvuf/7p3n8AeAB1tofXERL9WRX3KkQARFJ66TVEyfgCFvsJWc7P75ckzHMBvXaiORdm/o3Qyq979XwfncxVEf8scZRao4hv52kEnx8+PL0coDklz8yB9LyosA5DgVBCR3wzzgGQyiSYrh89+sGNONqxjuc6fqUKB95fClBLkuuEfy7O2Pg9R4DyHp74F/8Gqee8+bepAAAAAElFTkSuQmCC"
                                                    src={sherlock.src}
                                                    alt="Sherlock"
                                                />
                                                <div>
                                                    <div>Arthur Conan Doyle</div>
                                                    <div>
                                                        A master detective solves crimes with logic, wit, and
                                                        a keen mind.
                                                    </div>
                                                </div>
                                            </Module>
                                            <Module>
                                                <LazyLoadImage
                                                    threshold={420}
                                                    delayTime={50}
                                                    effect="blur"
                                                    // eslint-disable-next-line max-len
                                                    placeholderSrc="data:image/png;base64,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"
                                                    src={poe.src}
                                                    alt="Poe"
                                                />
                                                <div>
                                                    <div>Edgar Allan Poe</div>
                                                    <div>
                                                        The darkness in man{"'"}s soul, the twisted will of a
                                                        madman.
                                                    </div>
                                                </div>
                                            </Module>
                                            <Module>
                                                <LazyLoadImage
                                                    threshold={420}
                                                    delayTime={50}
                                                    effect="blur"
                                                    // eslint-disable-next-line max-len
                                                    placeholderSrc="data:image/png;base64,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"
                                                    src={lovecraft.src}
                                                    alt="Lovecraft"
                                                />
                                                <div>
                                                    <div>H.P. Lovecraft</div>
                                                    <div>
                                                        In forgotten corners of the universe beings streach
                                                        out etheral tendrils.
                                                    </div>
                                                </div>
                                            </Module>
                                        </div>
                                        <ModuleCategory>
                                            <div>
                                                <div>Theme</div>
                                                <div>
                                                    Drive your writing in a specific thematic direction.
                                                </div>
                                            </div>
                                            <ArrowDownIcon />
                                        </ModuleCategory>
                                        <div>
                                            <Module>
                                                <LazyLoadImage
                                                    threshold={420}
                                                    delayTime={50}
                                                    effect="blur"
                                                    // eslint-disable-next-line max-len
                                                    placeholderSrc="data:image/png;base64,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"
                                                    src={darkfantasy.src}
                                                    alt="Dark Fantasy"
                                                />
                                                <div>
                                                    <div>Dark Fantasy</div>
                                                    <div></div>
                                                </div>
                                            </Module>
                                            <Module>
                                                <LazyLoadImage
                                                    threshold={420}
                                                    delayTime={50}
                                                    effect="blur"
                                                    // eslint-disable-next-line max-len
                                                    placeholderSrc="data:image/png;base64,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"
                                                    src={dragons.src}
                                                    alt="Dragons"
                                                />
                                                <div>
                                                    <div>Dragons</div>
                                                    <div></div>
                                                </div>
                                            </Module>
                                            <Module>
                                                <LazyLoadImage
                                                    threshold={420}
                                                    delayTime={50}
                                                    effect="blur"
                                                    // eslint-disable-next-line max-len
                                                    placeholderSrc="data:image/png;base64,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"
                                                    src={mars.src}
                                                    alt="Mars"
                                                />
                                                <div>
                                                    <div>Mars Colonization</div>
                                                    <div></div>
                                                </div>
                                            </Module>
                                        </div>
                                    </FakeModal>
                                </div>
                            </InfoCardReverse>
                        </Container>
                    </LandingCard>
                    <LandingCard>
                        <Container>
                            <MobilePaddedInfoBlock>
                                <MindIcon />
                                <TagLineSmall>Expand your story’s&nbsp;potential.</TagLineSmall>
                                <InfoSeparator />
                                <TagSublineSmall>
                                    Our service keeps up to 2048 tokens of memory, in non-nerd speak, the AI
                                    remembers a lot.
                                </TagSublineSmall>
                            </MobilePaddedInfoBlock>
                            <InfoBlock>
                                <SlidersIcon />
                                <TagLineSmall>
                                    Tweak the AI’s output the&nbsp;way&nbsp;you&nbsp;like&nbsp;it.
                                </TagLineSmall>
                                <InfoSeparator />
                                <TagSublineSmall>
                                    Take control of the AI with a plethora of options. Keep things consistent,
                                    or maybe&nbsp;go a little more random.
                                </TagSublineSmall>
                            </InfoBlock>
                            <HiddenInfoBlock>
                                <ArrowCircleIcon />
                                <TagLineSmall>Continue your old legacies here.</TagLineSmall>
                                <InfoSeparator />
                                <TagSublineSmall>
                                    Import from AIDCAT or AIDWI and revive those stories
                                    with&nbsp;a&nbsp;place to stay here.
                                </TagSublineSmall>
                            </HiddenInfoBlock>
                        </Container>
                    </LandingCard>
                    <LandingCard>
                        <Container>
                            <InfoCard>
                                <div>
                                    <FakeLorebook>
                                        <FakeSidebar>
                                            <div>
                                                <FakeLoreHeader>
                                                    <div>
                                                        <BookIcon />
                                                        <span>Lorebook</span>
                                                    </div>
                                                    <div>
                                                        <ImportIcon />
                                                        <ExportIcon />
                                                    </div>
                                                </FakeLoreHeader>
                                                <FakeSearch>
                                                    <span>Search here...</span>
                                                    <SearchIcon />
                                                </FakeSearch>
                                                <FakeCategory>
                                                    <span>Worlds</span>
                                                    <div>
                                                        <PlusIcon />
                                                        <ArrowDownIcon />
                                                    </div>
                                                </FakeCategory>
                                                <FakeEntrySelected>
                                                    <div>Abberantis</div>
                                                    <div>abberantis, world, the holy world, ho...</div>
                                                </FakeEntrySelected>
                                                <FakeEntry>
                                                    <div>Crooked Domain</div>
                                                    <div>dark elves, darkness, evil spirits, cata...</div>
                                                </FakeEntry>
                                                <FakeCategory>
                                                    <span>Characters</span>
                                                    <div>
                                                        <div />
                                                        <ArrowUpIcon />
                                                    </div>
                                                </FakeCategory>
                                            </div>
                                            <FakeAddNew>
                                                <div>
                                                    <PlusIcon />
                                                    <span>New Entry</span>
                                                </div>{' '}
                                                <div>
                                                    <PlusIcon />
                                                    <span>New Category</span>
                                                </div>
                                            </FakeAddNew>
                                        </FakeSidebar>
                                        <FakeRight>
                                            <FakeLoreTitle>
                                                <div style={{ marginRight: 200, flex: '1 1 0' }}>
                                                    Abberantis
                                                </div>
                                                <div>
                                                    <ModalClose />
                                                </div>
                                            </FakeLoreTitle>
                                            <FakeTextEdit>
                                                <div>Description</div>
                                                <div>
                                                    Abbarantis is a frigid desert planet where temperatures
                                                    can plummet to -100°F (-73°C) or soar above 200°F (93°C).
                                                    While some may enjoy the extreme heat, others prefer the
                                                    cold winds that sweep across the sand dunes.
                                                </div>
                                            </FakeTextEdit>
                                            <FakeKeyArea>
                                                <div>Keys</div>
                                                <div>Type in the area below, then press enter to save.</div>
                                                <FakeKeyEntry>
                                                    <div>
                                                        <span>Enter a key here.</span>
                                                    </div>
                                                    <FakeKeyBox>
                                                        <div>
                                                            <div>Active Keys</div>
                                                            <div>Select a key to edit it</div>
                                                        </div>

                                                        <div>
                                                            <div>abberantis</div>
                                                            <div>world</div>
                                                            <div>the holy world</div>
                                                            <div>holiness</div>
                                                        </div>
                                                    </FakeKeyBox>
                                                </FakeKeyEntry>
                                            </FakeKeyArea>
                                        </FakeRight>
                                    </FakeLorebook>
                                </div>

                                <InfoBlock>
                                    <WorldIcon />

                                    <TagLineMed>Take control of your&nbsp;world.</TagLineMed>
                                    <TagSublineSmall>
                                        The Lorebook is the place to keep track&nbsp;of&nbsp;your world and
                                        make sure that&nbsp;the&nbsp;AI obeys its rules.
                                    </TagSublineSmall>
                                    <SkipButton onClick={scrollPricing}>
                                        <div>
                                            <span>Create your universe</span>
                                            <SmallArrowIcon />
                                        </div>
                                    </SkipButton>
                                </InfoBlock>
                            </InfoCard>
                        </Container>
                    </LandingCard>
                </LandingContent>
                <LandingSpacer />
                <LandingTestimonials>
                    <div>
                        <Testimonials>
                            <TestimonialLead>Still not convinced?</TestimonialLead>
                            <TestimonialTitle>Don’t just take our word for it.</TestimonialTitle>
                            <div>
                                <div>
                                    <TestimonialBox>
                                        <span>
                                            “NovelAI is an amazing tool that just keeps getting better and
                                            better.{' '}
                                        </span>
                                        <TestimonialEmphasis>
                                            It is a swiss army knife that has the feel of a writing helper to
                                            help fuel the writer{"'"}s creativity, and a game for users to
                                            play with.
                                        </TestimonialEmphasis>
                                        <span>”</span>
                                        <div>
                                            <img src={GooseBlue.src} alt="Blue Goose" />
                                            felixchowmusic
                                        </div>
                                    </TestimonialBox>
                                    <TestimonialBox>
                                        <span>“Amazing, a real successor to AI Dungeon. </span>
                                        <TestimonialEmphasis>
                                            All the bells and whistles really put you in control of the story,
                                            especially the modules.
                                        </TestimonialEmphasis>
                                        <span>”</span>
                                        <div>
                                            <img src={GooseGreen.src} alt="Green Goose" />
                                            Anonymous
                                        </div>
                                    </TestimonialBox>
                                    <TestimonialBox>
                                        <span>
                                            “As a hobby writer I use the AI to help me write my stories.
                                            Perhaps when I{"'"}m running out of ideas or just want to
                                            experiment.{' '}
                                        </span>
                                        <TestimonialEmphasis>
                                            It{"'"}s an extension to my creative writing and it{"'"}s
                                            absolutely fabulous.
                                        </TestimonialEmphasis>
                                        <span>”</span>
                                        <div>
                                            <img src={GooseOrange.src} alt="Orange Goose" />
                                            Otviss
                                        </div>
                                    </TestimonialBox>
                                </div>
                                <div>
                                    <TestimonialBox>
                                        <span>“</span>
                                        <TestimonialEmphasis>
                                            I really love the UI and general sense of authorial freedom that
                                            comes with Novel AI -{' '}
                                        </TestimonialEmphasis>
                                        <span>
                                            the ability to fiddle with systems as much or as little as I
                                            please, combined with decent presets for quicker {'"'}jump in and
                                            go{'"'} play makes it worth paying for just for that.”
                                        </span>
                                        <div>
                                            <img src={GoosePurple.src} alt="Purple Goose" />
                                            Anonymous
                                        </div>
                                    </TestimonialBox>
                                    <TestimonialBox>
                                        <span>“I am very much satisfied with NovelAI, and </span>
                                        <TestimonialEmphasis>
                                            it definitely scratches that storytelling itch I{"'"}ve had for a
                                            while.
                                        </TestimonialEmphasis>
                                        <span>”</span>
                                        <div>
                                            <img src={GooseYellow.src} alt="Yellow Goose" />
                                            TheIrishNinjas
                                        </div>
                                    </TestimonialBox>
                                    <TestimonialBox>
                                        <span>There{"'"}s a lot to love—responsive devs, privacy, and </span>
                                        <TestimonialEmphasis>
                                            the ability to write about anything you think of.
                                        </TestimonialEmphasis>
                                        <span></span>
                                        <div>
                                            <img src={GoosePink.src} alt="Pink Goose" />
                                            Anonymous
                                        </div>
                                    </TestimonialBox>
                                </div>
                            </div>
                        </Testimonials>
                    </div>
                </LandingTestimonials>
                <LandingTiers>
                    <Container style={{ flexDirection: 'column', height: '100%' }}>
                        <div style={{ width: '100%' }}>
                            <TierContainer ref={pricingRef}>
                                <TierText>
                                    <span>Begin your adventure.</span>
                                    <span>Select a Tier to get started.</span>
                                </TierText>
                                <div>
                                    <PaperTierContent>
                                        <Link href="/trial" passHref>
                                            <TierGetStarted color={'#D9FFE1'}>
                                                Get Started for Free
                                                <ArrowRightIcon />
                                            </TierGetStarted>
                                        </Link>
                                    </PaperTierContent>
                                    <TabletTierContent>
                                        <Link href="/register" passHref>
                                            <TierGetStarted color={Dark.colors.textHeadings}>
                                                Get Started
                                                <ArrowRightIcon />
                                            </TierGetStarted>
                                        </Link>
                                    </TabletTierContent>
                                    <ScrollTierContent>
                                        <Link href="/register" passHref>
                                            <TierGetStarted color={Dark.colors.textHeadingsOptions[1]}>
                                                Get Started
                                                <ArrowRightIcon />
                                            </TierGetStarted>
                                        </Link>
                                    </ScrollTierContent>
                                    <OpusTierContent>
                                        <Link href="/register" passHref>
                                            <TierGetStarted color={Dark.colors.textHeadingsOptions[3]}>
                                                Get Started
                                                <ArrowRightIcon />
                                            </TierGetStarted>
                                        </Link>
                                    </OpusTierContent>
                                </div>
                            </TierContainer>
                            <TierFinePrint>
                                *Users start with 50 free generations, which refill back to 50 upon creating
                                an account.
                            </TierFinePrint>
                            <TierFinePrint>
                                **For images of up to 640x640 pixels and up to 28 steps when generating a
                                single image. Does not include img2img generations.
                            </TierFinePrint>
                        </div>
                    </Container>
                </LandingTiers>
                <LandingBottom>
                    <Container>
                        <BottomLinks>
                            <Link href="/terms">Terms of Service</Link>
                            <Tooltip delay={500} tooltip="For Business / Press inquiries" inheritHeight>
                                <a href="mailto:contact@novelai.net">Contact Us</a>
                            </Tooltip>
                            <a href="mailto:support@novelai.net">Support</a>
                            <Link href="/assets">Press Kit</Link>
                            <Link href="/updates">Updates</Link>
                        </BottomLinks>
                    </Container>
                    <Container>
                        <Copyright>Copyright © 2022 Anlatan</Copyright>
                    </Container>
                </LandingBottom>
            </LandingPage>
        </>
    )
}

export const LandingPage = styled.div`
    z-index: 1;
    box-sizing: border-box;
    height: var(--app-height, 100%);
    max-height: 100%;
    width: 100vw;
    max-width: 100%;
    display: flex;
    flex-direction: column;
    overflow: auto;
    position: relative;

    min-height: 420px;
    min-width: 320px;

    ::-webkit-scrollbar {
        width: 0; /* Remove scrollbar space */
        background: transparent; /* Optional: just make scrollbar invisible */
    }
`

export const TopBackground = styled.div<{ src: string }>`
    > div {
        position: absolute;
        left: 0;
        top: 0;
        height: 100%;
        width: 100%;

        background: url(${(props) => props.src});
        background-color: transparent;
        background-position: 0% 0%;
        background-repeat: no-repeat;
        background-size: cover;
        pointer-events: none;

        transition: opacity 2s ease-in-out;
    }

    > :last-child {
        background-image: linear-gradient(
            0deg,
            ${() => transparentize(0.28, Dark.colors.bg1) + ', ' + transparentize(0.28, Dark.colors.bg1)}
        );
    }
`
export const LandingTop = styled.div`
    display: flex;
    flex: 0 0 auto;
    flex-direction: column;
    justify-content: stretch;
    align-items: center;
    width: 100%;
    height: var(--app-height, 100%);
    height: 1015px;
    position: relative;
    background-color: ${(props) => props.theme.colors.bg4};
    padding: 20px 20px;
    @media (max-width: ${mobileSize}) {
        height: 695px;
        padding: 0px;
    }
`

export const LandingCardFirst = styled.div`
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-content: space-between;
    align-items: center;
    width: 100%;
    height: 1072px;
    @media (max-width: ${mobileSize}) {
        height: 1144px;
    }
`

export const LandingCard = styled.div`
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-content: space-between;
    align-items: center;
    width: 100%;
    height: 868px;

    max-width: 100vw;
    overflow: hidden;

    &:nth-child(2),
    &:nth-child(3),
    &:nth-child(5),
    &:nth-child(7) {
        height: 768px;
        background: ${(props) => props.theme.colors.bg1};
        border: 1px solid ${(props) => props.theme.colors.bg3};
        border-right: 0;
    }

    @media (max-width: ${mobileSize}) {
        height: auto;
        &:nth-child(1),
        &:nth-child(4),
        &:nth-child(6) {
            padding: 80px 0;
            height: auto;
        }

        &:nth-child(2n) {
            height: auto;
        }
    }
`

export const LandingContent = styled.div`
    width: 100%;
    flex: 0 0 auto;
    background: url(${ContentBackground.src}), ${(props) => props.theme.colors.bg1};
    background-repeat: no-repeat;
    background-size: cover;
`

export const LandingSpacer = styled.div`
    height: 155px;
    width: 100%;
    flex: 0 0 155px;
    flex: 0 0 auto;
    background: ${(props) => props.theme.colors.bg1};
    @media (max-width: ${mobileSize}) {
        height: 0;
    }
`

export const LandingTestimonials = styled.div`
    width: 100%;
    background: ${(props) => props.theme.colors.bg2};
    flex: 1 0 1057px;
    max-width: 100vw;
    > div {
        height: 1057px;
        padding-top: 125px;
        width: 100%;
        display: flex;
        justify-content: space-around;

        background: linear-gradient(
            180deg,
            ${(props) => props.theme.colors.bg1} 0%,
            ${(props) => props.theme.colors.bg2} 60.35%
        );
        @media (max-width: ${mobileSize}) {
            padding-top: 105px;
            height: auto;
        }
    }
`

export const LandingTiers = styled.div`
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-content: space-between;
    align-items: center;
    width: 100%;
    height: 1254px;
    flex: 0 0 auto;
    background: linear-gradient(
        180deg,
        ${(props) => props.theme.colors.bg2},
        ${(props) => transparentize(1, props.theme.colors.bg2)}
    );
    @media (max-width: ${mobileSize}) {
        height: auto;
    }
`

export const LandingBottom = styled.div`
    display: flex;
    flex-direction: column;
    justify-content: stretch;
    align-items: center;
    width: 100%;
    position: relative;
    background: ${(props) => props.theme.colors.bg1};
    padding-bottom: 30px;
    padding: 30px 20px;
    flex: 0 0 auto;
`

export const BottomLinks = styled.div`
    display: flex;
    flex-direction: row;
    gap: 20px;
    a {
        font-weight: 600;
    }
`

export const Header = styled.div`
    align-items: center;
    display: flex;
    flex-wrap: wrap-reverse;
    flex: 1;
    flex-direction: row;
    height: auto;
    padding: 20px 0;
    position: relative;
    margin: 0 auto;
    width: 100%;
    justify-content: center;
    padding-bottom: 70px;

    @media (max-width: ${smallerMobileSize}) {
        > :nth-child(4) {
            display: none;
        }
        > :nth-child(5) {
            margin-top: -5px;
        }
        padding: 30px 0 60px 0;
    }
`

export const Container = styled.div`
    max-width: 1200px;
    display: flex;
    flex-direction: row;
    justify-content: space-around;
    align-self: center;
    width: 100%;
    padding: 0 20px;

    flex: 0 0 auto;
    align-items: center;
    position: relative;

    @media (max-width: ${smallMobileSize}) {
        padding: 0 30px;
        flex-direction: column;
        align-items: center;
    }
`

export const LandingRow = styled.div`
    max-width: 1200px;
    display: flex;
    flex-direction: row;
    justify-content: space-around;
    width: 100%;
    padding: 0 20px;

    flex: 0 0 auto;
    font-weight: 900;
    align-items: center;
    position: relative;

    @media (max-width: ${smallerMobileSize}) {
        justify-content: flex-start;
        padding: 0 30px;
    }
`

export const Copyright = styled.div`
    margin-top: 15px;
    opacity: 0.7;
    font-weight: 600;
`

export const Title = styled.h1`
    color: ${(props) => props.theme.colors.textMain};
    font-family: ${(props) => props.theme.fonts.headings};
    font-size: 1.5rem;
    margin: 0.25rem 0px 0px;
    user-select: none;
    word-break: keep-all;
`
export const Logo = styled.div`
    aspect-ratio: 1/1;
    background-image: url(${PenTip.src});
    background-repeat: no-repeat;
    background-size: contain;
    height: 20px;
    width: 20px;
    margin: 0 10px 0 0;
`

export const Placeholder = styled.div`
    flex: 1%;
`

export const LandingLink = styled.a`
    font-weight: 400;
    color: ${(props) => props.theme.colors.textHeadings};
    text-decoration: underline;
`

export const Login = css`
    background: none;
    border: 0;
    color: ${(props) => props.theme.colors.textMain};
    cursor: pointer;
    font-family: ${(props) => props.theme.fonts.default};
    font-size: 1rem;
    font-weight: 600;
    padding: 5px 15px;
    text-decoration: none;

    &::selection {
        background-color: transparent;
    }
`
export const LoginLink = styled.a`
    ${Login}
`
export const LoginButton = styled.button`
    ${Login}
`

export const TagLine = styled.h2`
    font-size: 3.25rem;
    line-height: 3.25rem;

    @media (max-width: ${smallerMobileSize}) {
        font-size: 1.75rem;
        line-height: 2.5rem;
    }
`

export const TagLineOther = styled.h2`
    font-size: 3.25rem;
    line-height: 3.25rem;
    color: ${(props) => props.theme.colors.textMain};

    @media (max-width: ${smallerMobileSize}) {
        height: 3em;
        font-size: 1.75rem;
        line-height: 2.5rem;
    }
`

export const IntroLine = styled.h2`
    font-size: 2.625rem;
    font-weight: 700;
    color: ${(props) => props.theme.colors.textMain};

    @media (max-width: ${smallerMobileSize}) {
        font-size: 1.5rem;
    }
`

export const IntroSubLine = styled.p`
    color: ${(props) => props.theme.colors.textHeadings};
    font-weight: 700;
    font-size: 1.375rem;
    max-width: 596px;
    line-height: 38px;

    @media (max-width: ${mobileSize}) {
        font-size: 1rem;
        line-height: 30px;
    }
`

export const IntroText = styled.p`
    font-size: 1rem;
    font-weight: 600;
    line-height: 35px;

    @media (max-width: ${mobileSize}) {
        font-size: 0.875rem;
        line-height: 28px;
    }
`
export const IntroButtons = styled.div`
    display: flex;
    margin-top: 40px;
    > :last-child {
        padding: 12px 40px;
        font-weight: bold;
    }
    @media (max-width: ${mobileSize}) {
        margin-top: 30px;
        flex-direction: column;
        gap: 15px;
        width: 100%;
        align-items: center;
        > {
            max-width: 315px;
        }
    }
`

export const TagLineSmall = styled.h2`
    font-size: 1.375rem;
    & > * {
        color: ${(props) => props.theme.colors.textMain};
    }
`

export const TagLineMed = styled.h2`
    font-size: 1.625rem;
    & > * {
        color: ${(props) => props.theme.colors.textMain};
    }
`

export const TagSubline = styled.h3`
    max-width: 484px;
    text-align: center;
    font-size: 1rem;
    color: ${(props) => props.theme.colors.textMain};
    font-family: ${(props) => props.theme.fonts.default};
    line-height: 24px;

    @media (max-width: ${smallerMobileSize}) {
        text-align: start;
    }

    @media (max-width: ${smallerMobileSize}) {
        font-size: 0.875rem;
        line-height: 22px;
    }
`
export const TagSublineSmall = styled.h3`
    font-size: 0.875rem;
    font-weight: 600;
    line-height: 24px;
    color: ${(props) => props.theme.colors.textMain};
    font-family: ${(props) => props.theme.fonts.default};
`

export const MainButtons = styled.div`
    max-width: 1200px;
    display: flex;
    flex-direction: row;
    justify-content: space-around;
    width: 100%;
    padding: 0 20px;

    flex: 0 0 auto;
    font-weight: 900;
    align-items: center;
    position: relative;

    margin-top: 25px;
    margin-bottom: 25px;
    font-size: 1rem;

    > div {
        display: flex;
    }
    @media (max-width: ${smallerMobileSize}) {
        > div {
            flex-direction: column-reverse;
        }
        margin-bottom: 0px;
        font-size: 0.875rem;
    }
`

export const Start = styled.a`
    height: 70px;
    display: inline-flex;
    flex-direction: column;
    justify-content: center;
    font-family: ${(props) => props.theme.fonts.default};
    padding: 26px 60px;
    cursor: pointer;
    white-space: nowrap;
    user-select: none;
    flex: 0;
    text-transform: uppercase;
    letter-spacing: 0.05rem;
    line-height: 0;
    color: ${(props) => props.theme.colors.bg1};
    background-color: ${(props) => props.theme.colors.textHeadings};
    font-weight: 700;

    flex: 0 0 auto;
    transition: all ${(props) => props.theme.transitions.interactive};
    &:hover,
    &:focus {
        color: ${(props) => props.theme.colors.bg1};
        background-color: ${(props) => props.theme.colors.textMain};
    }
    &:active {
        background-color: ${(props) => darken(0.1, props.theme.colors.textMain)};
    }

    @media (max-width: ${smallerMobileSize}) {
        height: 58px;
    }
`

export const LearnBox = styled.button`
    height: 70px;
    outline: none !important;
    border: none;
    display: flex;
    flex-direction: row;
    align-items: center;
    font-family: ${(props) => props.theme.fonts.default};
    text-decoration: none;
    cursor: pointer;
    letter-spacing: 0.05rem;
    text-align: left;
    margin-right: 20px;
    background: none;

    flex: 0 0 auto;
    padding: 28px 64px;
    border: 2px solid #fff;

    > div {
        line-height: 1.5rem;
    }
    line-height: 0;

    > div {
        background-color: ${(props) => props.theme.colors.textMain};
    }

    transition: all ${(props) => props.theme.transitions.interactive};
    &:hover {
        color: ${(props) => props.theme.colors.textHeadings};
        border-color: ${(props) => props.theme.colors.textHeadings};
        > div {
            background-color: ${(props) => props.theme.colors.textHeadings};
        }
    }

    @media (max-width: ${smallerMobileSize}) {
        margin-right: 0px;
        margin-top: 10px;
        height: 48px;
        padding: 14px 64px;
    }
`

export const Arrow = styled.div`
    margin-top: 132px;
    background-image: url(${ArrowImage.src});
    background-repeat: no-repeat;
    background-size: contain;
    height: 28px;
    width: 11px;
    cursor: pointer;
    @media (max-width: ${mobileSize}) {
        width: 100%;
        margin: auto;
        margin-top: 50px;
        background-position: center;
    }
`

export const PreviewImage = styled.div`
    position: relative;
    width: 1028px;
    margin-top: 100px;
    border: solid 2px ${(props) => props.theme.colors.bg1};
    border-radius: 5px;
    max-width: 90vw;
    flex: 0 0 auto;
    @media (max-width: ${smallerMobileSize}) {
        display: none;
    }
`

export const MobilePreviewImage = styled.div`
    position: relative;
    display: none;
    margin-top: 60px;
    border: solid 2px ${(props) => props.theme.colors.bg1};
    border-radius: 5px;
    flex: 0 0 auto;
    @media (max-width: ${smallerMobileSize}) {
        display: block;
        width: 268px;
    }
`

export const IntroCard = styled.div`
    margin-top: 160px;
    display: flex;
    flex-direction: row;
    justify-content: space-between;
    width: 100%;
    align-items: flex-start;
    position: relative;
    overflow: hidden;
    @media (max-width: ${mobileSize}) {
        margin-top: 351px;

        width: auto;
        margin-left: auto;
        margin-right: auto;
    }
`

export const IntroTextSection = styled.div`
    max-width: 700px;
    @media (max-width: ${mobileSize}) {
        z-index: 1;
        background-color: ${(props) => transparentize(0.1, props.theme.colors.bg1)};
        padding-top: 90px;
    }
`

export const PrefixImages = styled.div`
    display: flex;
    width: 471px;
    flex-direction: row;
    flex: 0 0 471px;
    flex-wrap: wrap;
    .lazy-load-image-background {
        img {
            object-fit: cover;
            display: inline;
        }
        margin: 0;
        margin-bottom: 20px;
    }
    .lazy-load-image-background:nth-child(1) {
        img {
            width: 359px;
            height: 160px;
        }
    }
    .lazy-load-image-background:nth-child(2) {
        img {
            width: 326px;
            height: 120px;
        }
        margin-left: 145px;
    }
    .lazy-load-image-background:nth-child(3) {
        img {
            width: 207px;
            height: 100px;
        }
        margin-left: 65px;
    }
    .lazy-load-image-background:nth-child(4) {
        img {
            width: 121px;
            height: 66px;
        }
        margin-left: 20px;
    }

    @media (max-width: ${mobileSize}) {
        position: absolute;
        top: 0;
        left: 0;
    }
`

export const InfoCard = styled.div`
    display: flex;
    flex-direction: row;
    justify-content: space-between;
    width: 100%;
    align-items: center;
    position: relative;

    > div {
        max-width: 100%;
    }

    @media (max-width: ${mobileSize}) {
        align-items: center;
        padding-top: 40px;
        padding-bottom: 65px;
        justify-content: space-around;
    }

    @media (max-width: ${smallMobileSize}) {
        flex-direction: column;
        align-items: center;
        > div > div {
            transform-origin: bottom center;
        }
    }

    @media (max-width: 750px) {
        flex-direction: column;
        align-items: flex-start;
        > div > div {
            transform-origin: bottom left;
        }
    }
`

export const InfoCardReverse = styled(InfoCard)`
    @media (max-width: ${mobileSize}) {
        padding-top: 80px;
        padding-bottom: 53px;
        > :last-child {
            padding-bottom: 50px;
        }
    }

    @media (max-width: ${smallMobileSize}) {
        flex-direction: column-reverse;
    }
`

export const InfoBlock = styled.div`
    display: flex;
    flex-direction: column;
    width: 275px;
    flex: 0 0 auto;
    > h3 {
        margin: 0;
    }
    > h2 {
        margin-bottom: 10px;
    }
    > button:last-child {
        margin-top: 40px;
    }
    @media (max-width: ${smallMobileSize}) {
        align-self: center;
    }
`

export const HiddenInfoBlock = styled(InfoBlock)`
    @media (max-width: ${mobileSize}) {
        display: none;
    }
`

export const MobilePaddedInfoBlock = styled(InfoBlock)`
    @media (max-width: ${smallMobileSize}) {
        padding-bottom: 100px;
    }
`

export const InfoSeparator = styled.div`
    width: 40px;
    margin: 20px 0;
    border-top: solid 1px ${(props) => props.theme.colors.textHeadings};
`

export const SkipButton = styled(SubtleButton)`
    background-color: ${(props) => props.theme.colors.textHeadings};
    display: flex;
    > div {
        display: flex;
        align-items: center;
    }
    width: 250px;
    padding: 12px 0;
    flex: 0 0 auto;
    color: ${(props) => props.theme.colors.bg1};
    font-weight: 700;
    justify-content: space-around;
    ${Icon} {
        background-color: ${(props) => props.theme.colors.bg1};
    }
    transition: all ${(props) => props.theme.transitions.interactive};
    &:hover,
    &:focus {
        background-color: ${(props) => props.theme.colors.textMain};
    }
    &:active {
        background-color: ${(props) => darken(0.1, props.theme.colors.textMain)};
    }
`
export const InfoIcon = styled(Icon)`
    height: 34px;
    width: 34px;
    cursor: default;
    margin-bottom: 40px;
    background-color: ${(props) => props.theme.colors.textHeadings};
`

export const SwordsIcon = styled(InfoIcon)`
    mask-image: url(${Swords.src});
`

export const LockIcon = styled(InfoIcon)`
    mask-image: url(${Lock.src});
`

export const TextIcon = styled(InfoIcon)`
    mask-image: url(${Text.src});
`

export const ThemeIcon = styled(InfoIcon)`
    mask-image: url(${Theme.src});
`
export const PencilIcon = styled(InfoIcon)`
    mask-image: url(${Pencil.src});
`

export const MindIcon = styled(InfoIcon)`
    mask-image: url(${Mind.src});
`

export const SlidersIcon = styled(InfoIcon)`
    mask-image: url(${Sliders.src});
`

export const ArrowCircleIcon = styled(InfoIcon)`
    mask-image: url(${ArrowCircle.src});
`

export const WorldIcon = styled(InfoIcon)`
    mask-image: url(${World.src});
`

export const SmallArrowIcon = styled(Icon)`
    mask-image: url(${SmallArrow.src});
    height: 0.8em;
    margin-left: 10px;
    background-color: ${(props) => props.theme.colors.bg1};
`

export const AdventureSample = styled.div<{ nodeTypes: SchemaNodes }>`
    > :first-child {
        margin-bottom: 15px;
    }
    textarea,
    button,
    p {
        cursor: default;
    }
    max-width: 670px;
    font-size: 0.6rem;
    flex: 1 1 auto;

    .ProseMirror {
        height: auto;
        .adventureStory {
            background: ${(props) => props.theme.colors.bg2};
        }
        .adventureInput {
        }
        textarea::placeholder {
        }
    }

    @media (max-width: ${mobileSize}) {
        margin-bottom: 50px;
        width: 670px;

        transform: scale(0.7);
        transform-origin: bottom left;

        margin-right: -200px;
    }

    @media (max-width: ${smallMobileSize}) {
        transform: scale(1);

        margin-top: 0;
    }

    @media (max-width: ${smallerMobileSize}) {
        transform: scale(0.7);
        transform-origin: bottom left;
        margin-top: -15%;
        margin-right: 0;
    }

    // Below is a copy of the global adventure style
    .ProseMirror {
        border: none;
        padding: 0 !important;
        display: flex;
        flex-direction: column;
        justify-content: flex-end;

        > p:not(.empty-node) {
            & {
                border: 1px solid ${(props) => props.theme.colors.bg3};
                border-bottom: none;
                padding: 15px !important;
                padding-left: 65px !important;
                position: relative;
                &:before {
                    content: '';
                    background: ${(props) => props.theme.colors.bg1};
                    width: 50px;
                    height: 100%;
                    position: absolute;
                    left: 0;
                    top: 0;
                    border-right: 1px solid ${(props) => props.theme.colors.bg3};
                }
                &:after {
                    content: '';
                    background-position: center;
                    background-size: contain;
                    background-repeat: no-repeat;
                    mask-repeat: no-repeat;
                    mask-size: contain;
                    mask-position: center;
                    background-color: ${(props) => props.theme.colors.textMain};
                    mask-image: url(${Book.src});
                    width: 20px;
                    height: 20px;
                    position: absolute;
                    left: 15px;
                    top: 15px;
                    opacity: 0.9;
                }
            }
            &.adventureInput,
            &.adventureStoryEnd {
                border-bottom: 1px solid ${(props) => props.theme.colors.bg3};
            }
            &.adventureStory + p.adventureStory {
                padding-top: 0px !important;
                border-top: none;
                &:after {
                    mask-image: none;
                    background-color: unset;
                }
            }
            &.adventureStoryEnd {
                margin-bottom: 15px;
                border-bottom: 1px solid ${(props) => props.theme.colors.bg3};
            }
            &.adventureInput {
                background: ${(props) => props.theme.colors.bg1};
                margin-left: 20px;
                margin-bottom: 15px;
                &:before {
                }
            }
            &.adventureStoryEnd:last-child {
                margin-bottom: 0;
            }
            &.adventureInput {
                &:after {
                    background-color: ${(props) => props.theme.colors.textMain} !important;
                }
            }

            ${(props) =>
                props.nodeTypes.adventure.map((node) => {
                    return `
            &.adventureInput.${node.name} {
                &:after {
                    mask-image: url(${node.icon});
                }
            }`
                })}
        }
    }
`

export const Testimonials = styled(Container)`
    display: flex;
    max-width: 100%;
    flex-direction: column;
    align-items: center;
    width: auto;
    > div {
        display: flex;
        flex-direction: row;

        > div {
            :nth-child(1) {
                max-width: min(600px, 100vw);
                margin-right: 15px;
                > div {
                    margin-bottom: 30px;
                }
                @media (max-width: ${mobileSize}) {
                    flex: 1 1 0;
                }

                @media (max-width: ${smallMobileSize}) {
                    display: none;
                }
            }
            :nth-child(2) {
                max-width: 530px;
                margin-left: 15px;
                > div {
                    margin-bottom: 30px;
                }
                @media (max-width: ${mobileSize}) {
                    flex: 1 1 0;
                }

                @media (max-width: ${smallMobileSize}) {
                    width: auto;
                    margin-left: 0px;
                }
            }
            display: flex;
            flex-direction: column;
        }
    }

    @media (max-width: ${smallMobileSize}) {
        align-items: flex-start;
    }
`

export const TestimonialLead = styled.div`
    font-size: 1.125rem;
    font-weight: 600;
    color: ${(props) => props.theme.colors.textHeadings};
    @media (max-width: ${smallMobileSize}) {
        font-size: 1rem;
    }
`

export const TestimonialTitle = styled.h2`
    font-size: 2.625rem;
    font-weight: 700;
    font-family: ${(props) => props.theme.fonts.headings};
    color: ${(props) => props.theme.colors.textMain};
    @media (max-width: ${smallMobileSize}) {
        font-size: 1.5rem;
        max-width: 234px;
        line-height: 36px;
    }
`

export const TestimonialBox = styled.div`
    border: 1px solid ${(props) => props.theme.colors.bg3};
    background: ${(props) => props.theme.colors.bg2};

    line-height: 32px;
    font-size: 1.375rem;
    padding: 30px 30px 20px 30px;
    span {
        font-weight: 400;
    }
    > div:last-child {
        font-size: 1.125rem;
        display: flex;
        align-items: center;
        margin-top: 15px;

        img {
            margin-right: 15px;
        }
        color: ${(props) => transparentize(0.3, props.theme.colors.textMain)};
    }

    @media (max-width: ${mobileSize}) {
        font-size: 1rem;
    }
`

export const TestimonialEmphasis = styled.span`
    font-weight: 700 !important;
    color: ${(props) => props.theme.colors.textHeadings};
`

export const TierContainer = styled.div`
    position: relative;
    padding-top: 142px;
    width: 100%;
    display: flex;
    align-items: center;
    flex-direction: column;

    @media (max-width: ${mobileSize}) {
        padding-top: 105px;
    }
    > :last-child {
        display: grid;
        grid-template-columns: 275px 275px 275px 275px;
        gap: 20px;
        align-items: flex-end;

        @media (max-width: ${mobileSize}) {
            grid-template-columns: 315px 315px;
            > div {
                margin-right: 0px !important;
            }

            > :nth-child(3) {
                margin-top: 40px;
            }

            > :nth-child(4) {
                margin-top: 40px;
            }
        }
        @media (max-width: ${smallMobileSize}) {
            grid-template-columns: 315px;
            gap: 0px;
        }
    }
`

export const TierText = styled.div`
    display: flex;
    flex-direction: column;
    position: absolute;
    top: 0;
    left: 10px;
    > span:first-child {
        font-weight: 600;
        font-size: 1.125rem;
        color: ${(props) => props.theme.colors.textHeadings};

        line-height: 50px;
    }
    > span:last-child {
        font-weight: 700;
        font-size: 2.625rem;
        font-family: ${(props) => props.theme.fonts.headings};
        max-width: 353px;

        line-height: 50px;
    }

    @media (max-width: ${mobileSize}) {
        align-items: center;
        position: unset;
        > span:last-child {
            max-width: unset;
        }
        margin-bottom: 20px;
    }

    @media (max-width: ${smallMobileSize}) {
        align-items: flex-start;
        width: 305px;
        > span:first-child {
            font-size: 1;
        }

        > span:last-child {
            font-size: 1.5rem;
            max-width: 200px;
            line-height: 36px;
        }
    }
`

export const TierGetStarted = styled.a<{ color: string }>`
    cursor: pointer;
    position: absolute;
    width: calc(100% - 50px);
    bottom: -24px;
    left: calc(25px);
    border: 2px solid ${(props) => props.color};
    color: ${(props) => props.color};
    background-color: ${(props) => props.theme.colors.bg1};
    padding: 14px 20px;
    display: flex;
    flex-direction: row;
    justify-content: space-between;
    align-items: center;

    > div {
        background-color: ${(props) => props.color};
    }

    :hover {
        color: ${(props) => props.theme.colors.bg1};
        background-color: ${(props) => props.color};
        > div {
            background-color: ${(props) => props.theme.colors.bg1};
        }
    }
    font-weight: 700;
`

export const FakeModal = styled.div`
    background-color: ${(props) => props.theme.colors.bg2};
    border: 1px solid ${(props) => props.theme.colors.bg3};
    border-radius: 2px;
    display: flex;
    flex-direction: column;

    width: 850px;
    height: 600px;
    padding: 30px;

    > div {
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    padding-bottom: 0px;
    overflow: hidden;
    margin-right: -100px;
    @media (max-width: ${mobileSize}) {
        transform: scale(0.6);
        transform-origin: bottom left;

        margin-top: -37%;
        margin-right: -300px;
    }

    @media (max-width: ${smallMobileSize}) {
        transform: scale(0.8);

        margin-top: -15%;
    }

    @media (max-width: ${smallerMobileSize}) {
        transform: scale(0.6);
        transform-origin: bottom left;
        margin-top: -50%;
        margin-right: 0;
    }
`

export const FakeModalTitle = styled.div`
    font-size: 1.4rem;
    font-family: ${(props) => props.theme.fonts.headings};
    font-weight: 700;
`

export const ModalClose = styled(CrossIcon)`
    height: 14px;
    width: 14px;
`

export const ModuleCategory = styled.div`
    margin-top: 20px;
    > div {
        > :first-child {
            font-size: 1rem;
            font-family: ${(props) => props.theme.fonts.headings};
            font-weight: 600;
        }
        > :last-child {
            font-size: 0.875rem;
            font-weight: 600;
        }
    }
    border-bottom: 1px solid ${(props) => props.theme.colors.bg3};
    padding-bottom: 10px;
    margin-bottom: 10px;
`

export const Module = styled.div`
    display: flex;
    flex-direction: column;
    border: 1px solid ${(props) => props.theme.colors.bg3};

    width: 250px;
    height: 206px;
    img {
        height: 110px;
        width: 100%;
        object-fit: cover;
    }
    > div {
        padding: 15px;
        > :first-child {
            font-size: 1rem;
            font-family: ${(props) => props.theme.fonts.headings};
            color: ${(props) => props.theme.colors.textHeadings};
            font-weight: 600;
        }
        > :last-child {
            font-size: 0.875rem;
            font-weight: 600;
        }
    }
`

export const FakeLorebook = styled.div`
    background-color: ${(props) => props.theme.colors.bg2};
    border: 1px solid ${(props) => props.theme.colors.bg3};
    border-radius: 2px;
    display: flex;
    flex-direction: row;
    width: 850px;
    height: 548px;
    font-size: 0.75rem;
    font-weight: 600;
    margin-left: -100px;

    @media (max-width: ${mobileSize}) {
        transform: scale(0.6);
        transform-origin: bottom left;
        margin-top: -23%;
        margin-left: 0px;
        margin-right: -300px;
        margin-bottom: 50px;
    }

    @media (max-width: ${smallMobileSize}) {
        margin-right: 0;
        transform: scale(0.8);
        margin-top: -10%;
    }

    @media (max-width: ${smallerMobileSize}) {
        transform: scale(0.6);
        transform-origin: bottom left;
        margin-top: -23%;

        margin-right: 0;
    }
`

export const FakeSidebar = styled.div`
    width: 242px;
    background-color: ${(props) => props.theme.colors.bg1};
    border-right: 2px solid ${(props) => props.theme.colors.bg3};
    display: flex;
    flex-shrink: 0;
    flex-direction: column;
    justify-content: space-between;
    > :first-child {
        border-bottom: 2px solid ${(props) => props.theme.colors.bg3};
    }
`

export const FakeLoreHeader = styled.div`
    padding: 20px 11px 15px 15px;
    font-weight: 600;
    display: flex;
    flex-direction: row;
    justify-content: space-between;
    align-items: center;
    font-size: 0.875rem;
    > :first-child {
        > :first-child {
            height: 10px;
            width: 11px;
            margin-right: 10px;
        }
    }
    > :last-child {
        > div {
            width: 14px;
        }
        > :first-child {
            margin-right: 16px;
        }
    }
    > div {
        display: flex;
        flex-direction: row;
        align-items: center;
    }
`

export const FakeSearch = styled.div`
    display: flex;
    justify-content: space-between;
    align-items: center;
    background-color: ${(props) => props.theme.colors.bg0};
    font-weight: 600;
    padding: 11px 15px;
    > span {
        opacity: 0.5;
    }
    > div {
        height: 13px;
        width: 13px;
    }
`

export const FakeCategory = styled.div`
    display: flex;
    padding: 8px 8px 8px 15px;
    align-items: center;
    justify-content: space-between;
    border-top: 2px solid ${(props) => props.theme.colors.bg3};

    > div {
        display: flex;
        align-items: center;

        > :first-child {
            margin-right: 8px;
        }
    }
    > div > div {
        width: 10px;
    }
`

export const FakeEntry = styled.div`
    padding: 12px 12px 12px 25px;
    > :first-child {
        font-size: 0.813rem;
        opacity: 0.7;
    }
    > :last-child {
        opacity: 0.7;
    }
`

export const FakeEntrySelected = styled(FakeEntry)`
    background-color: ${(props) => props.theme.colors.bg2};
    > :first-child {
        opacity: 1;
        font-family: ${(props) => props.theme.fonts.headings};
        color: ${(props) => props.theme.colors.textHeadings};
    }
`

export const FakeAddNew = styled.div`
    padding: 10px;
    > :first-child {
        background-color: ${(props) => props.theme.colors.bg2};
        margin-bottom: 10px;
    }
    > :last-child {
        border: 1px solid ${(props) => props.theme.colors.bg3};
    }
    > div {
        display: flex;
        align-items: center;
        padding: 14px 20px 12px 20px;
        font-family: ${(props) => props.theme.fonts.headings};

        > :first-child {
            margin-right: 10px;
            height: 10px;
            width: 10px;
        }
    }
`

export const FakeLoreTitle = styled(FlexRow)`
    display: flex;
    flex-direction: row;
    align-items: center;
    width: 100%;
    padding: 30px 30px 12px 15px;
    > :first-child {
        flex-grow: 0;
        font-size: 1.375rem;
        padding: 5px 15px 5px 15px;
        border: 2px solid ${(props) => props.theme.colors.bg3};
        color: ${(props) => props.theme.colors.textHeadings};
        font-family: ${(props) => props.theme.fonts.headings};
    }
`

export const FakeRight = styled.div`
    flex-grow: 1;
`

export const FakeTextEdit = styled.div`
    > :first-child {
        font-size: 0.875rem;
        color: ${(props) => props.theme.colors.textHeadings};
        margin-bottom: 7px;
        margin-left: 30px;
    }
    > :last-child {
        padding: 10px 30px 68px 16px;
        border: 2px solid ${(props) => props.theme.colors.bg3};
        margin: 0 30px 17px 15px;
        line-height: 17px;
    }
`

export const FakeKeyArea = styled.div`
    font-size: 0.875rem;
    margin: 0px 30px;
    > div:nth-child(1) {
        color: ${(props) => props.theme.colors.textHeadings};
    }
    > div:nth-child(2) {
        opacity: 0.7;
        margin: 6px 0px;
    }
`

export const FakeKeyEntry = styled.div`
    border: 1px solid ${(props) => props.theme.colors.bg3};
    > :first-child {
        background-color: ${(props) => props.theme.colors.bg0};
        padding: 15px 15px;
        border-bottom: 1px solid ${(props) => props.theme.colors.bg3};
        > span {
            opacity: 0.5;
        }
    }
`

export const FakeKeyBox = styled.div`
    > :first-child {
        display: flex;
        justify-content: space-between;
        margin: 15px;
        > :last-child {
            opacity: 0.7;
        }
    }
    > :last-child {
        font-size: 0.75rem;
        display: flex;
        padding: 0px 15px 73px 15px;
        > div {
            padding: 6px 10px;
            background-color: ${(props) => props.theme.colors.bg3};
            margin-right: 10px;
        }
    }
`

export const HeaderEasel = styled(EaselIcon)`
    background-color: ${(props) => props.theme.colors.textHeadings};
`
