""" from https://github.com/keithito/tacotron """

'''
Cleaners are transformations that run over the input text at both training and eval time.

Cleaners can be selected by passing a comma-delimited list of cleaner names as the "cleaners"
hyperparameter. Some cleaners are English-specific. You'll typically want to use:
  1. "english_cleaners" for English text
  2. "transliteration_cleaners" for non-English text that can be transliterated to ASCII using
     the Unidecode library (https://pypi.python.org/pypi/Unidecode)
  3. "basic_cleaners" if you do not want to transliterate (in this case, you should also update
     the symbols in symbols.py to match your data).
'''

import re
from unidecode import unidecode
from phonemizer import phonemize
from phonemizer.backend import EspeakBackend

mode = "phonemize"

if mode == "phonemize":
    backend = EspeakBackend('en-us', preserve_punctuation=True, with_stress=True)
if mode == "ARPA":
    from arpa_to_ipa import *

# Regular expression matching whitespace:
_whitespace_re = re.compile(r'\s+')

# List of (regular expression, replacement) pairs for abbreviations:
_abbreviations = [(re.compile('\\b%s\\.' % x[0], re.IGNORECASE), x[1]) for x in [
  ('mrs', 'misess'),
  ('mr', 'mister'),
  ('dr', 'doctor'),
  ('st', 'saint'),
  ('co', 'company'),
  ('jr', 'junior'),
  ('maj', 'major'),
  ('gen', 'general'),
  ('drs', 'doctors'),
  ('rev', 'reverend'),
  ('lt', 'lieutenant'),
  ('hon', 'honorable'),
  ('sgt', 'sergeant'),
  ('capt', 'captain'),
  ('esq', 'esquire'),
  ('ltd', 'limited'),
  ('col', 'colonel'),
  ('ft', 'fort'),
]]


def expand_abbreviations(text):
  for regex, replacement in _abbreviations:
    text = re.sub(regex, replacement, text)
  return text


def expand_numbers(text):
  return normalize_numbers(text)


def lowercase(text):
  return text.lower()


def collapse_whitespace(text):
  return re.sub(_whitespace_re, ' ', text)


def convert_to_ascii(text):
  return unidecode(text)


def basic_cleaners(text):
  '''Basic pipeline that lowercases and collapses whitespace without transliteration.'''
  text = lowercase(text)
  text = collapse_whitespace(text)
  return text


def transliteration_cleaners(text):
  '''Pipeline for non-English text that transliterates to ASCII.'''
  text = convert_to_ascii(text)
  text = lowercase(text)
  text = collapse_whitespace(text)
  return text


def english_cleaners(text):
  '''Pipeline for English text, including abbreviation expansion.'''
  text = convert_to_ascii(text)
  text = lowercase(text)
  text = expand_abbreviations(text)
  phonemes = backend.phonemize(text, strip=True)
  phonemes = collapse_whitespace(phonemes)
  return phonemes

def english_cleaners2(text):
  '''Pipeline for English text, including abbreviation expansion. + punctuation + stress'''
  text = convert_to_ascii(text)
  text = lowercase(text)
  text = expand_abbreviations(text)
  if mode == "phonemize":
    if "{{ipa." in text:
        parts = re.split(r'(\{\{ipa..*?\}\})', text)
        phonemes = []
        for part in parts:
            if part == "":
                continue
            try:
                if part[0:6] == "{{ipa." and part[-2:] == "}}":
                    phonemes.append(part[6:-2])
                else:
                    part = backend.phonemize(part.strip(), strip=True)
                    part = re.sub(r"\b((?:ˌ?(?:eɪtʃ|ɛm)ˈ?ˌ?)+)\b", lambda t: re.sub(r"(?:ˌ?(?:eɪtʃ|ɛm)ˈ?ˌ?)", "m", t.group()), part)
                    phonemes.append(part)
            except:
                part = backend.phonemize(part.strip(), strip=True)
                part = re.sub(r"\b((?:ˌ?(?:eɪtʃ|ɛm)ˈ?ˌ?)+)\b", lambda t: re.sub(r"(?:ˌ?(?:eɪtʃ|ɛm)ˈ?ˌ?)", "m", t.group()), part)
                phonemes.append(part)
        phonemes = " ".join(phonemes)
    else:
        phonemes = backend.phonemize(text, strip=True)
        phonemes = re.sub(r"\b((?:ˌ?(?:eɪtʃ|ɛm)ˈ?ˌ?)+)\b", lambda t: re.sub(r"(?:ˌ?(?:eɪtʃ|ɛm)ˈ?ˌ?)", "m", t.group()), phonemes)
  elif mode == "ARPA":
    phonemes = IPA(text)# + " " + phonemes
    phonemes = re.sub(r"\b((?:ˌ?(?:eɪtʃ|ɛm)ˈ?ˌ?)+)\b", lambda t: re.sub(r"(?:ˌ?(?:eɪtʃ|ɛm)ˈ?ˌ?)", "m", t.group()), phonemes)
  phonemes = collapse_whitespace(phonemes)
  return phonemes
