﻿' Imported from the C# version and adjusted slightly
' The C# class PonyInstance is not copied over, instead I have inherited the VB class Pony_Form.
' This allows me to copy the existing functionality and extend it with Update and Draw methods so it can be linked up.

Imports System.Collections.Generic
Imports System.Drawing
Imports System.Drawing.Drawing2D
Imports System.Drawing.Text
Imports System.Linq
Imports System.Windows.Forms

''' <summary>
''' Transparent form that handles the display of pony graphics.
''' </summary>
Partial Public Class PonyGraphicsForm
    Inherits Form
#Region "Variables"
    ''' <summary>
    ''' Indicates if the frame times should be shown
    ''' </summary>
    ''' <remarks></remarks>
    Private ShowFrameTimes As Boolean = False

    ''' <summary>
    ''' The context that creates our graphics buffer.
    ''' </summary>
    Private bufferedGraphicsContext As New BufferedGraphicsContext()
    ''' <summary>
    ''' The graphics buffer so we can take advantage of double buffering to prevent flicker.
    ''' Drawing is done to this buffer, then the bitmap of the whole buffer is displayed smoothly.
    ''' </summary>
    Private bufferedGraphics As BufferedGraphics
    ''' <summary>
    ''' Defines a brush that matches the TransparencyKey of the form.
    ''' Painting with this brush therefore causes transparent regions to appear.
    ''' </summary>
    Private transparencyBrush As SolidBrush
    ''' <summary>
    ''' Represents the area that becomes invalidated when updating.
    ''' Redrawing this region will be faster than redrawing the whole screen.
    ''' </summary>
    Private invalidRegion As New Region()

    ''' <summary>
    ''' The manager handling the images for our pony templates.
    ''' </summary>
    Private imageManager As ImageManager
    ''' <summary>
    ''' Indicates if a pony is being dragged.
    ''' </summary>
    Private draggingPony As Boolean = False
    ''' <summary>
    ''' If draggingPony is ture, indicates the index of which instance in ponies is being dragged.
    ''' </summary>
    Private draggedPony As Pony
    Private draggedEffect As Pony.Behavior.effect
    Private menuPony As Pony = Nothing
    Private menuHouse As Pony.House = Nothing
    Private menuHouseEffect As Pony.Behavior.effect = Nothing

    ' ''' <summary>
    ' ''' Indicates if the mouse is currently over the form.
    ' ''' </summary>
    'Private MouseOverForm As Boolean
    ''' <summary>
    ''' Contains the location of the cursors current position.
    ''' </summary>
    Private cursorLocation As Point

    Friend Screen As Screen

    ' Resources for drawing text to the screen.
    Private fontFamily As FontFamily = fontFamily.GenericMonospace
    Private textFont As Font
    Private blackBrush As New SolidBrush(Color.Black)
    Private whiteBrush As New SolidBrush(Color.White)

    Friend is_closing As Boolean = False
    Friend Screen_to_Form_Translation As New Point

    Dim Drawing_Corner As Point = New Point
    Dim Drawing_Corner_Previous As Point = New Point
    Dim sleeping_before_drag As Boolean = False

    Dim Drawing_Size As Size = New Size

    Dim graphics_is_fullscreen As Boolean = True

    Dim monitor_number As Integer

    Friend Cleared As Boolean = False
    Dim dirty As Boolean = False

    Dim Initializing As Boolean = True

    Friend Original_Size As Size = New Size()

    'list of currently displayed images
    Friend Displayed_Ponies As New List(Of Pony)

    Friend Displayed_Houses As New List(Of Pony.Behavior.effect)

#End Region

    ''' <summary>
    ''' Creates a new form to handle the drawing of pony graphics. See the PonyGraphicsForm.cs file for explanation.
    ''' </summary>
    Friend Sub New(ByVal manager As ImageManager, monitor As Screen, number As Integer)
        ' This is code generated by the form designer. We should use the designer if we need to change something.
        InitializeComponent()

        ' Make the form as big as the screen.
        Size = New Size(monitor.Bounds.Size.Width, monitor.Bounds.Size.Height)
        Location = monitor.Bounds.Location

        Original_Size = Size

        monitor_number = number

        Screen_to_Form_Translation = New Point(monitor.Bounds.X, monitor.Bounds.Y)

        ' Tell the OS we'll be handling the drawing.
        ' This causes the OnPaint and OnPaintBackground events to be raised for us.
        SetStyle(ControlStyles.UserPaint, True)
        ' Prevents the OS from raising the OnPaintBackground event on us.
        SetStyle(ControlStyles.AllPaintingInWmPaint, True)
        SetStyle(ControlStyles.ResizeRedraw, False)

        'If the OS is older than Vista/2008, then we need to use an alternate method of getting transparancy.
        If My.Forms.Main.OS_Is_Old Then
            SetStyle(ControlStyles.SupportsTransparentBackColor, True)
        End If

        ' Windows Forms provides easy ways to double buffer forms.
        ' Double buffering prevents flicker by drawing to a back buffer, then displaying it all at once.
        ' It can be quickly set with DoubleBuffered = true or SetStyle(ControlStyles.OptimizedDoubleBuffer, true);
        ' However this method creates a lot of garbage, so it's better to handle it manually.
        ' We create our own buffer surface, draw to that and render when done.
        bufferedGraphicsContext.MaximumBuffer = New Size(Width + 1, Height + 1)
        bufferedGraphics = bufferedGraphicsContext.Allocate(CreateGraphics(), New Rectangle(New Point(), Size))
        ' Set the quality of drawing.
        ' As a result of having to use a transparency key, we can't use anti-aliasing functions.
        ' If we did, semi-transparent pixels would be blended with the form background color.
        bufferedGraphics.Graphics.SmoothingMode = SmoothingMode.None
        bufferedGraphics.Graphics.TextRenderingHint = TextRenderingHint.SingleBitPerPixelGridFit

        ' Forms don't truly support a transparent background, however a transparency key can be used
        ' to tell it to treat a certain color as transparent. So we'll try and use an uncommon color.
        ' This also has the bonus effect of making interaction fall through the transparent areas.
        ' This means interaction with the desktop is possible.
        ' On XP this doesn't work unless the background color is transparent black (0 alpha).
        ' In ImageMnaagement.vb, we change all the blacks to slight-grays as well.

        If My.Forms.Main.OS_Is_Old Then
            BackColor = Color.FromArgb(0, 0, 0, 0)
        Else
            BackColor = Color.FromArgb(255, 192, 255)
        End If

        TransparencyKey = BackColor
        transparencyBrush = New SolidBrush(TransparencyKey)

        textFont = New Font(fontFamily, 18, GraphicsUnit.Pixel)

        Me.imageManager = manager

        ' Do a garbage collection so we can free garbage caused by startup.
        GC.Collect()
        GC.WaitForPendingFinalizers()

        ' Show the form and set the timers and ponies running.
        Me.Show()

        With My.Forms.Main
            .formStartTime = DateTime.Now
            .loopStartTime = .formStartTime
            .loopEndTime = .formStartTime
            .totalElaspedTime = TimeSpan.Zero
            cursorLocation = Cursor.Position
        End With
    End Sub

    'Clears the scren of all graphics.
    Friend Sub ClearScreen()

        graphics_is_fullscreen = False
        Resize_Graphics(0, True)

    End Sub

    'This sub resizes the bufferedgraphics.graphics object to only contain those ponies and effects that are on this monitor.
    'This improves performance as the system does not need to redraw the entire screen.
    Friend Sub Resize_Graphics(threshold As Integer, Optional make_fullscreen As Boolean = False)

        'don't resize when in the editor.
        'My.Forms.Main.Preview_Mode OrElse
        If make_fullscreen OrElse Not IsNothing(My.Forms.Main.current_game) Then
            If graphics_is_fullscreen = False Then
                bufferedGraphics.Dispose()
                bufferedGraphics = bufferedGraphicsContext.Allocate(CreateGraphics(), New Rectangle(New Point(), Size))
                bufferedGraphics.Graphics.SmoothingMode = SmoothingMode.None
                bufferedGraphics.Graphics.TextRenderingHint = TextRenderingHint.SingleBitPerPixelGridFit

                invalidRegion.MakeEmpty()
                bufferedGraphics.Graphics.FillRegion(transparencyBrush, invalidRegion)
                bufferedGraphics.Graphics.Clear(TransparencyKey)

                bufferedGraphics.Render()

                Drawing_Corner = New Point()

                graphics_is_fullscreen = True
            End If
            Exit Sub
        End If


        graphics_is_fullscreen = False

        Dim left As Nullable(Of Integer) = Nothing
        Dim right As Nullable(Of Integer) = Nothing
        Dim top As Nullable(Of Integer) = Nothing
        Dim bottom As Nullable(Of Integer) = Nothing

        For Each Pony As Pony In My.Forms.Main.Active_Ponies

            If Pony.IsPonyOnScreen(Pony.location, Screen) = False Then Continue For

            If IsNothing(Pony.current_behavior) Then Continue For

            Dim pre_infinite = False
            Dim post_infinite = False

            Dim prebounds As System.Drawing.RectangleF = Pony.PreUpdate_DrawAreas(monitor_number).GetBounds(bufferedGraphics.Graphics)
            Dim postbounds As System.Drawing.RectangleF = Pony.PostUpdate_DrawAreas(monitor_number).GetBounds(bufferedGraphics.Graphics)

            If Pony.PreUpdate_DrawAreas(monitor_number).IsInfinite(bufferedGraphics.Graphics) Then
                pre_infinite = True
            End If
            If Pony.PostUpdate_DrawAreas(monitor_number).IsInfinite(bufferedGraphics.Graphics) Then
                post_infinite = True
            End If

            If pre_infinite = False Then
                If Not left.HasValue OrElse prebounds.Left < left Then
                    left = prebounds.Left
                End If
                If Not right.HasValue OrElse prebounds.Right > right Then
                    right = prebounds.Right
                End If
                If Not top.HasValue OrElse prebounds.Top < top Then
                    top = prebounds.Top
                End If
                If Not bottom.HasValue OrElse prebounds.Bottom > bottom Then
                    bottom = prebounds.Bottom
                End If
            End If

            If post_infinite = False Then
                If Not left.HasValue OrElse postbounds.Left < left Then
                    left = postbounds.Left
                End If
                If Not right.HasValue OrElse postbounds.Right > right Then
                    right = postbounds.Right
                End If
                If Not top.HasValue OrElse postbounds.Top < top Then
                    top = postbounds.Top
                End If
                If Not bottom.HasValue OrElse postbounds.Bottom > bottom Then
                    bottom = postbounds.Bottom
                End If
            End If


            
        Next

        For Each effect In My.Forms.Main.Active_Effects

            If IsNothing(effect.PreUpdate_DrawAreas(monitor_number)) OrElse IsNothing(effect.PostUpdate_DrawAreas(monitor_number)) Then Continue For

            Dim prebounds As System.Drawing.RectangleF = effect.PreUpdate_DrawAreas(monitor_number).GetBounds(bufferedGraphics.Graphics)
            Dim postbounds As System.Drawing.RectangleF = effect.PostUpdate_DrawAreas(monitor_number).GetBounds(bufferedGraphics.Graphics)

            Dim pre_infinite = False
            Dim post_infinite = False

            If effect.PreUpdate_DrawAreas(monitor_number).IsInfinite(bufferedGraphics.Graphics) Then
                pre_infinite = True
            End If
            If effect.PostUpdate_DrawAreas(monitor_number).IsInfinite(bufferedGraphics.Graphics) Then
                post_infinite = True
            End If

            If pre_infinite = False Then
                If Not left.HasValue OrElse prebounds.Left < left Then
                    left = prebounds.Left
                End If
                If Not right.HasValue OrElse prebounds.Right > right Then
                    right = prebounds.Right
                End If
                If Not top.HasValue OrElse prebounds.Top < top Then
                    top = prebounds.Top
                End If
                If Not bottom.HasValue OrElse prebounds.Bottom > bottom Then
                    bottom = prebounds.Bottom
                End If
            End If

            If post_infinite = False Then
                If Not left.HasValue OrElse postbounds.Left < left Then
                    left = postbounds.Left
                End If
                If Not right.HasValue OrElse postbounds.Right > right Then
                    right = postbounds.Right
                End If
                If Not top.HasValue OrElse postbounds.Top < top Then
                    top = postbounds.Top
                End If
                If Not bottom.HasValue OrElse postbounds.Bottom > bottom Then
                    bottom = postbounds.Bottom
                End If
            End If
        Next

        For Each effect In My.Forms.Main.Dead_Effects

            Dim pre_infinite = False
            Dim post_infinite = False

            Dim pre_area = effect.PreUpdate_DrawAreas(monitor_number)
            Dim post_area = effect.PostUpdate_DrawAreas(monitor_number)

            Dim prebounds As System.Drawing.RectangleF = Nothing
            Dim postbounds As System.Drawing.RectangleF = Nothing

            If IsNothing(pre_area) Then
                pre_infinite = True
            Else
                prebounds = effect.PreUpdate_DrawAreas(monitor_number).GetBounds(bufferedGraphics.Graphics)
                If effect.PreUpdate_DrawAreas(monitor_number).IsInfinite(bufferedGraphics.Graphics) Then
                    pre_infinite = True
                End If
            End If
            If IsNothing(post_area) Then
                post_infinite = True
            Else
                postbounds = effect.PostUpdate_DrawAreas(monitor_number).GetBounds(bufferedGraphics.Graphics)
                If effect.PostUpdate_DrawAreas(monitor_number).IsInfinite(bufferedGraphics.Graphics) Then
                    post_infinite = True
                End If
            End If

            If pre_infinite = False Then
                If Not left.HasValue OrElse prebounds.Left < left Then
                    left = prebounds.Left
                End If
                If Not right.HasValue OrElse prebounds.Right > right Then
                    right = prebounds.Right
                End If
                If Not top.HasValue OrElse prebounds.Top < top Then
                    top = prebounds.Top
                End If
                If Not bottom.HasValue OrElse prebounds.Bottom > bottom Then
                    bottom = prebounds.Bottom
                End If
            End If

            If post_infinite = False Then
                If Not left.HasValue OrElse postbounds.Left < left Then
                    left = postbounds.Left
                End If
                If Not right.HasValue OrElse postbounds.Right > right Then
                    right = postbounds.Right
                End If
                If Not top.HasValue OrElse postbounds.Top < top Then
                    top = postbounds.Top
                End If
                If Not bottom.HasValue OrElse postbounds.Bottom > bottom Then
                    bottom = postbounds.Bottom
                End If
            End If
        Next

        My.Forms.Main.Dead_Effects.Clear()

        If Not left.HasValue OrElse Not right.HasValue OrElse Not top.HasValue OrElse Not bottom.HasValue Then
            Exit Sub
        End If


        Drawing_Corner_Previous = Drawing_Corner
       
        Drawing_Corner = New Point(left, top)
        Drawing_Size = New Size(right - left, bottom - top)

        bufferedGraphics.Dispose()
        bufferedGraphics = bufferedGraphicsContext.Allocate(CreateGraphics(), New Rectangle(Drawing_Corner, Drawing_Size))
        bufferedGraphics.Graphics.SmoothingMode = SmoothingMode.None
        bufferedGraphics.Graphics.TextRenderingHint = TextRenderingHint.SingleBitPerPixelGridFit

    End Sub


    ''' <summary>
    ''' Occurs when UpdateTimer ticks.
    ''' This will be used as our main loop, to update and draw the ponies.
    ''' If the UI thread is busy, the tick is ignored which means we'll run slower.
    ''' (This is better than queueing it for later, since calls would stack up)
    ''' </summary>
    Friend Sub Update_Graphics()
        ' Keep track of the time we started the tick.
        Dim thisLoopStartTime As DateTime = DateTime.Now

        Displayed_Ponies.Clear()
        Displayed_Houses.Clear()

        ' Reset the clipping area.
        invalidRegion.MakeEmpty()

        'If we have less ponies than this, do region based invalidation to save CPU.
        'If we have more, it's cheaper to just redraw everything than to calculate the region.
        Dim threshold As Integer = 20
        If Initializing Then ' OrElse My.Forms.Main.Preview_Mode Then
            threshold = 0
            Initializing = False
        End If

        'Redraw the entire screen when playing a game to capture the
        'scoreboard and goals.
        If Not IsNothing(My.Forms.Main.current_game) Then
            threshold = 0
        End If


        If My.Forms.Main.Active_Ponies.Count < threshold Then
            For Each Pony In My.Forms.Main.Active_Ponies
                ' Add the pre-update region to our list, we'll need to clear it.
                invalidRegion.Union(Pony.PreUpdate_DrawAreas(monitor_number))
                ' Add the post-update region to our list, we'll need to draw it.
                invalidRegion.Union(Pony.PostUpdate_DrawAreas(monitor_number))

                For Each effect In My.Forms.Main.Active_Effects
                    If Not IsNothing(effect.PreUpdate_DrawAreas(monitor_number)) Then
                        invalidRegion.Union(effect.PreUpdate_DrawAreas(monitor_number))
                    End If
                    If Not IsNothing(effect.PostUpdate_DrawAreas(monitor_number)) Then
                        invalidRegion.Union(effect.PostUpdate_DrawAreas(monitor_number))
                    End If
                Next


                For Each deadeffect In My.Forms.Main.Dead_Effects
                    If Not IsNothing(deadeffect.PreUpdate_DrawAreas(monitor_number)) Then
                        invalidRegion.Union(deadeffect.PreUpdate_DrawAreas(monitor_number))
                    End If
                    If Not IsNothing(deadeffect.PostUpdate_DrawAreas(monitor_number)) Then
                        invalidRegion.Union(deadeffect.PostUpdate_DrawAreas(monitor_number))
                    End If
                Next

            Next

            'Draw Houses
            For Each house As Pony.Behavior.effect In My.Forms.Main.Active_Effects

                If Pony.IsPonyInBox(house.translated_location, New Rectangle(Me.Location, Me.Size)) Then
                    house.PreUpdate_DrawAreas(monitor_number) = New Region(New Rectangle(Pony.Translate_Current_Location(house.location, Me), Pony.House.ImageScale(house.Rightanimatedimage.Size)))
                    house.translated_location = Pony.Translate_Current_Location(house.location, Me)
                    house.PostUpdate_DrawAreas(monitor_number) = New Region(New Rectangle(Pony.Translate_Current_Location(house.location, Me), Pony.House.ImageScale(house.Rightanimatedimage.Size)))
                    Displayed_Houses.Add(house)
                End If

            Next

        End If

        If ShowFrameTimes Then
            ' Add a region for the stats in the upper left.
            If My.Forms.Main.Active_Ponies.Count < threshold Then
                invalidRegion.Union(New Rectangle(0, 0, 300, 40))
            End If
        End If

        'Apply the clipping area.
        If My.Forms.Main.Active_Ponies.Count < threshold Then
            invalidRegion.Intersect(New Rectangle(New Point, Size))
        Else
            invalidRegion.Union(New Rectangle(New Point, Size))
        End If


        'Normally you would set the clip before clearing.
        'However, doing that when having ponies on both monitors results in a huge slowdown for reasons I don't yet understand.
        'A middle-ground-workaround is to use the clip only for drawing, and then resize the graphics objects to only
        'be as large as the ponies you have on each screen.
        '(Using the clip when clearing and using resizing causes junk to fill the screen as it is not cleared properly.
        bufferedGraphics.Graphics.Clear(TransparencyKey)
        bufferedGraphics.Graphics.SetClip(invalidRegion, CombineMode.Replace)


        Dim HavePonies As Boolean = False

        'Draw Houses

        If Displayed_Houses.Count <> 0 Then HavePonies = True

        For Each house As Pony.Behavior.effect In Displayed_Houses
            house.Draw(bufferedGraphics.Graphics, My.Forms.Main.totalElaspedTime, My.Forms.Main.PonyScale)
        Next

        ' Draw all the ponies on screen.
        For Each pony As Pony In My.Forms.Main.Active_Ponies
            If pony.Draw(bufferedGraphics.Graphics, My.Forms.Main.totalElaspedTime, Me, monitor_number) = True Then
                HavePonies = True

                Displayed_Ponies.Add(pony)

                'Dim rects(1) As RectangleF
                'rects(0) = pony.PreUpdate_DrawAreas(monitor_number).GetBounds(bufferedGraphics.Graphics)
                'bufferedGraphics.Graphics.DrawRectangles(Pens.Red, rects)

            End If
        Next

        'we don't have any ponies to draw, just quit.
        If HavePonies = False Then
            'except that we'll want to run one more time to clear any junk that was left on the screen.
            If Cleared = True Then
                Exit Sub
            Else
                If dirty Then
                    Resize_Graphics(threshold, True)
                    invalidRegion.MakeEmpty()
                    bufferedGraphics.Graphics.Clear(TransparencyKey)
                    dirty = False
                Else
                    Exit Sub
                End If
            End If

        Else
            If Not IsNothing(My.Forms.Main.current_game) Then
                My.Forms.Main.current_game.ScoreBoard.Paint(bufferedGraphics.Graphics, Me)
                For Each goal As Game.Goal_Area In My.Forms.Main.current_game.Goals
                    goal.paint(bufferedGraphics.Graphics, Me)
                Next
            End If

        End If


        ' Work out how long this frame took and display it.
        'If ShowFrameTimes Then
        '    Dim frameTimes As String = CalculateFrameTime()
        '    bufferedGraphics.Graphics.DrawString(frameTimes, textFont, blackBrush, Drawing_Corner)
        '    bufferedGraphics.Graphics.DrawString(frameTimes, textFont, whiteBrush, Drawing_Corner.X + 3, Drawing_Corner.Y + 3)
        'End If

        ' Render our buffer to the screen.
        bufferedGraphics.Render()
        If HavePonies = True Then
            dirty = True
        End If

        Resize_Graphics(threshold)

        If HavePonies = True Then
            Cleared = False
        Else
            Cleared = True
        End If

    End Sub

    ''' <summary>
    ''' Work out how long the last update took.
    ''' Several recent values are stored for smooth results.
    ''' </summary>
    ''' <returns>Printable string containing the min/mean/max frame time values.</returns>
    Private Function CalculateFrameTime() As String
        ' Store the new frame time.
        With My.Forms.Main

            .frameTimesInMs(.frameTimesMarker) = CInt((.loopEndTime - .loopStartTime).TotalMilliseconds)
            .frameTimesMarker += 1
            If .frameTimesMarker >= .frameTimesInMs.Length Then
                .frameTimesMarker = 0
            End If
            ' Calculate the min.mean/max times in our buffer.
            Dim minFrameTime As Integer = .frameTimesInMs.Min()
            Dim avgFrameTime As Integer = .frameTimesInMs.Sum() / .frameTimesInMs.Length
            Dim maxFrameTime As Integer = .frameTimesInMs.Max()

            ' Format our results.
            Dim frameTimes As String = String.Format("time: {0}ms/{1}ms/{2}ms", minFrameTime, avgFrameTime, maxFrameTime)
            Console.WriteLine("Time: {0}ms/{1}ms/{2}ms  Collections: {3}/{4}/{5}", minFrameTime, avgFrameTime, maxFrameTime, GC.CollectionCount(0), GC.CollectionCount(1), _
             GC.CollectionCount(2))
            Return frameTimes

        End With
    End Function

    ''' <summary>
    ''' Occurs when a mouse button is pressed down initially.
    ''' Selects the nearest pony to be dragged by the mouse.
    ''' </summary>
    ''' <param name="sender">The object that raised the event.</param>
    ''' <param name="e">Data about the event.</param>
    Private Sub PonyGraphicsForm_MouseDown(ByVal sender As Object, ByVal e As MouseEventArgs) Handles Me.MouseDown

        If e.Button = MouseButtons.Left Then
            If My.Forms.Options.Pony_Dragging_Enabled.Checked = False Then Exit Sub

            Dim SelectedPony = find_closest_pony_to_click(e)

            If IsNothing(SelectedPony) Then

                Dim SelectedEffect = find_closest_effect_to_click(e)

                If IsNothing(SelectedEffect) Then Exit Sub

                DragEffect(SelectedEffect)

                Exit Sub

            End If


            SelectedPony.BeingDragged = True
            draggedPony = SelectedPony
            My.Forms.Main.dragging = True
            draggingPony = True
            If Main.MoveTimer.Enabled = True Then
                SelectedPony.BeingDragged = True
                If SelectedPony.sleeping = False Then
                    SelectedPony.should_be_sleeping = True
                    sleeping_before_drag = False
                Else
                    sleeping_before_drag = True
                End If
            End If
        End If

        If e.Button = Windows.Forms.MouseButtons.Right Then

            menuPony = find_closest_pony_to_click(e)
            If IsNothing(menuPony) Then

                Dim SelectedEffect = find_closest_effect_to_click(e)

                If IsNothing(SelectedEffect) Then Exit Sub

                If IsNothing(SelectedEffect.Owning_Pony) Then
                    DisplayHouseMenu(SelectedEffect)
                End If

                Exit Sub

            End If
        
            If My.Forms.Main.screen_saver_mode Then Main.Close()

            If e.Button = Windows.Forms.MouseButtons.Right Then

                My.Forms.Main.Right_Click_Menu_Active = True
                If menuPony.should_be_sleeping Then
                    RightClickMenu.Items("Sleep").Text = "Wake up/Resume"
                Else
                    RightClickMenu.Items("Sleep").Text = "Sleep/Pause"
                End If

                If My.Forms.Main.MoveTimer.Enabled = False Then
                    RightClickMenu.Items("Sleep_All").Text = "Wake up/Resume All"
                Else
                    RightClickMenu.Items("Sleep_All").Text = "Sleep/Pause All"
                End If

                RightClickMenu.Items("Quit_AllType").Text = "Quit all copies of " & menuPony.Name
                RightClickMenu.Items("Pony_Name").Text = menuPony.Name & "'s Menu"

                RightClickMenu.Show(menuPony.location)
            End If
        End If

    End Sub

    Friend Sub DisplayHouseMenu(Effect As Pony.Behavior.effect)

        For Each house In My.Forms.Main.Houses
            Dim done = False
            If house.Active Then
                For Each Instance In house.Instances
                    If ReferenceEquals(Effect, Instance) Then
                        menuHouse = house
                        menuHouseEffect = Effect
                        done = True
                        Exit For
                    End If
                Next
            End If
            If done Then Exit For
        Next

        If IsNothing(menuHouse) Then Exit Sub

        My.Forms.Main.Right_Click_Menu_Active = True

        HouseRightClickMenu.Items("HouseName").Text = menuHouse.name
        HouseRightClickMenu.Show(Effect.location)

    End Sub

    Friend Sub DragEffect(effect As Pony.Behavior.effect)

        effect.beingDragged = True
        draggingPony = True
        draggedEffect = effect
        My.Forms.Main.dragging = True

    End Sub

    Friend Function find_closest_pony_to_click(ByVal e As MouseEventArgs) As Pony

        'note that the ponies are already in the correct z-order because we sort them every loop in Main()
        'If we want to get the top pony we need to go in reverse order

        Dim i = Displayed_Ponies.Count - 1

        Do Until i < 0

            Dim pony As Pony = Displayed_Ponies(i)

            Dim ponyscale = pony.GetScale

            Dim formsize As Pony.RECT
            formsize.Top = Drawing_Corner.Y
            formsize.Left = Drawing_Corner.X
            formsize.Right = Drawing_Corner.X + Drawing_Size.Width
            formsize.Bottom = Drawing_Corner.Y + Drawing_Size.Height

            Dim point_on_bitmap As Point = New Point(e.X - pony.Translated_Location.X, e.Y - pony.Translated_Location.Y)

            If (point_on_bitmap.X >= 0 AndAlso point_on_bitmap.X <= pony.current_bitmap.Size.Width * ponyscale) AndAlso _
                (point_on_bitmap.Y >= 0 AndAlso point_on_bitmap.Y <= pony.current_bitmap.Size.Height * ponyscale) Then

                Dim pixel = Nothing

                'even though we did a check to get here, exceptions for checking points outside the image may still occur
                Try
                    pixel = pony.current_bitmap.GetPixel((e.X - pony.Translated_Location.X) / ponyscale, (e.Y - pony.Translated_Location.Y) / ponyscale)
                Catch ex As Exception
                    Return Nothing
                End Try

                If pixel.A <> 0 Then Return pony

            End If

            i -= 1
        Loop

        Return Nothing

    End Function

    Friend Function find_closest_effect_to_click(ByVal e As MouseEventArgs) As Pony.Behavior.effect

        For Each effect In My.Forms.Main.Active_Effects

            Dim point_on_bitmap As Point = New Point(e.X - effect.translated_location.X, e.Y - effect.translated_location.Y)

            If (point_on_bitmap.X >= 0 AndAlso point_on_bitmap.X <= effect.current_bitmap.Size.Width * My.Forms.Main.PonyScale) AndAlso _
                (point_on_bitmap.Y >= 0 AndAlso point_on_bitmap.Y <= effect.current_bitmap.Size.Height * My.Forms.Main.PonyScale) Then

                Dim pixel = Nothing

                Try
                    pixel = effect.current_bitmap.GetPixel((e.X - effect.translated_location.X) / My.Forms.Main.PonyScale, (e.Y - effect.translated_location.Y) / My.Forms.Main.PonyScale)
                Catch ex As Exception
                    Return Nothing
                End Try

                If pixel.A <> 0 Then Return effect

            End If

        Next

        Return Nothing

    End Function

    Friend Sub PonySpeak(Text As String, pony_location As Point)
        Me.Focus()
        Dim tt As New ToolTip()
        If Not Me.IsDisposed() Then
            tt.Show(Text, Me, pony_location.X, pony_location.Y, 2000)
        End If

    End Sub

    ''' <summary>
    ''' Occurs when the mouse moves.
    ''' If a pony is being dragged, we'll move it to the mouse position.
    ''' </summary>
    ''' <param name="sender">The object that raised the event.</param>
    ''' <param name="e">Data about the event.</param>
    Private Sub PonyGraphicsForm_MouseMove(ByVal sender As Object, ByVal e As MouseEventArgs) Handles Me.MouseMove
        cursorLocation = e.Location
        If draggingPony Then

            If Not IsNothing(draggedPony) Then
                draggedPony.SetLocationFromTranslated(e.Location, Me)
            End If
            If Not IsNothing(draggedEffect) Then

                Dim real_location = New Point(cursorLocation.X + Me.Screen_to_Form_Translation.X, _
                          cursorLocation.Y + Me.Screen_to_Form_Translation.Y)

                Dim current_center = draggedEffect.Center()

                Dim x_difference As Integer = real_location.X - current_center.X
                Dim y_difference As Integer = real_location.Y - current_center.Y

                draggedEffect.location = New Point(draggedEffect.location.X + x_difference, draggedEffect.location.Y + y_difference)
                draggedEffect.translated_location = Pony.Translate_Current_Location(draggedEffect.location, Me)

                'Console.WriteLine("----")
                'Console.WriteLine(e.Location)
                'Console.WriteLine(draggedEffect.location)
                'Console.WriteLine(draggedEffect.translated_location)


            End If
        End If
    End Sub

    ''' <summary>
    ''' Occurs when a mouse button is raised.
    ''' Any dragged pony will be dropped.
    ''' </summary>
    ''' <param name="sender">The object that raised the event.</param>
    ''' <param name="e">Data about the event.</param>
    Private Sub PonyGraphicsForm_MouseUp(ByVal sender As Object, ByVal e As MouseEventArgs) Handles Me.MouseUp

        If e.Button = Windows.Forms.MouseButtons.Left Then
            If draggingPony Then
                If Not IsNothing(draggedPony) Then
                    draggedPony.BeingDragged = False
                    If sleeping_before_drag = False Then
                        draggedPony.should_be_sleeping = False
                    End If
                End If
                If Not IsNothing(draggedEffect) Then
                    draggedEffect.beingDragged = False
                End If
                My.Forms.Main.Dragging = False
              
                draggingPony = False

                draggedPony = Nothing
                draggedEffect = Nothing
            End If
        End If

    End Sub

    ' ''' <summary>
    ' ''' Occurs when the mouse enters the form.
    ' ''' We know it is being hovered over.
    ' ''' </summary>
    ' ''' <param name="sender">The object that raised the event.</param>
    ' ''' <param name="e">Data about the event.</param>
    'Private Sub PonyGraphicsForm_MouseEnter(ByVal sender As Object, ByVal e As EventArgs) Handles Me.MouseEnter
    '    MouseOverForm = True
    'End Sub

    ' ''' <summary>
    ' ''' Occurs when the mouse leaves the form.
    ' ''' We know it is not longer being hovered over.
    ' ''' </summary>
    ' ''' <param name="sender">The object that raised the event.</param>
    ' ''' <param name="e">Data about the event.</param>
    'Private Sub PonyGraphicsForm_MouseLeave(ByVal sender As Object, ByVal e As EventArgs) Handles Me.MouseLeave
    '    MouseOverForm = False
    'End Sub

    ''' <summary>
    ''' Occurs when a key is pressed (down and up again) and our form has focus.
    ''' We'll use it to close the graphics form and return to the selection menu.
    ''' </summary>
    ''' <param name="sender">The object that raised the event.</param>
    ''' <param name="e">Data about the event.</param>
    Private Sub PonyGraphicsForm_KeyPress(ByVal sender As Object, ByVal e As KeyEventArgs) Handles Me.KeyDown, Me.KeyUp

        If My.Forms.Main.screen_saver_mode Then Main.Close()

        Dim keyboardinfo As New KeyboardInfo

        With My.Forms.Main
            If keyboardinfo.IsPressed(Keys.Down) Then
                .PonyDown = True
            Else
                .PonyDown = False
            End If
            If keyboardinfo.IsPressed(Keys.Up) Then
                .PonyUp = True
            Else
                .PonyUp = False
            End If

            If keyboardinfo.IsPressed(Keys.Right) Then
                .PonyRight = True
            Else
                .PonyRight = False
            End If
            If keyboardinfo.IsPressed(Keys.Left) Then
                .PonyLeft = True
            Else
                .PonyLeft = False
            End If

            If keyboardinfo.IsPressed(Keys.RShiftKey) Then
                .PonySpeed = True
            Else
                .PonySpeed = False
            End If

            If keyboardinfo.IsPressed(Keys.RControlKey) Then
                .PonyAction = True
            Else
                .PonyAction = False
            End If

            If keyboardinfo.IsPressed(Keys.S) Then
                .PonyDown_2 = True
            Else
                .PonyDown_2 = False
            End If
            If keyboardinfo.IsPressed(Keys.W) Then
                .PonyUp_2 = True
            Else
                .PonyUp_2 = False
            End If

            If keyboardinfo.IsPressed(Keys.D) Then
                .PonyRight_2 = True
            Else
                .PonyRight_2 = False
            End If
            If keyboardinfo.IsPressed(Keys.A) Then
                .PonyLeft_2 = True
            Else
                .PonyLeft_2 = False
            End If

            If keyboardinfo.IsPressed(Keys.LShiftKey) Then
                .PonySpeed_2 = True
            Else
                .PonySpeed_2 = False
            End If

            If keyboardinfo.IsPressed(Keys.LControlKey) Then
                .PonyAction_2 = True
            Else
                .PonyAction_2 = False
            End If
        End With

    End Sub



    Friend Sub setup_addpony_menu()

        Dim selection_menu As ToolStripMenuItem = RightClickMenu.Items("AddPonyMenu")

        For Each pony_tag In My.Forms.Main.PonyFilter_Box.Items

            Dim new_tag As ToolStripMenuItem = selection_menu.DropDownItems.Add(pony_tag)

            For Each Pony In My.Forms.Main.Selectable_Ponies
                For Each category In Pony.Tags

                    If UCase(category) = UCase(pony_tag) Then

                        Dim new_pony_selection As ToolStripMenuItem = new_tag.DropDownItems.Add(Pony.Name)
                        AddHandler new_pony_selection.Click, AddressOf AddPony_Selection

                    End If
                Next

                If pony_tag = "Not Tagged" AndAlso Pony.Tags.Count = 0 Then
                    Dim new_pony_selection As ToolStripMenuItem = new_tag.DropDownItems.Add(Pony.Name)
                    AddHandler new_pony_selection.Click, AddressOf AddPony_Selection
                End If

            Next
        Next

        Dim house_selection_menu As ToolStripMenuItem = RightClickMenu.Items("AddHouseMenu")

        For Each house As Pony.House In My.Forms.Main.Houses
            Dim new_house As ToolStripMenuItem = house_selection_menu.DropDownItems.Add(house.name)
            AddHandler new_house.Click, AddressOf AddHouse_Selection
        Next

    End Sub

    Sub AddPony_Selection(ByVal sender As ToolStripMenuItem, ByVal e As System.EventArgs)

        Dim pony_to_add = sender.Text

        If pony_to_add = "Random Pony" Then
            Dim selection = Math.Round(Rnd() * (My.Forms.Main.Selectable_Ponies.Count - 1), 0)

            pony_to_add = My.Forms.Main.Selectable_Ponies(selection).Name

            If pony_to_add = "Random Pony" Then
                pony_to_add = My.Forms.Main.Selectable_Ponies(selection + 1).Name
            End If
        End If

        For Each Pony In My.Forms.Main.Selectable_Ponies
            If Pony.Name = pony_to_add Then
                Dim new_pony = Pony.Duplicate
                My.Forms.Main.Active_Ponies.Add(new_pony)
                For Each other_Pony In Main.Active_Ponies
                    'we need to set up interactions again to account for new ponies.
                    other_Pony.Initialize_Interactions()
                Next
                Exit For
            End If
        Next

        My.Forms.Main.Right_Click_Menu_Active = False

    End Sub

    Sub AddHouse_Selection(sender As ToolStripMenuItem, e As System.EventArgs)

        Dim house_to_add = sender.Text

        For Each house In My.Forms.Main.Houses
            If house.name = house_to_add Then
                Dim new_house = house.duplicate
                house.Active = True
                house.Instances.Add(new_house)
                new_house.Teleport()
                new_house.PreUpdate_DrawAreas(monitor_number) = New Region(New Rectangle(Pony.Translate_Current_Location(new_house.location, Me), Pony.House.ImageScale(new_house.Rightanimatedimage.Size)))
                new_house.translated_location = Pony.Translate_Current_Location(new_house.location, Me)
                new_house.PostUpdate_DrawAreas(monitor_number) = New Region(New Rectangle(Pony.Translate_Current_Location(new_house.location, Me), Pony.House.ImageScale(new_house.Rightanimatedimage.Size)))
                My.Forms.Main.Active_Effects.Add(new_house)
                Exit For
            End If
        Next

        My.Forms.Main.Right_Click_Menu_Active = False

    End Sub

    Private Sub ContextMenu_Closed() Handles RightClickMenu.Closed
        My.Forms.Main.Right_Click_Menu_Active = False
    End Sub

    Private Sub HouseContextMenu_Closed() Handles HouseRightClickMenu.Closed
        My.Forms.Main.Right_Click_Menu_Active = False
    End Sub

    Private Sub HouseContextMenu_ItemSelected(ByVal sender As System.Object, ByVal e As System.Windows.Forms.ToolStripItemClickedEventArgs) Handles HouseRightClickMenu.ItemClicked

        Select Case e.ClickedItem.Name
            Case "Close_House"
                menuHouse.Instances.Remove(menuHouseEffect)
                My.Forms.Main.Active_Effects.Remove(menuHouseEffect)
                If menuHouse.Instances.Count = 0 Then
                    menuHouse.Active = False
                End If
            Case "HouseMenu"
                My.Forms.HouseOptions_Form.EditHouse(menuHouse)
        End Select

        My.Forms.Main.Right_Click_Menu_Active = False

    End Sub


    Private Sub ContextMenu_ItemSelected(ByVal sender As System.Object, ByVal e As System.Windows.Forms.ToolStripItemClickedEventArgs) Handles RightClickMenu.ItemClicked

        Select Case e.ClickedItem.Name
            Case "Quit"
                menuPony.Cancel_Interaction()
                My.Forms.Main.Active_Ponies.Remove(menuPony)
            Case "Quit_AllType"
                Dim ponies_to_remove As New List(Of Pony)

                For Each Pony In My.Forms.Main.Active_Ponies
                    If Pony.Name = menuPony.Name Then
                        ponies_to_remove.Add(Pony)
                    End If
                Next

                For Each Pony In ponies_to_remove
                    Pony.Cancel_Interaction()
                    My.Forms.Main.Active_Ponies.Remove(Pony)
                Next

            Case "Quit_ALL"
                My.Forms.Main.Close()
            Case "Return_To_Menu"
                If My.Forms.Pony_Editor.Visible = True Then My.Forms.Pony_Editor.Close()
                My.Forms.Main.Pony_Shutdown(True)
                My.Forms.Main.Visible = True
                My.Forms.Main.Opacity = 100 'for when autostarted
                My.Forms.Main.Redraw_Menu()

            Case "Take_Control_P1"
                menuPony.ManualControl_P1 = Not menuPony.ManualControl_P1
                If My.Forms.Main.controlled_pony <> "" Then
                    My.Forms.Main.controlled_pony = menuPony.Name
                Else
                    My.Forms.Main.controlled_pony = ""
                End If
                If Not menuPony.ManualControl_P1 Then
                    RightClickMenu.Items("Take_Control_P1").Text = "Take Control - Player 1"
                Else
                    RightClickMenu.Items("Take_Control_P1").Text = "Release Control - Player 1"
                End If
            Case "Take_Control_P2"
                menuPony.ManualControl_P2 = Not menuPony.ManualControl_P2
                If My.Forms.Main.controlled_pony <> "" Then
                    My.Forms.Main.controlled_pony = menuPony.Name
                Else
                    My.Forms.Main.controlled_pony = ""
                End If
                If Not menuPony.ManualControl_P2 Then
                    RightClickMenu.Items("Take_Control_P2").Text = "Take Control - Player 2"
                Else
                    RightClickMenu.Items("Take_Control_P2").Text = "Release Control - Player 2"
                End If
            Case "Show_Options"
                If My.Forms.Main.Options_Is_Created Then
                    My.Forms.Options.Visible = True
                    My.Forms.Options.Opacity = 100
                Else
                    My.Forms.Options.Show()
                    My.Forms.Main.Options_Is_Created = True
                End If
            Case "Sleep_All"
                My.Forms.Main.sleep_all()
            Case "Sleep"
                menuPony.should_be_sleeping = Not menuPony.should_be_sleeping
        End Select

        My.Forms.Main.Right_Click_Menu_Active = False

    End Sub
End Class


