﻿Imports System.Collections.Generic
Imports System.Drawing
Imports System.Drawing.Imaging
Imports System.Globalization
Imports System.IO
Imports System.Runtime.InteropServices

Module GifDecoding


#Region "GifDecoder class"
    ''' <summary>
    ''' Decodes a gif file, providing the resulting bitmaps for each frame and other information.
    ''' </summary>
    ''' <remarks>
    ''' <para>The goal of this class is to provide a more performant alternative to the existing System.Drawing.Image for gif files. It
    ''' provides the resulting bitmaps for each frame, along with other pertinent information.</para>
    ''' <para>If you wish to use the resulting frames in an animated image, be aware of frames with a zero length duration. These should
    ''' not be displayed. You will need special handling for a file with all-zero durations.</para>
    ''' <para>The System.Drawing.Image class can handle animated gifs fine, but incurs a slight decoding penalty when switching frames.
    ''' This can become a problem when attempting to animate several images, or a large image, in a performant manner. One possibility is
    ''' to save the resulting bitmaps ahead of time, this saves on recalculation but is very memory expensive. The resulting bitmaps
    ''' provided are full 32bbp ARGB, which is four times the maximum gif palette of 8bbp Indexed.</para>
    ''' <para>The class seeks to provide indexed bitmaps of each resulting frame, thus allowing good performance without the same high
    ''' memory costs. However as a result this decoder cannot support files using a full 256 color palette and that also require
    ''' transparency. Additionally this decoder does not support a DisposalMethod of RestoreBackground.</para>
    ''' </remarks>
    ''' System.NotSupportedException - The file used a feature of the gif specification that this partial implementation
    ''' does not support.
    ''' System.exception - Stream was not a gif file, or was a badly formed gif file.
    Public NotInheritable Class GifDecoder
        Implements IDisposable
#Region "BlockCode enum"
        ''' <summary>
        ''' Defines block codes used to identify the type of block.
        ''' </summary>
        Private Enum BlockCode As Byte
            ''' <summary>
            ''' Specifies the block is an image descriptor block (Value 0x2C, 44).
            ''' </summary>
            ImageDescriptor = 44
            ''' <summary>
            ''' Specifies the block is an extension block (Value 0x21, 33).
            ''' </summary>
            Extension = 33
            ''' <summary>
            ''' Specifies the block is the data stream trailer, and that the stream has ended (Value 0x3B, 59).
            ''' </summary>
            Trailer = 59
        End Enum
#End Region

#Region "ExtensionLabel enum"
        ''' <summary>
        ''' Defines extension labels used within extension blocks to identify the type of extension.
        ''' </summary>
        Private Enum ExtensionLabel As Byte
            ''' <summary>
            ''' Specifies the extension is a graphic control extension (Value 0xF9, 249).
            ''' </summary>
            GraphicControl = &HF9
            ''' <summary>
            ''' Specifies the extension is an application extension (Value 0xFF, 255).
            ''' </summary>
            Application = &HFF
            ''' <summary>
            ''' Specifies the extension is a comment extension (Value 0xFE, 254).
            ''' </summary>
            Comment = &HFE
            ''' <summary>
            ''' Specifies the extension is a plain text extension (Value 0x01, 1).
            ''' </summary>
            PlainText = &H1
        End Enum
#End Region

#Region "DataBuffer class"
        ''' <summary>
        ''' Provides a backing store for a set of values in 2-dimensions, with a given number of bits used for each value.
        ''' </summary>
        Private Class DataBuffer
            ''' <summary>
            ''' Gets the underlying array of bytes for this buffer.
            ''' </summary>
            Public Property Buffer() As Byte()
                Get
                    Return m_Buffer
                End Get
                Private Set(value As Byte())
                    m_Buffer = Value
                End Set
            End Property
            Private m_Buffer As Byte()

            ''' <summary>
            ''' The number of bits used per value in the buffer.
            ''' </summary>
            Private m_bitsPerValue As Byte

            ''' <summary>
            ''' Gets the number of bits used per value in the buffer.
            ''' </summary>
            Public ReadOnly Property BitsPerValue() As Byte
                Get
                    Return m_bitsPerValue
                End Get
            End Property
            ''' <summary>
            ''' Gets the number of values that are stored in each byte of the buffer.
            ''' </summary>
            Public ReadOnly Property ValuesPerByte() As Byte
                Get
                    Return CByte(8 \ m_bitsPerValue)
                End Get
            End Property
            ''' <summary>
            ''' Gets the maximum value that may be stored given the current BitsPerValue of the buffer.
            ''' </summary>
            Public ReadOnly Property MaxValue() As Byte
                Get
                    Return CByte((1 << m_bitsPerValue) - 1)
                End Get
            End Property
            ''' <summary>
            ''' Gets the logical dimensions of the buffer.
            ''' </summary>
            Public Property Size() As Size
                Get
                    Return m_Size
                End Get
                Private Set(value As Size)
                    m_Size = Value
                End Set
            End Property
            Private m_Size As Size
            ''' <summary>
            ''' Gets the stride width of the buffer.
            ''' </summary>
            Public Property Stride() As Integer
                Get
                    Return m_Stride
                End Get
                Private Set(value As Integer)
                    m_Stride = Value
                End Set
            End Property
            Private m_Stride As Integer
            ''' <summary>
            ''' Gets the indexed PixelFormat of the smallest size that will support the current BitsPerValue of the buffer.
            ''' </summary>
            Public ReadOnly Property TargetFormat() As PixelFormat
                Get
                    If BitsPerValue = 1 Then
                        Return PixelFormat.Format1bppIndexed
                    End If
                    If BitsPerValue <= 4 Then
                        Return PixelFormat.Format4bppIndexed
                    End If
                    Return PixelFormat.Format8bppIndexed
                End Get
            End Property

            ''' <summary>
            ''' Initializes a new instance of the DataBuffer class capable of holding enough values for the specified dimensions, using the
            ''' given number of bits to store each value.
            ''' </summary>
            ''' <param name="dimensions">The dimensions of the buffer to create.</param>
            ''' <param name="bitsPerValue__1">The number of bits used to store each value.
            ''' Any values given must be representable in this number of bits.
            ''' The number of bits can only be 1, 2, 4 or 8.</param>
            ''' <param name="initialValue">The initial value to use for each entry in the buffer.</param>
            Public Sub New(dimensions As Size, bitsPerValue__1 As Byte, Optional initialValue As Byte = 0)
                If bitsPerValue__1 <> 1 AndAlso bitsPerValue__1 <> 2 AndAlso bitsPerValue__1 <> 4 AndAlso bitsPerValue__1 <> 8 Then
                    Throw New ArgumentException("bitsPerValue may only be 1, 2, 4 or 8.", "bitsPerValue")
                End If

                Size = dimensions
                Me.m_bitsPerValue = bitsPerValue__1

                Stride = CInt(Math.Truncate(Math.Ceiling(CSng(Size.Width * BitsPerValue) / 8.0F))) * ValuesPerByte
                Buffer = New Byte(Stride * Size.Height \ ValuesPerByte - 1) {}

                If initialValue <> 0 Then
                    FillBuffer(initialValue)
                End If
            End Sub

            ''' <summary>
            ''' Efficiently sets all values in the buffer to the given value.
            ''' </summary>
            ''' <param name="value">The value to set for each entry in the buffer.</param>
            Public Sub FillBuffer(value As Byte)
                Dim resultValue As Byte = AllValuesInByte(value)

                ' Apply this byte over the whole buffer.
                For i As Integer = 0 To Buffer.Length - 1
                    Buffer(i) = resultValue
                Next
            End Sub

            ''' <summary>
            ''' Efficiently sets all values in the buffer within the given bounds to the given value.
            ''' </summary>
            ''' <param name="value">The value to set for each entry in the buffer.</param>
            ''' <param name="bounds">The area in which to apply the value. Areas outside these bounds will not be changed.</param>
            Public Sub FillBuffer(value As Byte, bounds As Rectangle)
                If Not New Rectangle(Point.Empty, Size).Contains(bounds) Then
                    Throw New ArgumentException("Given bounds must not extend outside the area of the buffer.", "bounds")
                End If

                Dim resultValue As Byte = AllValuesInByte(value)

                ' We want to set all the indices within the given area only.

                ' Easy method: iterate over the values within area and SetValue everything.
                ' Repeated bit twiddling on the same bytes isn't the fastest though.
                ' for (int y = bounds.Top; y < bounds.Bottom; y++)
                '     for (int x = bounds.Left; x < bounds.Right; x++)
                '         SetValue(x, y, value);

                ' Iterate over each row.
                ' Use SetValue on the left and right edges of the row, when bit twiddling is required.
                ' We can then speed things up by using the contiguous middle region and setting it with our precomputed value.
                For y As Integer = bounds.Top To bounds.Bottom - 1
                    Dim x As Integer = bounds.Left
                    Dim i As Integer = Seek(x, y)
                    ' Set values until we are aligned on a byte
                    While i Mod ValuesPerByte <> 0 AndAlso x < bounds.Right
                        SetValue(x, y, value)
                        x += 1
                        i += 1
                    End While
                    ' Set all the bytes we can on this row.
                    If x < bounds.Right Then
                        Dim rowAlignmentEnd As Integer = (bounds.Right + bounds.Width * y) \ ValuesPerByte
                        For j As Integer = i \ ValuesPerByte To rowAlignmentEnd - 1
                            Buffer(j) = resultValue
                            x += ValuesPerByte
                        Next
                    End If
                    ' Set values in the unaligned area until the row is done.
                    While x < bounds.Right
                        SetValue(x, y, value)
                        x += 1
                    End While
                Next
            End Sub

            ''' <summary>
            ''' Gets a byte with all values in that byte set to the given value.
            ''' </summary>
            ''' <param name="value">The value to set for each entry in the byte.</param>
            ''' <returns>A byte with all values in that byte set to the given value.</returns>
            Private Function AllValuesInByte(value As Byte) As Byte
                Dim resultValue As Byte = value
                If BitsPerValue <> 8 AndAlso value <> Byte.MinValue AndAlso value <> Byte.MaxValue Then
                    If BitsPerValue = 4 Then
                        resultValue = CByte((value << 4) Or value)
                    ElseIf BitsPerValue = 2 Then
                        resultValue = CByte((value << 6) Or (value << 4) Or (value << 2) Or value)
                    ElseIf BitsPerValue = 1 Then
                        resultValue = CByte((value << 7) Or (value << 6) Or (value << 5) Or (value << 4) Or (value << 3) Or (value << 2) Or (value << 1) Or value)
                    End If
                End If
                Return resultValue
            End Function

            ''' <summary>
            ''' Gets the index of the value to seek along the single dimension of the buffer.
            ''' </summary>
            ''' <param name="x">The x co-ordinate of the value.</param>
            ''' <param name="y">The y co-ordinate of the value.</param>
            ''' <returns>The index to seek, along the single dimension of the buffer.</returns>
            Private Function Seek(x As Integer, y As Integer) As Integer
                Return Stride * y + x
            End Function

            ''' <summary>
            ''' Sets a value at the given location in the buffer. The given value is expected to be less than or equal to MaxValue, values
            ''' greater than this will produce incorrect results.
            ''' </summary>
            ''' <param name="x">The x co-ordinate of the value to set.</param>
            ''' <param name="y">The y co-ordinate of the value to set.</param>
            ''' <param name="v">The value to set at this location.</param>
            Public Sub SetValue(x As Integer, y As Integer, v As Byte)
                Dim seek__1 As Integer = Seek(x, y)
                If BitsPerValue = 8 Then
                    Buffer(seek__1) = v
                Else
                    Dim index As Integer = seek__1 \ ValuesPerByte
                    Dim shift As Integer = 8 - BitsPerValue - BitsPerValue * (seek__1 Mod ValuesPerByte)
                    Dim vout As Integer = v << shift
                    Dim mask As Integer = ((&HFF >> 8 - BitsPerValue) << shift) Xor &HFF
                    Buffer(index) = Buffer(index) And CByte(mask)
                    Buffer(index) = Buffer(index) Or CByte(vout)
                End If
            End Sub

            ''' <summary>
            ''' Gets the value from the given location in the buffer.
            ''' </summary>
            ''' <param name="x">The x co-ordinate of the value to get.</param>
            ''' <param name="y">The y co-ordinate of the value to get.</param>
            ''' <returns>The value at this location.</returns>
            Public Function GetValue(x As Integer, y As Integer) As Byte
                Dim seek__1 As Integer = Seek(x, y)
                If BitsPerValue = 8 Then
                    Return Buffer(seek__1)
                Else
                    Dim index As Integer = seek__1 \ ValuesPerByte
                    Dim vin As Integer = Buffer(index) >> (8 - BitsPerValue - BitsPerValue * (seek__1 Mod ValuesPerByte))
                    Dim mask As Integer = &HFF >> 8 - BitsPerValue
                    Return CByte(vin And mask)
                End If
            End Function

            ''' <summary>
            ''' Doubles the current BitsPerValue used in the buffer by doubling its size and copying over the existing values.
            ''' </summary>
            Public Sub UpsizeBuffer()
                Dim newBitsPerValue As Byte = CByte(BitsPerValue * 2)
                Dim newDataBuffer As New DataBuffer(Size, newBitsPerValue)

                For y As Integer = 0 To Size.Height - 1
                    For x As Integer = 0 To Size.Width - 1
                        newDataBuffer.SetValue(x, y, GetValue(x, y))
                    Next
                Next

                Stride = newDataBuffer.Stride
                m_bitsPerValue = newBitsPerValue
                Buffer = newDataBuffer.Buffer
            End Sub
        End Class
#End Region

        ''' <summary>
        ''' Gets the number of times this image plays.
        ''' If 0, it loops indefinitely.
        ''' </summary>
        Public Property Iterations() As Integer
            Get
                Return m_Iterations
            End Get
            Private Set(value As Integer)
                m_Iterations = Value
            End Set
        End Property
        Private m_Iterations As Integer
        ''' <summary>
        ''' Gets the frames that make up this gif.
        ''' </summary>
        Public Property Frames() As IList(Of GifFrame)
            Get
                Return m_Frames
            End Get
            Private Set(value As IList(Of GifFrame))
                m_Frames = Value
            End Set
        End Property
        Private m_Frames As IList(Of GifFrame)
        ''' <summary>
        ''' Gets the description of the logical screen.
        ''' </summary>
        Public Property ScreenDescriptor() As GifLogicalScreenDescriptor
            Get
                Return m_ScreenDescriptor
            End Get
            Private Set(value As GifLogicalScreenDescriptor)
                m_ScreenDescriptor = Value
            End Set
        End Property
        Private m_ScreenDescriptor As GifLogicalScreenDescriptor
        ''' <summary>
        ''' Gets the size of the image.
        ''' </summary>
        Public ReadOnly Property Size() As Size
            Get
                Return ScreenDescriptor.Size
            End Get
        End Property

        ''' <summary>
        ''' Defines a desired remapping of palette colors. If a palette entry matches [i,0] it should be replaced with [i,1].
        ''' </summary>
        Private ReadOnly paletteMap As Color(,)
        ''' <summary>
        ''' Indicates if extended information should be provided.
        ''' </summary>
        Private ReadOnly provideExtendedInfo As Boolean
        ''' <summary>
        ''' The input stream from which the gif will be decoded.
        ''' </summary>
        Private input As BinaryReader

        ''' <summary>
        ''' The global color table.
        ''' </summary>
        Private globalTable As Color()
        ''' <summary>
        ''' The color table currently in use (either global or local).
        ''' </summary>
        Private currentTable As Color()

        ' The values read in are variable length codewords, which can be of any length.
        ' The gif specification sets a maximum code size of 12, and thus a maximum codeword count of 2^12 = 4096.
        ' Thus we need to maintain a dictionary of up to 4096 codewords (which each can be of any length).
        ' These codewords have a property that make them easy to store however.
        ' If some word w plus some character c is in the dictionary, then the word w will be in the dictionary.
        ' Thus the prefix word w can be used to lookup the character at the end of its length.
        ' The suffix character c on this word maps to an actual value that is added to the pixel stack.
        ''' <summary>
        ''' The maximum number of codewords that can occur according to the gif specification.
        ''' </summary>
        Private Const MaxCodeWords As Integer = 4096
        ''' <summary>
        ''' The lookup array of "prefix" words. The value is the index of the "suffix" character at the end of the given codeword.
        ''' </summary>
        Private prefix As Short() = New Short(MaxCodeWords - 1) {}
        ''' <summary>
        ''' The lookup array of "suffix" characters. This holds the actual decompressed byte value for the given code character.
        ''' </summary>
        Private suffix As Byte() = New Byte(MaxCodeWords - 1) {}
        ''' <summary>
        ''' This array is used as a stack to store resulting pixel values. These values are the indexes in the color palette.
        ''' </summary>
        Private pixelStack As Byte() = New Byte(MaxCodeWords) {}
        ''' <summary>
        ''' This array is a buffer to hold the data read in from an image data sub block.
        ''' </summary>
        Private block As Byte() = New Byte(255) {}

        ''' <summary>
        ''' Buffer holding the information for the frame.
        ''' </summary>
        Private frameBuffer As DataBuffer
        ''' <summary>
        ''' Buffer holding information for the subframe, if provideExtendedInfo is true.
        ''' </summary>
        Private subframeBuffer As DataBuffer

        ''' <summary>
        ''' The index in the current color table which we are using for transparency.
        ''' </summary>
        Private globalTransparentIndex As Byte
        ''' <summary>
        ''' An index in the current color table with the same color value as the globalTransparentIndex.
        ''' Therefore, indices clashing with our chosen index can safely be remapped to this index and the result will be the same.
        ''' </summary>
        Private globalTransparentIndexRemap As Byte

        ''' <summary>
        ''' Initializes a new instance of the GifDecoder class by decoding a GIF from the given stream.
        ''' </summary>
        ''' <param name="stream">The Stream containing a file in .gif format.</param>
        ''' <param name="extendedInfo">True to provide extended frame information under Frames[i].Info.
        ''' If false, Frames[i].Info will be null.</param>
        Public Sub New(stream As Stream, Optional extendedInfo As Boolean = False)
            provideExtendedInfo = extendedInfo
            Using binaryReader As New BinaryReader(New BufferedStream(stream))
                DecodeGif(binaryReader)
            End Using
        End Sub

        ''' <summary>
        ''' Initializes a new instance of the GifDecoder class by decoding a GIF from the given stream.
        ''' </summary>
        ''' <param name="stream">The Stream containing a file in .gif format.</param>
        ''' <param name="paletteRemap">An array of dimensions [n,2]. The colors is this array specify how a palette should be remapped. If
        ''' the color in [i,0] is encountered, it will be remapped to [i,1].</param>
        ''' <param name="extendedInfo">True to provide extended frame information under Frames[i].Info.
        ''' If false, Frames[i].Info will be null.</param>
        Public Sub New(stream As Stream, paletteRemap As Color(,), Optional extendedInfo As Boolean = False)
            If paletteRemap IsNot Nothing AndAlso paletteRemap.GetLength(1) <> 2 Then
                Throw New ArgumentException("The dimensions of the paletteRemap array must be [n,2]. " & "The two colors specify a source color and the resulting destination color. You may have n pairs of colors.")
            End If

            paletteMap = paletteRemap
            provideExtendedInfo = extendedInfo
            Using binaryReader As New BinaryReader(New BufferedStream(stream))
                DecodeGif(binaryReader)
            End Using
        End Sub

        ''' <summary>
        ''' Uses the given reader to decode a gif.
        ''' </summary>
        ''' <param name="reader">The BinaryReader that will be used to read input.</param>
        Private Sub DecodeGif(reader As BinaryReader)
            Iterations = 1
            Frames = New List(Of GifFrame)()
            input = reader

            ReadGifDataStream()

            If Frames.Count = 0 Then
                Throw New exception("No frames found within this gif file.")
            End If
        End Sub

        ''' <summary>
        ''' Returns the number of bits per pixel that should be used in an indexed
        ''' bitmap in order to accommodate the color table of given size.
        ''' </summary>
        ''' <param name="sizeOfColorTable">The size of the color table that must be accommodated.</param>
        ''' <returns>The lowest number of bits that is an acceptable indexing format that can
        ''' represent sufficient indexes for the color table of given size.</returns>
        Private Shared Function TargetBitsPerPixel(sizeOfColorTable As Integer) As Byte
            Dim tableBitsPerPixel As Integer = 1
            Dim maxIndexValue As Integer = sizeOfColorTable - 1
            While maxIndexValue > 1
                tableBitsPerPixel += 1
                maxIndexValue >>= 1
            End While
            Select Case tableBitsPerPixel
                Case 1, 4
                    Return CByte(tableBitsPerPixel)
                Case 2, 3
                    Return 4
                Case Else
                    Return 8
            End Select
        End Function

        ''' <summary>
        ''' Determines the global transparent indices to use in the given color table.
        ''' </summary>
        ''' <param name="colorTable">The color table in which transparent indices should be selected.</param>
        Private Sub DetermineGlobalTransparentIndices(colorTable As Color())
            ' Determine the value to use for the global transparent index.
            ' Find a duplicated value if possible, either both are unused, or one
            ' is used for transparency and one as an index.
            ' We'll process in reverse, since padding the table usually leaves the
            ' end with lots of zeroed out black colors.
            Dim duplicateValueFound As Boolean = False
            Dim i As Integer = colorTable.Length - 1
            While i >= 0 AndAlso Not duplicateValueFound
                Dim j As Integer = colorTable.Length - 1
                While j >= 0 AndAlso Not duplicateValueFound
                    If i <> j AndAlso colorTable(i) = colorTable(j) Then
                        globalTransparentIndex = CByte(i)
                        globalTransparentIndexRemap = CByte(j)
                        duplicateValueFound = True
                    End If
                    j -= 1
                End While
                i -= 1
            End While

            ' If there is no duplicated value, two things might happen:
            ' A) The palette was full after reserving a transparent index, so we don't know it.
            ' B) Clever/sadistic encoders are using the whole palette and swapping the local transparent index each frame.
            ' We can't know unless we process the whole image. Instead we'll take a stab at the last index, since several encoders
            ' often reserve that. If we don't get away with it, we can bump up the size of the table.
            If Not duplicateValueFound Then
                globalTransparentIndex = CByte(colorTable.Length - 1)
                globalTransparentIndexRemap = CByte(colorTable.Length - 1)
            End If
        End Sub

        ''' <summary>
        ''' Reads the gif data stream.
        ''' </summary>
        Private Sub ReadGifDataStream()
            ' <GIF Data Stream> ::= Header <Logical Screen> <Data>* Trailer
            ReadHeader()
            ReadLogicalScreen()
            ReadDataAndTrailer()
        End Sub

        ''' <summary>
        ''' Reads the header block. This is a required block, one per data stream.
        ''' </summary>
        Private Sub ReadHeader()
            Const SignatureExpected As String = "GIF"
            Dim signature As New String(input.ReadChars(3))
            If signature <> SignatureExpected Then
                Throw New exception(String.Format(CultureInfo.CurrentCulture, "Invalid signature in header. Expected '{0}'. Read '{1}'.", SignatureExpected, signature))
            End If

            Const Version87a As String = "87a"
            Const Version89a As String = "89a"
            Dim version As New String(input.ReadChars(3))
            If version <> Version87a AndAlso version <> Version89a Then
                Throw New exception(String.Format(CultureInfo.CurrentCulture, "Invalid version in header. Expected '{0}' or '{1}'. Read '{2}'.", Version87a, Version89a, version))
            End If
        End Sub

        ''' <summary>
        ''' Reads the logical screen section.
        ''' </summary>
        Private Sub ReadLogicalScreen()
            ' <Logical Screen> ::= Logical Screen Descriptor [Global Color Table]
            ScreenDescriptor = ReadLogicalScreenDescriptor()
            If ScreenDescriptor.GlobalTableExists Then
                globalTable = ReadColorTable(ScreenDescriptor.GlobalTableSize)

                DetermineGlobalTransparentIndices(globalTable)
                frameBuffer = New DataBuffer(ScreenDescriptor.Size, TargetBitsPerPixel(globalTable.Length), globalTransparentIndex)
            Else
                globalTransparentIndex = 255
                globalTransparentIndexRemap = 255
                frameBuffer = New DataBuffer(ScreenDescriptor.Size, 8, globalTransparentIndex)
            End If
        End Sub

        ''' <summary>
        ''' Reads the logical screen descriptor block. This is a required block, one per data stream.
        ''' </summary>
        ''' <returns>A new LogicalScreenDescriptor describing the logical screen.</returns>
        Private Function ReadLogicalScreenDescriptor() As GifLogicalScreenDescriptor
            Dim logicalScreenWidth As UShort = input.ReadUInt16()
            Dim logicalScreenHeight As UShort = input.ReadUInt16()

            Dim packedFields As Byte = input.ReadByte()
            Dim globalColorTableFlag As Boolean = (packedFields And &H80) <> 0
            Dim colorResolution As Byte = CByte(((packedFields And &H70) >> 4) + 1)
            Dim sortFlag As Boolean = (packedFields And &H8) >> 3 = 1
            Dim globalBitsPerPixel As Integer = (packedFields And &H7) + 1
            Dim sizeOfGlobalColorTable As Integer = 1 << globalBitsPerPixel
            Dim backgroundColorIndex As Byte = input.ReadByte()
            Dim pixelAspectRatio As Byte = input.ReadByte()

            Return New GifLogicalScreenDescriptor(logicalScreenWidth, logicalScreenHeight, globalColorTableFlag, colorResolution, sortFlag, sizeOfGlobalColorTable, _
                backgroundColorIndex, pixelAspectRatio)
        End Function

        ''' <summary>
        ''' Reads multiple data sections and the stream trailer.
        ''' </summary>
        Private Sub ReadDataAndTrailer()
            ' <GIF Data Stream> ::= Header <Logical Screen> <Data>* Trailer
            ' <Data> ::= <Graphic Block> | <Special-Purpose Block>

            Dim blockCode__1 As BlockCode

            ' <Graphic Block> ::= [Graphic Control Extension] <Graphic-Rendering Block>
            ' <Special-Purpose Block> ::= Application Extension | Comment Extension
            ' <Graphic-Rendering Block> ::= <Table-Based Image> | Plain Text Extension
            ' <Table-Based Image> ::= Image Descriptor [Local Color Table] Image Data

            Do
                blockCode__1 = CType(input.ReadByte(), BlockCode)
                Select Case blockCode__1
                    Case BlockCode.ImageDescriptor
                        ReadTableBasedImage()
                        Exit Select
                    Case BlockCode.Extension
                        Dim extensionLabel__2 As ExtensionLabel = CType(input.ReadByte(), ExtensionLabel)
                        Select Case extensionLabel__2
                            Case ExtensionLabel.GraphicControl
                                ReadGraphicBlock()
                                Exit Select
                            Case ExtensionLabel.Application
                                ReadApplicationExtension()
                                Exit Select
                            Case Else
                                SkipExtensionBlock()
                                Exit Select
                        End Select
                        Exit Select
                    Case BlockCode.Trailer
                        Exit Select
                    Case Else
                        Throw New exception("Expected the start of a data section but block code did not match any known values.")
                End Select
            Loop While blockCode__1 <> BlockCode.Trailer
        End Sub

        ''' <summary>
        ''' Reads a graphic block section.
        ''' </summary>
        Private Sub ReadGraphicBlock()
            ' <Graphic Block> ::= [Graphic Control Extension] <Graphic-Rendering Block>
            ' <Graphic-Rendering Block> ::= <Table-Based Image> | Plain Text Extension
            ' <Table-Based Image> ::= Image Descriptor [Local Color Table] Image Data
            Dim gce As GifGraphicControlExtension = ReadGraphicControlExtension()

            Dim blockCode__1 As BlockCode = CType(input.ReadByte(), BlockCode)
            Dim extensionLabel__2 As ExtensionLabel = ExtensionLabel.GraphicControl
            If blockCode__1 = BlockCode.Extension Then
                extensionLabel__2 = CType(input.ReadByte(), ExtensionLabel)
                '#Region "Read extension blocks."
                If extensionLabel__2 <> ExtensionLabel.PlainText Then
                    ' We have some other extension here, we need to skip them until we meet something valid.
                    Do
                        ' Read block.
                        If extensionLabel__2 = ExtensionLabel.Application Then
                            ReadApplicationExtension()
                        Else
                            SkipExtensionBlock()
                        End If

                        ' Determine next block.
                        blockCode__1 = CType(input.ReadByte(), BlockCode)
                        If blockCode__1 = BlockCode.Extension Then
                            extensionLabel__2 = CType(input.ReadByte(), ExtensionLabel)

                            ' Continue until we have a valid start for a graphic-rendering block.
                        End If
                    Loop While blockCode__1 <> BlockCode.ImageDescriptor AndAlso extensionLabel__2 <> ExtensionLabel.PlainText
                    '#End Region
                End If
            End If

            ' Read graphic rendering block.
            If blockCode__1 = BlockCode.ImageDescriptor Then
                ' Read table based image data to finish the block.
                ReadTableBasedImage(gce)
            ElseIf extensionLabel__2 = ExtensionLabel.PlainText Then
                ' "Read" a plain text extension to finish the block.
                SkipExtensionBlock()
            Else
                Throw New exception("Encountered an unexpected block code after reading the graphic control extension in a graphic block.")
            End If
        End Sub

        ''' <summary>
        ''' Reads a graphic control extension block.
        ''' </summary>
        ''' <returns>A new GraphicControlExtension describing how a graphic rendering block section is to be modified.</returns>
        Private Function ReadGraphicControlExtension() As GifGraphicControlExtension
            ' Read block size (4).
            input.ReadByte()

            Dim packedFields As Byte = input.ReadByte()
            Dim disposalMethod As GifDisposalMethod = CType((packedFields And &H1C) >> 2, GifDisposalMethod)
            Dim userInputFlag As Boolean = (packedFields And &H2) <> 0
            Dim transparencyFlag As Boolean = (packedFields And &H1) <> 0

            Dim delayTime As UShort = input.ReadUInt16()
            Dim transparentColorIndex As Byte = input.ReadByte()

            ' Read block terminator.
            input.ReadByte()

            Return New GifGraphicControlExtension(disposalMethod, userInputFlag, transparencyFlag, delayTime, transparentColorIndex)
        End Function

        ''' <summary>
        ''' Reads an application extension block.
        ''' </summary>
        Private Sub ReadApplicationExtension()
            ' Read block size (11).
            input.ReadByte()

            Dim applicationIdentifier As New String(input.ReadChars(8))
            Dim applicationAuthenticationCode As New String(input.ReadChars(3))

            If applicationIdentifier = "NETSCAPE" AndAlso applicationAuthenticationCode = "2.0" Then
                ReadNetscapeApplicationExtension()
            Else
                SkipDataSubBlocks()
            End If
        End Sub

        ''' <summary>
        ''' Reads the Netscape application extension sub-block, which defines an iteration count for the file.
        ''' </summary>
        Private Sub ReadNetscapeApplicationExtension()
            ' Read block size (3).
            input.ReadByte()

            ' Read empty byte.
            input.ReadByte()

            ' Read iteration count.
            Iterations = input.ReadUInt16()

            ' Read sub-block terminator.
            input.ReadByte()
        End Sub

        ''' <summary>
        ''' Reads a series of data sub blocks until a terminator is encountered.
        ''' </summary>
        Private Sub SkipDataSubBlocks()
            Dim blockSize As Byte
            Do
                blockSize = input.ReadByte()
                input.ReadBytes(blockSize)
            Loop While blockSize > 0
        End Sub

        ''' <summary>
        ''' Reads an extension block.
        ''' </summary>
        Private Sub SkipExtensionBlock()
            ' Read block size.
            Dim blockSize As Byte = input.ReadByte()

            If blockSize = 0 Then
                Return
            End If

            ' Read block data.
            input.ReadBytes(blockSize)

            ' Skip data sub blocks.
            SkipDataSubBlocks()
        End Sub

        ''' <summary>
        ''' Reads a color table block containing the given number of colors. These are optional blocks.
        ''' Up to one global table may exist per data stream. Up to one local table may exist per image.
        ''' </summary>
        ''' <param name="colorCount">The number of colors in the table. Colors are 24bbp RGB values.</param>
        ''' <returns>A new Color[] containing the colors read in from the block.</returns>
        Private Function ReadColorTable(colorCount As Integer) As Color()
            Dim colors As Color() = New Color(colorCount - 1) {}
            For i As Integer = 0 To colors.Length - 1
                colors(i) = Color.FromArgb(input.ReadByte(), input.ReadByte(), input.ReadByte())
            Next
            Return colors
        End Function

        ''' <summary>
        ''' Reads the image descriptor block. This is a required block, one is required per image in the data stream.
        ''' </summary>
        ''' <returns>A new ImageDescriptor describing the subframe.</returns>
        Private Function ReadImageDescriptor() As GifImageDescriptor
            Dim imageLeftPosition As UShort = input.ReadUInt16()
            Dim imageTopPosition As UShort = input.ReadUInt16()
            Dim imageWidth As UShort = input.ReadUInt16()
            Dim imageHeight As UShort = input.ReadUInt16()

            Dim packedFields As Byte = input.ReadByte()
            Dim localColorTableFlag As Boolean = (packedFields And &H80) <> 0
            Dim interlaceFlag As Boolean = (packedFields And &H40) <> 0
            Dim sortFlag As Boolean = (packedFields And &H20) <> 0
            Dim localBitsPerPixel As Integer = (packedFields And &H7) + 1
            Dim sizeOfLocalColorTable As Integer = 1 << localBitsPerPixel

            Return New GifImageDescriptor(imageLeftPosition, imageTopPosition, imageWidth, imageHeight, localColorTableFlag, interlaceFlag, _
                sortFlag, sizeOfLocalColorTable)
        End Function

        ''' <summary>
        ''' Reads a table based image section.
        ''' </summary>
        ''' <param name="graphicControl">The GraphicControlExtension specifying how the value from the subframe is to be applied.</param>
        Private Sub ReadTableBasedImage(Optional graphicControl As GifGraphicControlExtension = Nothing)
            ' <Table-Based Image> ::= Image Descriptor [Local Color Table] Image Data
            Dim image As GifImageDescriptor = ReadImageDescriptor()
            If Not New Rectangle(Point.Empty, Size).Contains(image.Subframe) Then
                Throw New exception("Subframe area extended outside the logical screen area.")
            End If
            If image.LocalTableExists Then
                currentTable = ReadColorTable(image.LocalTableSize)
                Dim localBbp As Integer = TargetBitsPerPixel(currentTable.Length)
                While localBbp > frameBuffer.BitsPerValue
                    frameBuffer.UpsizeBuffer()
                End While
            ElseIf ScreenDescriptor.GlobalTableExists Then
                currentTable = globalTable
            Else
                ' A file is obliged to provide at least one color table.
                ' Some files using local colors tables in every frame skip this step if they don't need any colors because the subframe is
                ' fully transparent. We'll provide this table with one entry.
                currentTable = New Color() {Color.Transparent}
            End If

            ' Set our buffer up, then read in the image data from the stream.
            ' After this, the buffer will contain the frame we want.
            DetermineAndRemapGlobalTransparentIndices(currentTable, frameBuffer)
            ReadImageData(image.Interlaced, image, graphicControl)

            ' Create the frame bitmap, we'll be reusing the buffer so the byte array will need to be duplicated.
            Dim transparentIndex As Integer = If((graphicControl IsNot Nothing AndAlso graphicControl.TransparencyUsed), globalTransparentIndex, -1)
            Dim frameBitmap As Bitmap = CreateBitmap(frameBuffer, True, transparentIndex)

            ' Apply the disposal method.
            If graphicControl IsNot Nothing Then
                ' GifDisposalMethod.Undefined is undefined, if encountered it is treated as DoNotDispose.
                ' GifDisposalMethod.DoNotDispose indicates that the buffer should remain as it is, so do nothing.

                ' Clear the background of the subframe area.
                If graphicControl.DisposalMethod = GifDisposalMethod.RestoreBackground Then
                    frameBuffer.FillBuffer(globalTransparentIndex, image.Subframe)
                End If

                ' Restore the subframe area to the image from the previous frame.
                ' The spec is forgiving of not implementing this, plus nobody uses it, so we'll be fine.
                If graphicControl.DisposalMethod = GifDisposalMethod.RestorePrevious Then
                    Throw New NotSupportedException("Cannot decode a gif with a DisposalMethod of RestorePrevious")
                End If
            End If

            If provideExtendedInfo Then
                ' Create the subframe bitmap and save full information.
                Dim subframeTransparent As Integer = If((graphicControl IsNot Nothing AndAlso graphicControl.TransparencyUsed), graphicControl.TransparentIndex, -1)
                Dim subframeBitmap As Bitmap = CreateBitmap(subframeBuffer, False, subframeTransparent)
                Frames.Add(New GifFrame(frameBitmap, If(graphicControl IsNot Nothing, graphicControl.Delay, 0), subframeBitmap, image, graphicControl))
            Else
                ' Just use the frame bitmap and save minimal information.
                Frames.Add(New GifFrame(frameBitmap, If(graphicControl IsNot Nothing, graphicControl.Delay, 0)))
            End If
        End Sub

        ''' <summary>
        ''' Creates a new bitmap from the given DataBuffer.
        ''' </summary>
        ''' <param name="buffer">The DataBuffer that holds information to format the bitmap and the underlying data.</param>
        ''' <param name="duplicateUnderlyingBuffer">If true, a copy of the underlying bytes in the buffer will be made before passing that
        ''' data to the bitmap. This is required if the buffer will be modified further after this method returns.
        ''' If false, the underlying bytes will be directly used. The buffer may not be modified if this option is used of the image will
        ''' be overwritten. In either case, the DataBuffer need not be kept in scope.</param>
        ''' <param name="transparentIndex">The index of the transparent color in the image, or -1 to specify no transparency.</param>
        ''' <returns>A new Bitmap in the format specified by the given buffer that is the image held by that buffer.</returns>
        <System.Security.Permissions.SecurityPermissionAttribute(System.Security.Permissions.SecurityAction.Demand, Flags:=System.Security.Permissions.SecurityPermissionFlag.UnmanagedCode)> _
        Private Function CreateBitmap(buffer As DataBuffer, duplicateUnderlyingBuffer As Boolean, Optional transparentIndex As Integer = -1) As Bitmap
            ' Create the bitmap and lock it.
            Dim bitmap As New Bitmap(buffer.Size.Width, buffer.Size.Height, buffer.TargetFormat)
            Dim data As BitmapData = bitmap.LockBits(New Rectangle(0, 0, buffer.Size.Width, buffer.Size.Height), ImageLockMode.[WriteOnly], bitmap.PixelFormat)

            Dim bytes As Byte() = buffer.Buffer
            If duplicateUnderlyingBuffer Then
                ' Make a copy of the buffer which can be inserted into a bitmap.
                ' Otherwise, the bitmap image will be overridden when the buffer is modified.
                bytes = New Byte(buffer.Buffer.Length - 1) {}
                Array.Copy(buffer.Buffer, bytes, bytes.Length)
            End If

            ' Copy the frame buffer to the bitmap. To account for stride padding, copy row by row. Then unlock it.
            For row As Integer = 0 To data.Height - 1
                Marshal.Copy(bytes, row * buffer.Stride \ buffer.ValuesPerByte, New IntPtr(data.Scan0.ToInt64 + (row * data.Stride)), buffer.Stride \ buffer.ValuesPerByte)
                'inptr.add is not supported in .net 3.5
                'Marshal.Copy(bytes, row * buffer.Stride \ buffer.ValuesPerByte, IntPtr.Add(data.Scan0, row * data.Stride), buffer.Stride \ buffer.ValuesPerByte)
            Next
            bitmap.UnlockBits(data)

            ' Fill in the color palette from the current table.
            Dim palette As ColorPalette = bitmap.Palette
            Dim i As Integer = 0
            While i < palette.Entries.Length AndAlso i < currentTable.Length
                palette.Entries(i) = currentTable(i)
                i += 1
            End While

            ' Apply giving palette mapping.
            If paletteMap IsNot Nothing Then
                For paletteIndex As Integer = 0 To palette.Entries.Length - 1
                    For mapIndex As Integer = 0 To paletteMap.GetLength(0) - 1
                        If palette.Entries(paletteIndex) = paletteMap(mapIndex, 0) Then
                            palette.Entries(paletteIndex) = paletteMap(mapIndex, 1)
                        End If
                    Next
                Next
            End If

            ' Apply transparency.
            If transparentIndex <> -1 Then
                palette.Entries(transparentIndex) = Color.Transparent
            End If

            ' Set palette on bitmap.
            bitmap.Palette = palette
            Return bitmap
        End Function

        ''' <summary>
        ''' Reads image data onto the frame buffer. This is the LZW compressed information about the pixels in the subframe.
        ''' </summary>
        ''' <param name="interlacing">Indicates if interlacing is used in this image.
        ''' If true, image will be de-interlaced as it is read.</param>
        ''' <param name="image">The ImageDescriptor describing the subframe.</param>
        ''' <param name="graphicControl">The GraphicControlExtension specifying how the value from the subframe is to be applied.</param>
        Private Sub ReadImageData(interlacing As Boolean, image As GifImageDescriptor, Optional graphicControl As GifGraphicControlExtension = Nothing)
            If provideExtendedInfo Then
                subframeBuffer = New DataBuffer(image.Subframe.Size, frameBuffer.BitsPerValue)
            End If

            Dim lzwMinimumCodeSize As Byte = input.ReadByte()

            '#Region "Initialize decoder."
            ' Image pixel position data.
            Dim width As Integer = image.Subframe.Width
            Dim height As Integer = image.Subframe.Height
            Dim x As Integer = 0
            Dim y As Integer = 0
            Dim interlacePass As Integer = 1
            Dim yIncrement As Integer = If(interlacing, 8, 1)

            ' Indicates a null codeword.
            Const NullCode As Integer = -1
            ' The clear code is the first available unused code. We will use values below this as root characters in the dictionary.
            Dim clearCode As Integer = 1 << lzwMinimumCodeSize
            ' This code indicates the end of LZW compressed information.
            Dim endOfInformation As Integer = clearCode + 1
            ' This index indicates the first unused index in the dictionary.
            Dim available As Integer = clearCode + 2
            ' The size of codes being read in, in bits.
            Dim codeSize As Integer = lzwMinimumCodeSize + 1
            ' The mask used to get the current code from the bit buffer.
            Dim codeMask As Integer = (1 << codeSize) - 1
            ' Stores the old codeword.
            Dim oldCode As Integer = NullCode
            ' Stores the root character at the end of a codeword.
            Dim rootCode As Byte = 0

            ' Variables managing the data block buffer from which data is read in.
            Dim blockIndex As Integer = 0
            Dim bytesLeftInBlock As Integer = 0
            Dim bitsBuffered As Integer = 0
            Dim bitBuffer As Integer = 0
            ' Variable for the next available index in the stack.
            Dim stackIndex As Integer = 0

            ' Initialize table with root characters.
            For i As Integer = 0 To clearCode - 1
                prefix(i) = 0
                suffix(i) = CByte(i)
            Next
            '#End Region

            '#Region "Decode GIF pixel stream."
            Dim skipDataSubBlocks__1 As Boolean = True
            Dim pixelIndex As Integer = 0
            While pixelIndex < width * height
                ' Read some values into the pixel stack as it is empty.
                If stackIndex = 0 Then
                    '#Region "Read in another byte from the block buffer if we need more bits."
                    If bitsBuffered < codeSize Then
                        ' Read in a new data block if we exhausted the block buffer.
                        If bytesLeftInBlock = 0 Then
                            ' Read a new data block.
                            bytesLeftInBlock = input.ReadByte()
                            block = input.ReadBytes(bytesLeftInBlock)
                            blockIndex = 0

                            ' If we happen to read the block terminator, we are done reading image data.
                            If bytesLeftInBlock = 0 Then
                                ' No need to skip remaining blocks, since we happened to read the terminator ourselves.
                                skipDataSubBlocks__1 = False
                                Exit While
                            End If
                        End If
                        bitBuffer += CInt(block(blockIndex)) << bitsBuffered
                        bitsBuffered += 8
                        blockIndex += 1
                        bytesLeftInBlock -= 1
                        Continue While
                    End If
                    '#End Region

                    ' Get the next code from out bit buffer.
                    Dim code As Integer = bitBuffer And codeMask
                    bitBuffer >>= codeSize
                    bitsBuffered -= codeSize

                    '#Region "Interpret special codes."
                    ' Indicates the end of input data.
                    If (code > available) OrElse (code = endOfInformation) Then
                        Exit While
                    End If

                    ' A clear code means the decoder dictionary should be reset.
                    If code = clearCode Then
                        codeSize = lzwMinimumCodeSize + 1
                        codeMask = (1 << codeSize) - 1
                        available = clearCode + 2
                        oldCode = NullCode
                        Continue While
                    End If

                    ' Record the first code after an initialization/reset.
                    If oldCode = NullCode Then
                        pixelStack(stackIndex) = suffix(code)
                        stackIndex += 1
                        oldCode = code
                        rootCode = CByte(code)
                        Continue While
                    End If
                    '#End Region

                    ' Save the code we read in.
                    ' We will be using the code variable as an indexer to move through the word.
                    Dim inCode As Integer = code

                    '#Region "Get codeword values and push them to the stack."
                    ' Handle the case where a code word was not yet defined. This only happens in one special case.
                    ' Given string s and character c, this happens only when we see the sequence s.c.s.c.s and the word s.c was already in
                    ' the dictionary beforehand.
                    If code = available Then
                        pixelStack(stackIndex) = rootCode
                        stackIndex += 1
                        code = oldCode
                    End If
                    ' Enumerate through the code word and push values to the stack.
                    While code > clearCode
                        pixelStack(stackIndex) = suffix(code)
                        stackIndex += 1
                        code = prefix(code)
                    End While
                    ' Record and push the root character to the stack.
                    rootCode = suffix(code)
                    pixelStack(stackIndex) = rootCode
                    stackIndex += 1
                    '#End Region

                    '#Region "Add a new codeword to the dictionary."
                    ' Add words to the dictionary whilst it is not full.
                    ' If it is full, we keep the same dictionary until the encoder sends a clear code.
                    If available < MaxCodeWords Then
                        ' Add a new word to the dictionary.
                        prefix(available) = CShort(oldCode)
                        suffix(available) = rootCode
                        available += 1

                        ' Increase the code size if the number of available codes now exceeds the number addressable by the current size.
                        If available < MaxCodeWords AndAlso (available And codeMask) = 0 Then
                            codeSize += 1
                            codeMask += available
                        End If
                    End If
                    '#End Region

                    ' Note the previous codeword.
                    oldCode = inCode
                End If

                '#Region "Pop a pixel off the pixel stack."
                stackIndex -= 1
                pixelIndex += 1

                ' Apply pixel to our buffers.
                ApplyPixelToFrame(x, y, pixelStack(stackIndex), image, graphicControl)
                If provideExtendedInfo Then
                    While pixelStack(stackIndex) > subframeBuffer.MaxValue
                        subframeBuffer.UpsizeBuffer()
                    End While
                    subframeBuffer.SetValue(x, y, pixelStack(stackIndex))
                End If

                ' Move right one pixel.
                x += 1
                If x >= width Then
                    ' Move to next row. If we're not interlacing this just means the next row.
                    ' If interlacing, we must fill in every 8th row, then every 4th, then every 2nd then every other row.
                    x = 0
                    y += yIncrement

                    ' If we reached the end of this interlacing pass, go back to the top and fill in every row between the current rows.
                    If interlacing AndAlso y >= height Then
                        '#Region "Choose next interlacing line."
                        Do
                            interlacePass += 1
                            Select Case interlacePass
                                Case 2
                                    y = 4
                                    Exit Select
                                Case 3
                                    y = 2
                                    yIncrement = 4
                                    Exit Select
                                Case 4
                                    y = 1
                                    yIncrement = 2
                                    Exit Select
                            End Select
                            '#End Region
                        Loop While y >= height
                    End If
                    '#End Region
                End If
            End While
            '#End Region

            ' Read block terminator (and any trailing sub-blocks, though there should not be any).
            If skipDataSubBlocks__1 Then
                SkipDataSubBlocks()
            End If
        End Sub

        ''' <summary>
        ''' Determines the global transparent indices to use in the given color table.
        ''' If these differ from the current indices, remaps the old indices to the new values in the given buffer.
        ''' </summary>
        ''' <param name="colorTable">The color table in which transparent indices should be selected.</param>
        ''' <param name="buffer">The buffer in which to remap indices if new ones are selected.</param>
        Private Sub DetermineAndRemapGlobalTransparentIndices(colorTable As Color(), buffer As DataBuffer)
            Dim oldGlobalTransparentIndex As Integer = globalTransparentIndex
            DetermineGlobalTransparentIndices(colorTable)
            If globalTransparentIndex <> oldGlobalTransparentIndex Then
                For y As Integer = 0 To buffer.Size.Height - 1
                    For x As Integer = 0 To buffer.Size.Width - 1
                        If buffer.GetValue(x, y) = oldGlobalTransparentIndex Then
                            buffer.SetValue(x, y, globalTransparentIndex)
                        End If
                    Next
                Next
            End If
        End Sub

        ''' <summary>
        ''' Uses the values in the subframe buffer and applies them onto the frame buffer.
        ''' </summary>
        ''' <param name="x">The x co-ordinate, relative to the subframe bounds, where the pixel is to be applied.</param>
        ''' <param name="y">The y co-ordinate, relative to the subframe bounds, where the pixel is to be applied.</param>
        ''' <param name="pixel">The value to be applied, in accordance with the graphicControl parameters if specified.</param>
        ''' <param name="image">The ImageDescriptor describing the subframe.</param>
        ''' <param name="graphicControl">The GraphicControlExtension specifying how the value from the subframe is to be applied.</param>
        Private Sub ApplyPixelToFrame(x As Integer, y As Integer, pixel As Byte, image As GifImageDescriptor, Optional graphicControl As GifGraphicControlExtension = Nothing)
            ' This is logically nasty. I've chosen to use lots of returns because massively nested ifs were a poorer choice.
            ' The short version of this function goes like this:
            ' If the pixel is not transparent, set it. Otherwise do nothing.
            ' The full version for this must handle a possibly null graphicsControl.
            ' Plus this implementation is hoping to steal a spot in the color table, which it won't always get away with.
            ' So:
            ' If transparency is not used, we can set the pixel.
            ' If transparency is used, and this pixel is transparent, we can skip it.
            ' If transparency is used, but this pixel is not transparent, we have an additional check.
            ' We need to set the pixel, but since we're using a spot in the table for global transparency we must resolve conflicts.
            ' If there's no conflict, we can set the pixel.
            ' If there's a conflict, we can set the pixel to the backup value if one exists.
            ' If there's no backup value, we need to upsize the color table, and possibly the buffer, which will resolve the conflict.
            ' Then we can set the pixel.

            ' We can set the pixel if there is no transparency to worry about.
            If graphicControl Is Nothing OrElse Not graphicControl.TransparencyUsed Then
                ' Just check the encoded value is within the table size, in case of a badly encoded file.
                CheckValueInRange(pixel)
                frameBuffer.SetValue(x + image.Subframe.X, y + image.Subframe.Y, pixel)
                Return
            End If

            ' We can skip setting a transparent pixel.
            If pixel = graphicControl.TransparentIndex Then
                Return
            End If

            ' If there's no conflict with the global transparent index, we can set the pixel safely.
            If pixel <> globalTransparentIndex Then
                ' Just check the encoded value is within the table size, in case of a badly encoded file.
                CheckValueInRange(pixel)
                frameBuffer.SetValue(x + image.Subframe.X, y + image.Subframe.Y, pixel)
                Return
            End If

            ' There was a conflict, but we can try the backup value.
            If globalTransparentIndex <> globalTransparentIndexRemap Then
                ' A remap exists, so we can safely remap to it.
                frameBuffer.SetValue(x + image.Subframe.X, y + image.Subframe.Y, globalTransparentIndexRemap)
                Return
            End If

            ' There was a conflict and there is no backup value.
            ' We will have to bite the bullet and upsize the color table.
            ' If the color table gets too big, we might have to upscale the buffer size to accommodate as well.
            If currentTable.Length < 256 Then
                ' Create a new table with an extra color.
                Dim newTable As Color() = New Color(currentTable.Length) {}
                Array.Copy(currentTable, newTable, currentTable.Length)

                ' Use the new table instead of our old ones.
                If Not image.LocalTableExists Then
                    globalTable = newTable
                End If
                currentTable = newTable

                ' If the new table needs a higher BBP than we currently have, upscale the buffer too.
                Dim newBbp As Integer = TargetBitsPerPixel(newTable.Length)
                While newBbp > frameBuffer.BitsPerValue
                    frameBuffer.UpsizeBuffer()
                End While

                ' Remap global transparent indices.
                DetermineAndRemapGlobalTransparentIndices(currentTable, frameBuffer)

                ' We can now safely set the pixel value as we have moved the global index to our new slot.
                frameBuffer.SetValue(x + image.Subframe.X, y + image.Subframe.Y, pixel)
                Return
            End If

            ' The color table was already at maximum size. Since we had a conflict we need a table of size 257 to accommodate transparent.
            ' This is impossible for an indexed bitmap, and we're screwed. Luckily this is unlikely to happen so we can just die.
            ' TODO: Fall back to 32bbp ARGB buffer instead. This would require a non-indexed buffer and translating the table lookup back
            ' into full colors.
            Throw New NotSupportedException("Decoder does not support images using transparency without a reserved index in the palette when the palette is at " & "maximum size.")
        End Sub

        ''' <summary>
        ''' Checks the given pixel value in within the size of the current color table, or else throws a exception.
        ''' </summary>
        ''' <param name="value">The value which is expected to fit within the current color table.</param>
        Private Sub CheckValueInRange(value As Byte)
            If value >= currentTable.Length Then
                Throw New exception(String.Format(CultureInfo.CurrentCulture, "Indexed value of {0} was larger that the maximum of {1}.", value, currentTable.Length - 1))
            End If
        End Sub

        ''' <summary>
        ''' Releases all resources used by this object.
        ''' </summary>
        Public Sub Dispose() Implements IDisposable.Dispose
            For Each frame As GifFrame In Frames
                frame.Dispose()
            Next
        End Sub
    End Class
#End Region

#Region "GifFrame class"
    ''' <summary>
    ''' Defines single frames within an animation.
    ''' </summary>
    Public NotInheritable Class GifFrame
        Implements IDisposable
        ''' <summary>
        ''' Gets the full image for this frame.
        ''' </summary>
        Public Property Image() As Bitmap
            Get
                Return m_Image
            End Get
            Private Set(value As Bitmap)
                m_Image = Value
            End Set
        End Property
        Private m_Image As Bitmap
        ''' <summary>
        ''' Gets the duration, in milliseconds, that this frame should be displayed.
        ''' </summary>
        Public Property Duration() As Integer
            Get
                Return m_Duration
            End Get
            Private Set(value As Integer)
                m_Duration = Value
            End Set
        End Property
        Private m_Duration As Integer
        ''' <summary>
        ''' Gets extended frame information.
        ''' </summary>
        Public Property Info() As ExtendedInfo
            Get
                Return m_Info
            End Get
            Private Set(value As ExtendedInfo)
                m_Info = Value
            End Set
        End Property
        Private m_Info As ExtendedInfo

        ''' <summary>
        ''' Initializes a new instance of the GifFrame class without extended information.
        ''' </summary>
        ''' <param name="frame">The bitmap for this frame.</param>
        ''' <param name="duration__1">The duration of this frame.</param>
        Friend Sub New(frame As Bitmap, duration__1 As Integer)
            Image = frame
            Duration = duration__1
        End Sub

        ''' <summary>
        ''' Initializes a new instance of the GifFrame class with extended information.
        ''' </summary>
        ''' <param name="frame">The bitmap for this frame.</param>
        ''' <param name="duration__1">The duration of this frame.</param>
        ''' <param name="subframeBitmap">The bitmap of the original subframe that was decoded.</param>
        ''' <param name="descriptor">ImageDescriptor for this frame.</param>
        ''' <param name="graphicControl">GraphicControlExtension for this frame.</param>
        Friend Sub New(frame As Bitmap, duration__1 As Integer, subframeBitmap As Bitmap, descriptor As GifImageDescriptor, graphicControl As GifGraphicControlExtension)
            Image = frame
            Duration = duration__1
            Info = New ExtendedInfo(subframeBitmap, descriptor, graphicControl)
        End Sub

        ''' <summary>
        ''' Provides extended frame information from the decoding of the gif.
        ''' </summary>
        Public NotInheritable Class ExtendedInfo
            Implements IDisposable
            ''' <summary>
            ''' Gets the original subframe that was decoded.
            ''' This is combined with the previous frames to get the final frame output.
            ''' </summary>
            Public Property SubframeImage() As Bitmap
                Get
                    Return m_SubframeImage
                End Get
                Private Set(value As Bitmap)
                    m_SubframeImage = Value
                End Set
            End Property
            Private m_SubframeImage As Bitmap
            ''' <summary>
            ''' Gets the description of the subframe, including its location and information about the local color table.
            ''' </summary>
            Public Property ImageDescriptor() As GifImageDescriptor
                Get
                    Return m_ImageDescriptor
                End Get
                Private Set(value As GifImageDescriptor)
                    m_ImageDescriptor = Value
                End Set
            End Property
            Private m_ImageDescriptor As GifImageDescriptor
            ''' <summary>
            ''' Gets the graphic control information, which may be null, describing transparency and how subframes are combined with
            ''' frames.
            ''' </summary>
            Public Property GraphicControl() As GifGraphicControlExtension
                Get
                    Return m_GraphicControl
                End Get
                Private Set(value As GifGraphicControlExtension)
                    m_GraphicControl = Value
                End Set
            End Property
            Private m_GraphicControl As GifGraphicControlExtension

            ''' <summary>
            ''' Initializes a new instance of the ExtendedInfo class from the given attributes.
            ''' </summary>
            ''' <param name="subframe">The bitmap of the original subframe that was decoded.</param>
            ''' <param name="descriptor">ImageDescriptor for this frame.</param>
            ''' <param name="graphicControl__1">GraphicControlExtension for this frame.</param>
            Friend Sub New(subframe As Bitmap, descriptor As GifImageDescriptor, graphicControl__1 As GifGraphicControlExtension)
                SubframeImage = subframe
                ImageDescriptor = descriptor
                GraphicControl = graphicControl__1
            End Sub

            ''' <summary>
            ''' Releases all resources used by this object.
            ''' </summary>
            Public Sub Dispose() Implements IDisposable.Dispose
                SubframeImage.Dispose()
            End Sub
        End Class

        ''' <summary>
        ''' Releases all resources used by this object.
        ''' </summary>
        Public Sub Dispose() Implements IDisposable.Dispose
            Image.Dispose()
            If Info IsNot Nothing Then
                Info.Dispose()
            End If
        End Sub
    End Class
#End Region

#Region "GifLogicalScreenDescriptor class"
    ''' <summary>
    ''' Provides a description of the image dimensions and global color table.
    ''' </summary>
    Public Class GifLogicalScreenDescriptor
        ''' <summary>
        ''' The dimensions, in pixels, of the logical screen.
        ''' </summary>
        Private ReadOnly m_size As Size

        ''' <summary>
        ''' Gets the dimensions, in pixels, of the logical screen.
        ''' </summary>
        Public ReadOnly Property Size() As Size
            Get
                Return m_size
            End Get
        End Property
        ''' <summary>
        ''' Gets the width, in pixels, of the logical screen.
        ''' </summary>
        Public ReadOnly Property Width() As Integer
            Get
                Return m_size.Width
            End Get
        End Property
        ''' <summary>
        ''' Gets the height, in pixels, of the logical screen.
        ''' </summary>
        Public ReadOnly Property Height() As Integer
            Get
                Return m_size.Height
            End Get
        End Property
        ''' <summary>
        ''' Gets the area, in pixels, of the logical screen.
        ''' </summary>
        Public ReadOnly Property Area() As Integer
            Get
                Return Width * Height
            End Get
        End Property
        ''' <summary>
        ''' Gets a value indicating whether a global color table exists.
        ''' If true, the BackgroundIndex property is meaningful.
        ''' </summary>
        Public Property GlobalTableExists() As Boolean
            Get
                Return m_GlobalTableExists
            End Get
            Private Set(value As Boolean)
                m_GlobalTableExists = Value
            End Set
        End Property
        Private m_GlobalTableExists As Boolean
        ''' <summary>
        ''' Gets the number of bits available for each color in the original image.
        ''' </summary>
        Public Property OriginalBitsPerColor() As Byte
            Get
                Return m_OriginalBitsPerColor
            End Get
            Private Set(value As Byte)
                m_OriginalBitsPerColor = Value
            End Set
        End Property
        Private m_OriginalBitsPerColor As Byte
        ''' <summary>
        ''' Gets a value indicating whether the global color table is sorted.
        ''' If true, colors are listed in decreasing order of importance.
        ''' Otherwise, no ordering is defined.
        ''' </summary>
        Public Property GlobalTableSorted() As Boolean
            Get
                Return m_GlobalTableSorted
            End Get
            Private Set(value As Boolean)
                m_GlobalTableSorted = Value
            End Set
        End Property
        Private m_GlobalTableSorted As Boolean
        ''' <summary>
        ''' Gets the number of colors in the global color table.
        ''' Colors are 24bbp RGB values.
        ''' </summary>
        Public Property GlobalTableSize() As Integer
            Get
                Return m_GlobalTableSize
            End Get
            Private Set(value As Integer)
                m_GlobalTableSize = Value
            End Set
        End Property
        Private m_GlobalTableSize As Integer
        ''' <summary>
        ''' Gets the index of the background color in the global color table.
        ''' Only meaningful if GlobalTableExists is true.
        ''' </summary>
        Public Property BackgroundIndex() As Byte
            Get
                Return m_BackgroundIndex
            End Get
            Private Set(value As Byte)
                m_BackgroundIndex = Value
            End Set
        End Property
        Private m_BackgroundIndex As Byte
        ''' <summary>
        ''' Gets an approximation of the aspect ratio of the original image.
        ''' If null, no information exists.
        ''' </summary>
        Public Property ApproximateAspectRatio() As System.Nullable(Of Single)
            Get
                Return m_ApproximateAspectRatio
            End Get
            Private Set(value As System.Nullable(Of Single))
                m_ApproximateAspectRatio = Value
            End Set
        End Property
        Private m_ApproximateAspectRatio As System.Nullable(Of Single)

        ''' <summary>
        ''' Initializes a new instance of the GifLogicalScreenDescriptor class with given parameters.
        ''' </summary>
        ''' <param name="logicalScreenWidth">The width of the logical screen.</param>
        ''' <param name="logicalScreenHeight">The height of the logical screen.</param>
        ''' <param name="globalColorTableFlag">Indicates if a global color table is present.</param>
        ''' <param name="colorResolution">The original color resolution of the image.</param>
        ''' <param name="sortFlag">Indicates if the global color table is sorted.</param>
        ''' <param name="sizeOfGlobalColorTable">The number of colors in the global color table.</param>
        ''' <param name="backgroundColorIndex">The index of the background color in the global color table.</param>
        ''' <param name="pixelAspectRatio">The aspect ratio of the pixel dimensions of the image.</param>
        Friend Sub New(logicalScreenWidth As UShort, logicalScreenHeight As UShort, globalColorTableFlag As Boolean, colorResolution As Byte, sortFlag As Boolean, sizeOfGlobalColorTable As Integer, _
            backgroundColorIndex As Byte, pixelAspectRatio As Byte)
            m_size = New Size(logicalScreenWidth, logicalScreenHeight)
            GlobalTableExists = globalColorTableFlag
            OriginalBitsPerColor = colorResolution
            GlobalTableSorted = sortFlag
            GlobalTableSize = sizeOfGlobalColorTable
            BackgroundIndex = backgroundColorIndex
            If pixelAspectRatio = 0 Then
                ApproximateAspectRatio = Nothing
            Else
                ApproximateAspectRatio = CSng(pixelAspectRatio + 15) / 64.0F
            End If
        End Sub
    End Class
#End Region

#Region "GifImageDescriptor class"
    ''' <summary>
    ''' Provides a description of the subframe location and dimensions, and the local color table.
    ''' </summary>
    Public Class GifImageDescriptor
        ''' <summary>
        ''' The location and dimensions, in pixels, of the subframe area.
        ''' </summary>
        Private ReadOnly image As Rectangle

        ''' <summary>
        ''' Gets the location and dimensions, in pixels, of the subframe area.
        ''' </summary>
        Public ReadOnly Property Subframe() As Rectangle
            Get
                Return image
            End Get
        End Property
        ''' <summary>
        ''' Gets a value indicating whether a local color table exists.
        ''' </summary>
        Public Property LocalTableExists() As Boolean
            Get
                Return m_LocalTableExists
            End Get
            Private Set(value As Boolean)
                m_LocalTableExists = Value
            End Set
        End Property
        Private m_LocalTableExists As Boolean
        ''' <summary>
        ''' Gets a value indicating whether the subframe image is interlaced in a four-pass interlace pattern.
        ''' </summary>
        Public Property Interlaced() As Boolean
            Get
                Return m_Interlaced
            End Get
            Private Set(value As Boolean)
                m_Interlaced = Value
            End Set
        End Property
        Private m_Interlaced As Boolean
        ''' <summary>
        ''' Gets a value indicating whether the local color table is sorted.
        ''' If true, colors are listed in decreasing order of importance.
        ''' Otherwise, no ordering is defined.
        ''' </summary>
        Public Property LocalTableSorted() As Boolean
            Get
                Return m_LocalTableSorted
            End Get
            Private Set(value As Boolean)
                m_LocalTableSorted = Value
            End Set
        End Property
        Private m_LocalTableSorted As Boolean
        ''' <summary>
        ''' Gets the number of colors in the local color table.
        ''' Colors are 24bbp RGB values.
        ''' </summary>
        Public Property LocalTableSize() As Integer
            Get
                Return m_LocalTableSize
            End Get
            Private Set(value As Integer)
                m_LocalTableSize = Value
            End Set
        End Property
        Private m_LocalTableSize As Integer

        ''' <summary>
        ''' Initializes a new instance of the GifImageDescriptor class with given parameters.
        ''' </summary>
        ''' <param name="imageLeftPosition">The left position of the subframe.</param>
        ''' <param name="imageTopPosition">The top position of the subframe.</param>
        ''' <param name="imageWidth">The width of the subframe.</param>
        ''' <param name="imageHeight">The height of the subframe.</param>
        ''' <param name="localColorTableFlag">Indicates if a local color table is present.</param>
        ''' <param name="interlaceFlag">Indicates if the image is interlaced.</param>
        ''' <param name="sortFlag">Indicates if the local color table is sorted.</param>
        ''' <param name="sizeOfLocalColorTable">The number of colors in the local color table.</param>
        Friend Sub New(imageLeftPosition As UShort, imageTopPosition As UShort, imageWidth As UShort, imageHeight As UShort, localColorTableFlag As Boolean, interlaceFlag As Boolean, _
            sortFlag As Boolean, sizeOfLocalColorTable As Integer)
            image = New Rectangle(imageLeftPosition, imageTopPosition, imageWidth, imageHeight)
            LocalTableExists = localColorTableFlag
            Interlaced = interlaceFlag
            LocalTableSorted = sortFlag
            LocalTableSize = sizeOfLocalColorTable
        End Sub
    End Class
#End Region

#Region "GifGraphicControlExtension class"
    ''' <summary>
    ''' Provides a description of how subframes should be layered, the use of transparency and frame delay.
    ''' </summary>
    Public Class GifGraphicControlExtension
        ''' <summary>
        ''' Gets the method by which pixels should be overwritten.
        ''' </summary>
        Public Property DisposalMethod() As GifDisposalMethod
            Get
                Return m_DisposalMethod
            End Get
            Private Set(value As GifDisposalMethod)
                m_DisposalMethod = Value
            End Set
        End Property
        Private m_DisposalMethod As GifDisposalMethod
        ''' <summary>
        ''' Gets a value indicating whether user input is expected.
        ''' </summary>
        Public Property UserInputExpected() As Boolean
            Get
                Return m_UserInputExpected
            End Get
            Private Set(value As Boolean)
                m_UserInputExpected = Value
            End Set
        End Property
        Private m_UserInputExpected As Boolean
        ''' <summary>
        ''' Gets a value indicating whether transparency is used.
        ''' If true, indexes matching the TransparentIndex should be ignored.
        ''' </summary>
        Public Property TransparencyUsed() As Boolean
            Get
                Return m_TransparencyUsed
            End Get
            Private Set(value As Boolean)
                m_TransparencyUsed = Value
            End Set
        End Property
        Private m_TransparencyUsed As Boolean
        ''' <summary>
        ''' Gets the delay for this frame, in milliseconds, which indicates how much time should pass before the next frame is rendered.
        ''' </summary>
        Public Property Delay() As Integer
            Get
                Return m_Delay
            End Get
            Private Set(value As Integer)
                m_Delay = Value
            End Set
        End Property
        Private m_Delay As Integer
        ''' <summary>
        ''' Gets the index of the transparent color in the color table.
        ''' Only meaningful if TransparencyUsed is true.
        ''' </summary>
        Public Property TransparentIndex() As Byte
            Get
                Return m_TransparentIndex
            End Get
            Private Set(value As Byte)
                m_TransparentIndex = Value
            End Set
        End Property
        Private m_TransparentIndex As Byte

        ''' <summary>
        ''' Initializes a new instance of the GifGraphicControlExtension class with given parameters.
        ''' </summary>
        ''' <param name="disposalMethod__1">The DisposalMethod to be used after this subframe is rendered.</param>
        ''' <param name="userInputFlag">Indicates if user input is expected.</param>
        ''' <param name="transparencyFlag">Indicates if transparency is used.
        ''' If true the transparentColorIndex should be treated as transparent.</param>
        ''' <param name="delayTime">The delay, in hundredths of a second, before the next frame should be rendered.</param>
        ''' <param name="transparentColorIndex">The index of the color to treat as transparent, if the transparencyFlag is set.</param>
        Friend Sub New(disposalMethod__1 As GifDisposalMethod, userInputFlag As Boolean, transparencyFlag As Boolean, delayTime As UShort, transparentColorIndex As Byte)
            DisposalMethod = disposalMethod__1
            UserInputExpected = userInputFlag
            TransparencyUsed = transparencyFlag
            Delay = delayTime * 10
            TransparentIndex = transparentColorIndex
        End Sub
    End Class
#End Region

#Region "GifDisposalMethod enum"
    ''' <summary>
    ''' Defines how pixels are to be overwritten.
    ''' </summary>
    Public Enum GifDisposalMethod
        ''' <summary>
        ''' Undefined and should not be used. If encountered, treat as DoNotDispose.
        ''' </summary>
        Undefined = 0
        ''' <summary>
        ''' Keep the current buffer, later frames will layer above it.
        ''' </summary>
        DoNotDispose = 1
        ''' <summary>
        ''' Clear the background within the sub-frame that was just rendered.
        ''' </summary>
        RestoreBackground = 2
        ''' <summary>
        ''' Reset the background within the sub-frame that was just rendered to its previous values.
        ''' </summary>
        RestorePrevious = 3
    End Enum
#End Region
End Module
