﻿Module Game_m

    Class Game

        Enum BallType

            Soccer = 0 'Is pushed around and slows to a stop.  No Gravity
            Baseball = 1 'Is thrown and arcs, then slows if not caught.
            PingPong = 2 'Bounces around and doesn't stop/slow
            Dodge = 3 'Travels in a straight line and is destroyed when impacting barriers

        End Enum

        Enum GameStatus

            Setup = 0
            Ready = 1
            InProgress = 2
            '   Round_Finishing = 3
            'next_Round_Setup = 4
            'Finishing = 5
            Completed = 6

        End Enum

        Enum ScoreStyle

            Ball_at_Goal = 0
            Ball_Hits_Other_Team = 1
            Ball_Destroyed = 2
            Ball_At_Sides = 3

        End Enum

        Enum PlayerActionType

            Not_Set = 0
            Return_To_Start = 1
            Chase_Ball = 2
            Avoid_Ball = 3
            Throw_Ball_ToGoal = 4
            Throw_Ball_ToTeammate = 5
            Throw_Ball_AtTarget = 6
            Throw_Ball_Reflect = 7 'ball bounces off
            Approach_Own_Goal = 8
            Approach_Target_Goal = 9
            Lead_Ball = 10
            Approach_Target = 12
            Idle = 13

        End Enum

        Public Name As String = ""
        Friend Description As String = ""
        Dim files_path As String = ""

        Dim MinTeams As Integer = 2
        Dim MaxTeams As Integer = 2
        Friend Teams As New List(Of Team)
        Dim AllPlayers As New List(Of Position)

        Friend Balls As New List(Of Ball)

        Friend Active_Balls As New List(Of Ball)

        Dim MinBalls As Integer
        Dim MaxBalls As Integer

        Friend Status As GameStatus
        Friend MaxScore As Integer

        Dim WinningTeam As Team

        Friend ScoreBoard As Game_ScoreBoard
        Dim ScoreBoard_Location As Point
        Dim ScoringStyles As New List(Of ScoreStyle)

        Friend GameScreen As Screen = Nothing

        Friend Goals As New List(Of Goal_Area)

        Sub New(ByVal config_file_path As String, ByVal files_path As String)

            Dim config_file As System.IO.StreamReader
            config_file = New System.IO.StreamReader(config_file_path)

            Dim position_data As New List(Of String)
            Dim game_data As String = Nothing
            Dim ball_data As New List(Of String)
            Dim goal_data As New List(Of String)
            Dim description_data As String = ""

            Do Until config_file.EndOfStream

                Dim line = config_file.ReadLine

                'ignore blank or 'commented out' lines.
                If line = "" OrElse line(0) = "'" Then
                    Continue Do
                End If

                'SplitWithQualifiers is an improved split() that lets use use, as the name says, text qualifiers.
                'This lets you embed two different lists of items, or handle names that contain commas.
                Dim columns = SplitWithQualifiers(line, ",", ControlChars.Quote)

                If UBound(columns) < 1 Then
                    Continue Do
                End If

                Select Case LCase(columns(0))

                    Case "game"
                        game_data = line
                    Case "description"
                        description_data = line
                    Case "position"
                        position_data.Add(line)
                    Case "ball"
                        ball_data.Add(line)
                    Case "goal"
                        goal_data.Add(line)
                    Case "scoreboard"
                        ScoreBoard = New Game_ScoreBoard(columns(1) & "," & columns(2), files_path & Trim(columns(3).Replace(ControlChars.Quote, "")))
                    Case Else
                        Throw New Exception("Invalid line in config file: " & line)
                End Select

            Loop

            If IsNothing(game_data) OrElse position_data.Count = 0 OrElse ball_data.Count = 0 Then
                Throw New Exception("Game ini file does not define Game, Position, or Ball data.  It must contain all 3.")
            End If

            'process game data

            Dim game_columns = SplitWithQualifiers(game_data, ",", "{", "}")

            Name = game_columns(1).Replace(ControlChars.Quote, "")

            Dim description_columns = SplitWithQualifiers(description_data, ",", ControlChars.Quote)

            Try
                Description = description_columns(1)
            Catch ex As Exception
                MsgBox("Invalid description line for game: " & Name & " . Are you missing quotes around the text?")
            End Try


            MinBalls = game_columns(3)
            MaxBalls = game_columns(4)
            MaxScore = game_columns(5)
            If MaxScore < 1 Then
                Throw New Exception("The maximum score must be at least 1 - error loading name " & Name)
            End If
            If MaxBalls < MinBalls Then Throw New Exception("Minimum number of balls in play is larger than the Maximum setting defined for game: " & Name)

            If MinTeams < 2 Then Throw New Exception("You must have at least two teams for a game.  The minimum setting is too low for game: " & Name)
            'maybe later we can have them play tag or a zombie game... but for now:
            If MinBalls < 1 Then Throw New Exception("You must have at least one ball for the ponies to play with.  The minimum setting is too low for game: " & Name)

            'do scoring styles
            Dim ScoringStyles_list = Split(game_columns(6))
            For Each style In ScoringStyles_list
                Select Case LCase(style)
                    Case "ball_at_goal"
                        ScoringStyles.Add(ScoreStyle.Ball_at_Goal)
                    Case "ball_hits_other_team"
                        ScoringStyles.Add(ScoreStyle.Ball_Hits_Other_Team)
                    Case "ball_destroyed"
                        ScoringStyles.Add(ScoreStyle.Ball_Destroyed)
                    Case "ball_at_sides"
                        ScoringStyles.Add(ScoreStyle.Ball_At_Sides)
                    Case Else
                        Throw New Exception("Invalid scoring style: " & style)
                End Select
            Next

            'do teamnames
            Dim TeamNames = SplitWithQualifiers(game_columns(2), ",", ControlChars.Quote)

            Dim number = 1
            For Each line In TeamNames
                Dim new_team As New Team(line, number)
                Teams.Add(new_team)
                number += 1
            Next

            'do goals
            For Each line In goal_data
                Dim columns = SplitWithQualifiers(line, ",", "{", "}")
                Dim new_goal As New Goal_Area(columns(1), files_path & Trim(columns(2).Replace(ControlChars.Quote, "")), columns(3))
                Goals.Add(new_goal)
            Next

            For Each goal As Goal_Area In Goals
                If goal.team_number <> 0 Then
                    Teams(goal.team_number - 1).Goal = goal
                End If
            Next

            'do positions
            For Each line In position_data
                Dim columns = SplitWithQualifiers(line, ",", "{", "}")

                Dim new_position As New Position(columns(1), columns(2), columns(3), columns(4), columns(5), columns(6), _
                                                 columns(7), columns(8), columns(9), columns(10), columns(11))

                Teams(columns(2) - 1).Positions.Add(new_position)
            Next

            For Each line In ball_data
                Dim columns = SplitWithQualifiers(line, ",", ControlChars.Quote)

                Dim new_ball As New Ball(columns(1), columns(2), columns(3), columns(4), columns(5), columns(6), columns(7), columns(8), files_path)
                Balls.Add(new_ball)
            Next

            Status = GameStatus.Setup

        End Sub

        Friend Sub CleanUp()

            For Each Ball In Balls
                '  Ball.Handler.Close()
                Active_Balls.Remove(Ball)
            Next

            My.Forms.Main.Active_Ponies.Clear()

            'For Each goal In Goals
            '    goal.Visible = False
            'Next

            'ScoreBoard.Visible = False

            Teams(0).Score = 0
            Teams(1).Score = 0

            'For Each position As Position In AllPlayers
            '    '  position.Player.Close()
            'Next

            AllPlayers.Clear()

        End Sub

        Friend Sub Setup()


            If My.Forms.Options.Window_Avoidance_Enabled.Checked OrElse My.Forms.Options.Cursor_Avoidance_Enabled.Checked Then
                My.Forms.Options.Window_Avoidance_Enabled.Checked = False
                My.Forms.Options.Cursor_Avoidance_Enabled.Checked = False
                '  MsgBox("Note:  Window avoidance and cursor avoidance have been disabled as they may interfere with the game.")
            End If


            For Each Goal As Goal_Area In Goals
                Goal.Initialize(GameScreen)
            Next

            ScoreBoard.Initialize(GameScreen)
            ScoreBoard.SetScore(Teams(0).Name, Teams(0).Score, Teams(1).Name, Teams(1).Score)

            For Each Team As Team In Teams
                Dim positions_to_remove As New List(Of Position)
                For Each Position As Position In Team.Positions
                    Position.Initialize(GameScreen)
                    If Not IsNothing(Position.Player) Then

                        If IsNothing(Position.Player.Current_Screen) Then
                            Position.Player.Current_Screen = GameScreen
                        End If
                        Position.Player.Playing_Game = True
                        Main.Active_Ponies.Add(Position.Player)
                        AllPlayers.Add(Position)
                    Else
                        positions_to_remove.Add(Position)
                    End If
                Next
                For Each entry In positions_to_remove
                    Team.Positions.Remove(entry)
                Next
            Next

            For Each Ball In Balls
                Ball.Initialize(GameScreen)
            Next

            Dim monitor = GameScreen
            Main.screens_to_use.Clear()
            Main.screens_to_use.Add(monitor)

            If My.Forms.Main.PonyScale <> 1 Then
                MsgBox("Note:  Games may not work properly with the scale option set to values other than 1...  You are currently playing with scale " & My.Forms.Main.PonyScale & "x.")
            End If

        End Sub


        Friend Sub Update()

            Select Case Status

                Case GameStatus.Setup

                    Dim all_in_position = True

                    For Each Team As Team In Teams
                        For Each Position As Position In Team.Positions

                            If IsNothing(Position.Current_Action) OrElse Position.Current_Action <> PlayerActionType.Return_To_Start _
                                OrElse Position.Player.current_behavior.Allowed_Movement = Pony.Allowed_Moves.None Then
                                Position.SetFollowBehavior(Nothing, Nothing, True) 'go to starting position
                            End If

                            If Position.Player.AtDestination = False Then
                                all_in_position = False
                            End If

                            Position.Player.Update(My.Forms.Main.totalElaspedTime)

                        Next
                    Next

                    If all_in_position Then
                        Status = GameStatus.Ready
                    End If

                Case GameStatus.Ready

                    'For Each Ball In Balls
                    '    Ball.Handler.Visible = False
                    'Next

                    For Each position As Position In AllPlayers
                        position.Current_Action = Nothing
                        position.Current_Action_Group = Nothing
                    Next

                    For Each Ball In Balls
                        Active_Balls.Add(Ball)
                        Ball.Handler.current_behavior = Ball.Handler.GetAppropriateBehavior(Pony.Allowed_Moves.None, False)
                        Ball.Handler.location = Ball.StartPosition
                        Ball.Handler.Precise_X_Location = Ball.StartPosition.X
                        Ball.Handler.Precise_Y_Location = Ball.StartPosition.Y
                        'Ball.Handler.Visible = True
                        My.Forms.Main.Active_Ponies.Add(Ball.Handler)
                        Ball.Update()
                        If Ball.Type = BallType.PingPong Then
                            Ball.Kick(5, Rnd() * (2 * Math.PI), Nothing)
                        End If
                        If Active_Balls.Count >= MinBalls Then Exit For
                    Next

                    Status = GameStatus.InProgress

                Case GameStatus.InProgress

                    For Each Team As Team In Teams
                        For Each Position As Position In Team.Positions
                            Position.Decide_On_Action(My.Forms.Main.current_game)
                            Position.PushBackOverlappingPonies(AllPlayers)
                            Position.Player.Update(My.Forms.Main.totalElaspedTime)
                        Next
                    Next

                    For Each Ball In Active_Balls
                        Ball.Update()
                    Next

                    If CheckForScore() Then

                        For Each Ball In Balls
                            Active_Balls.Remove(Ball)
                            My.Forms.Main.Active_Ponies.Remove(Ball.Handler)
                        Next

                        For Each Team In Teams
                            If Team.Score >= MaxScore Then
                                Status = GameStatus.Completed
                                My.Forms.Main.MoveTimer.Enabled = False
                                MsgBox(Team.Name & " won!")
                                My.Forms.Main.Pony_Shutdown(True)
                                My.Forms.Main.Visible = True
                                Exit Sub
                            End If
                        Next

                        Status = GameStatus.Setup
                    End If


                Case Else
                    Throw New Exception("State not implemented: " & Status)

            End Select

        End Sub

        Function Get_Ball_LastHandler_Team(ByVal ball As Ball) As Team

            If IsNothing(ball.Last_Handled_By) Then Return Nothing

            Return ball.Last_Handled_By.Team

        End Function

        Function Get_Team_By_Player(ByVal Player As Pony) As Team

            For Each Team In Teams
                For Each Position In Team.Positions
                    If ReferenceEquals(Position.Player, Player) Then
                        Return Team
                    End If
                Next
            Next

            Throw New Exception("Player not found when searching team: " & Player.Name)

        End Function

        Function CheckForScore() As Boolean

            For Each ScoreStyle As ScoreStyle In My.Forms.Main.current_game.ScoringStyles

                For Each Ball In Balls

                    Select Case ScoreStyle
                        Case ScoreStyle.Ball_At_Sides
                            If Ball.Handler.location.X < GameScreen.WorkingArea.X + (GameScreen.WorkingArea.Width * 0.02) Then
                                Teams(1).Score += 1
                                ScoreBoard.SetScore(Teams(0).Name, Teams(0).Score, Teams(1).Name, Teams(1).Score)
                                Return True
                            Else
                                If Ball.Handler.location.X + Ball.Handler.current_behavior.current_image.Size.Width > (GameScreen.WorkingArea.X + GameScreen.WorkingArea.Width) - (GameScreen.WorkingArea.Width * 0.02) Then
                                    Teams(0).Score += 1
                                    ScoreBoard.SetScore(Teams(0).Name, Teams(0).Score, Teams(1).Name, Teams(1).Score)
                                    Return True
                                End If
                            End If

                        Case ScoreStyle.Ball_at_Goal

                            For Each goal In Goals
                                Dim goal_area As New Pony.RECT
                                goal_area.Top = goal.form.Location.Y
                                goal_area.Right = goal.form.location.X + goal.form.current_image.Size.Width
                                goal_area.Bottom = goal.form.location.Y + goal.form.current_image.Size.Height
                                goal_area.Left = goal.form.Location.X
                                If Ball.Handler.IsPonyInBox(Ball.Handler.Center, goal_area) Then
                                    For Each Team In Teams
                                        If ReferenceEquals(Team.Goal, goal) AndAlso Not ReferenceEquals(Team, Ball.Last_Handled_By.Team) Then
                                            For Each OtherTeam In Teams
                                                If Not ReferenceEquals(OtherTeam, Team) Then
                                                    OtherTeam.Score += 1
                                                    ScoreBoard.SetScore(Teams(0).Name, Teams(0).Score, Teams(1).Name, Teams(1).Score)
                                                    Return True
                                                End If
                                            Next
                                        End If
                                    Next
                                End If
                            Next

                    End Select
                Next
            Next

            Return False


        End Function

        Class Ball

            Friend Type As BallType
            Friend StartPosition As Point
            Dim Initial_Position As Point
            Friend friction As Double = 0.992
            Friend Last_Handled_By As Position = Nothing

            Friend Handler As Pony 'the ball is a pony type that move like a pony


            Sub New(ByVal _type As String, ByVal idle_image_filename As String, ByVal slow_right_image_filename As String, ByVal slow_left_image_filename As String,
                    ByVal fast_right_image_filename As String, ByVal fast_left_image_filename As String, ByVal x_location As Integer, ByVal y_location As Integer, ByVal files_path As String)

                'We need to duplicate the new pony as only "duplicates" are fully loaded.  A new pony by itself is considered a template.
                Handler = New Pony("ball", New List(Of Pony.Behavior.Speaking_Line), "", 1, -1, New List(Of Pony.BehaviorGroup)).Duplicate

                Handler.Add_Behavior(True, "idle", 100, 99, 99, 0, files_path & Replace(idle_image_filename, ControlChars.Quote, ""), files_path & Replace(idle_image_filename, ControlChars.Quote, ""), _
                                     Pony.Allowed_Moves.None, "", "", "")
                Handler.Add_Behavior(True, "slow", 100, 99, 99, 3, files_path & Replace(slow_right_image_filename, ControlChars.Quote, ""), files_path & Replace(slow_left_image_filename, ControlChars.Quote, ""), _
                          Pony.Allowed_Moves.All, "", "", "")
                Handler.Add_Behavior(True, "fast", 100, 99, 99, 5, files_path & Replace(fast_right_image_filename, ControlChars.Quote, ""), files_path & Replace(fast_left_image_filename, ControlChars.Quote, ""), _
                          Pony.Allowed_Moves.All, "", "", "")

                Initial_Position = New Point(x_location, y_location)

                Select Case LCase(Trim(_type))

                    Case "soccer"
                        Type = BallType.Soccer
                    Case "baseball"
                        Type = BallType.Baseball
                    Case "pingpong"
                        Type = BallType.PingPong
                    Case "dodge"
                        Type = BallType.Dodge
                    Case Else
                        Throw New Exception("Invalid ball type: " & Type)

                End Select

            End Sub

            Sub Initialize(ByVal gamescreen As Screen)

                StartPosition = New Point(Initial_Position.X * 0.01 * gamescreen.WorkingArea.Width + gamescreen.WorkingArea.X, _
                                   Initial_Position.Y * 0.01 * gamescreen.WorkingArea.Height + gamescreen.WorkingArea.Y)

                Handler.Location = StartPosition
                Handler.Precise_X_Location = StartPosition.X
                Handler.Precise_Y_Location = StartPosition.Y
            End Sub

            Friend Function Center() As Point
                Return New Point(Me.Handler.location.X + (Me.Handler.current_behavior.current_image.Size.Width / 2), Me.Handler.location.Y + (Me.Handler.current_behavior.current_image.Size.Height) / 2)
            End Function

            Sub Update()

                Dim speed As Double = 0
                Dim up = True
                Dim right = True
                Dim angle As Double = 0

                Select Case Type
                    Case BallType.Soccer
                        Handler.current_behavior.speed *= friction
                        speed = Handler.current_behavior.speed

                    Case BallType.PingPong
                        speed = Handler.current_behavior.speed
                End Select

                up = Handler.current_behavior.up
                right = Handler.current_behavior.right
                angle = Handler.current_behavior.diagonal

                Select Case Handler.current_behavior.speed
                    Case Is < 0.1
                        speed = 0
                        Handler.current_behavior = Handler.Behaviors(0)
                    Case Is < 3
                        Handler.current_behavior = Handler.Behaviors(1)
                    Case Is > 3
                        Handler.current_behavior = Handler.Behaviors(2)
                End Select

                Handler.current_behavior.speed = speed
                Handler.current_behavior.up = up
                Handler.current_behavior.right = right
                Handler.current_behavior.diagonal = angle
                Handler.Move()

            End Sub

            Friend Sub Kick(ByVal _speed As Double, ByVal _angle As Double, ByVal kicker As Position)
                Last_Handled_By = kicker

                Handler.current_behavior = Handler.GetAppropriateBehavior(Pony.Allowed_Moves.All, True)
                Handler.current_behavior.speed = _speed
                Handler.current_behavior.diagonal = _angle

            End Sub

        End Class


        Class Team

            Friend Name As String
            Friend Number As Integer
            Friend Positions As New List(Of Position)
            Friend Score As Integer = 0
            Friend Goal As Goal_Area = Nothing

            Sub New(ByVal _name As String, ByVal _number As Integer)
                Name = _name
                Number = _number
            End Sub

        End Class

        Class Goal_Area

            Friend form As Pony.Behavior.effect
            Friend team_number As Integer ' 0 = a score any team
            Dim start_point As Point = New Point
            Dim image As Image

            Sub New(ByVal _team_number As Integer, ByVal image_filename As String, ByVal location As String)

                team_number = _team_number
                form = New Pony.Behavior.effect
                form.name = "Team " & team_number & "'s Goal"
                If Not My.Computer.FileSystem.FileExists(image_filename) Then Throw New Exception("File does not exist: " & image_filename)
                form.right_image_path = image_filename
                form.left_image_path = image_filename

                Dim location_parts = Split(location, ",")
                start_point = New Point(location_parts(0), location_parts(1))

                image = image.FromFile(image_filename)

                'form.Size = form.Effect_Image.Image.Size
                'form.Effect_Image.Size = form.Effect_Image.Image.Size

            End Sub

            Friend Sub Paint(screengraphics As Graphics, ponyform As PonyGraphicsForm)
                Dim translated_location = Pony.Translate_Current_Location(form.location, ponyform)

                screengraphics.DrawImageUnscaled(Image, translated_location.X, translated_location.Y)
            End Sub

            Sub Initialize(ByVal gamescreen As Screen)
                form.location = New Point(start_point.X * 0.01 * gamescreen.WorkingArea.Width + gamescreen.WorkingArea.X, _
                                   start_point.Y * 0.01 * gamescreen.WorkingArea.Height + gamescreen.WorkingArea.Y)
                form.LoadImages()
                form.current_image = form.Rightanimatedimage
            End Sub

            Function Center() As Point
                Return New Point(form.location.X + (form.current_image.Size.Width / 2), Me.form.location.Y + (Me.form.current_image.Size.Height) / 2)
            End Function

        End Class

        Class Game_ScoreBoard

            Friend form As Pony.Behavior.effect
            Dim graphics As Graphics
            Dim start_point As Point = New Point
            Dim original_image As Image
            Friend Image As Image

            Dim team1 As String = ""
            Dim team2 As String = ""
            Dim team1_score As Integer = 0
            Dim team2_score As Integer = 0

            Sub New(ByVal location As String, ByVal image_filename As String)

                form = New Pony.Behavior.effect
                form.Name = "Scoreboard"
                If Not My.Computer.FileSystem.FileExists(image_filename) Then Throw New Exception("File does not exist: " & image_filename)
            
                Dim location_parts = Split(location, ",")
                start_point = New Point(location_parts(0), location_parts(1))

                original_image = Image.FromFile(image_filename)

            End Sub

            Sub Initialize(ByVal gamescreen As Screen)
                form.Location = New Point(start_point.X * 0.01 * gamescreen.WorkingArea.Width + gamescreen.WorkingArea.X, _
                                   start_point.Y * 0.01 * gamescreen.WorkingArea.Height + gamescreen.WorkingArea.Y)
            End Sub

            Function Center() As Point
                Return New Point(Me.form.location.X + (form.current_image.Size.Width / 2), Me.form.location.Y + (form.current_image.Size.Height) / 2)
            End Function

            Sub SetScore(ByVal _team1 As String, ByVal _team1_score As Integer, ByVal _team2 As String, ByVal _team2_score As Integer)
                team1 = _team1
                team1_score = _team1_score
                team2 = _team2
                team2_score = _team2_score
            End Sub

            Sub Paint(screengraphics As Graphics, ponyform As PonyGraphicsForm)

                Image = original_image.Clone()
                graphics = graphics.FromImage(Image)

                TextRenderer.DrawText(graphics, team1, New Font("Ariel", 8), New Rectangle(New Point(30, 82), New Size(75, 35)), Color.White)
                TextRenderer.DrawText(graphics, team2, New Font("Ariel", 8), New Rectangle(New Point(30, 126), New Size(75, 35)), Color.White)

                TextRenderer.DrawText(graphics, team1_score, New Font("Ariel", 8, FontStyle.Bold), New Rectangle(New Point(95, 85), New Size(75, 35)), Color.White)
                TextRenderer.DrawText(graphics, team2_score, New Font("Ariel", 8, FontStyle.Bold), New Rectangle(New Point(95, 130), New Size(75, 35)), Color.White)

                Dim translated_location = Pony.Translate_Current_Location(form.location, ponyform)

                screengraphics.DrawImageUnscaled(Image, translated_location.X, translated_location.Y)

            End Sub

        End Class

        Class Position

            Friend Name As String
            Friend Team_Number As Integer
            Friend Team As Team
            Friend Player As Pony = Nothing
            Friend Allowed_Area As Pony.RECT = Nothing  'nothing means allowed anywhere
            Friend Start_Location As Point
            Friend Current_Action As PlayerActionType = Nothing
            Friend Current_Action_Group As List(Of PlayerActionType) = Nothing
            Friend Required As Boolean

            Dim area_points As String() = Nothing

            Friend Last_Kick_Time As DateTime = CDate("1/1/1999")

            Friend Hasball As Ball = Nothing

            Friend nearest_ball_distance As Integer = 0

            Friend Selection_Menu_Picturebox As PictureBox = Nothing

            Friend Have_Ball_Actions As New List(Of PlayerActionType)
            Friend Hostile_Ball_Actions As New List(Of PlayerActionType)
            Friend Friendly_Ball_Actions As New List(Of PlayerActionType)
            Friend Neutral_Ball_Actions As New List(Of PlayerActionType)
            Friend Distant_Ball_Actions As New List(Of PlayerActionType)
            Friend No_Ball_Actions As New List(Of PlayerActionType)
            Private _get_Open_Teammate As Object


            Sub New(ByVal _Name As String, ByVal _team_number As Integer, ByVal _start_location As String, ByVal _Allowed_area As String,
                    ByVal _Have_Ball_Actions As String, ByVal _Hostile_Ball_Actions As String, ByVal _Friendly_Ball_Actions As String, _
                    ByVal _Neutral_Ball_Actions As String, ByVal _Distance_Ball_Actions As String, ByVal _No_Ball_Actions As String, ByVal _required As String)

                Name = Trim(_Name.Replace(ControlChars.Quote, ""))
                Team_Number = _team_number

                Select Case LCase(Trim(_required))
                    Case "required"
                        Required = True
                    Case "optional"
                        Required = False
                    Case Else
                        Throw New Exception("Invalid entry for required/optional setting of position " & Name & ". ")
                End Select

                Dim start_points = Split(_start_location, ",")
                Start_Location = New Point(start_points(0), start_points(1))

                If LCase(Trim(_Allowed_area)) <> "any" Then
                    area_points = Split(_Allowed_area, ",")
                Else
                    Allowed_Area = Nothing
                End If

                Dim Action_Lists As New List(Of List(Of PlayerActionType))
                Action_Lists.Add(Have_Ball_Actions)
                Action_Lists.Add(Hostile_Ball_Actions)
                Action_Lists.Add(Friendly_Ball_Actions)
                Action_Lists.Add(Neutral_Ball_Actions)
                Action_Lists.Add(Distant_Ball_Actions)
                Action_Lists.Add(No_Ball_Actions)

                Dim Actions_strings As New List(Of String)
                Actions_strings.Add(_Have_Ball_Actions)
                Actions_strings.Add(_Hostile_Ball_Actions)
                Actions_strings.Add(_Friendly_Ball_Actions)
                Actions_strings.Add(_Neutral_Ball_Actions)
                Actions_strings.Add(_Distance_Ball_Actions)
                Actions_strings.Add(_No_Ball_Actions)

                For i = 0 To Action_Lists.Count - 1
                    Dim list As List(Of PlayerActionType) = Action_Lists(i)
                    Dim action_string As String = Actions_strings(i)

                    Dim actions = Split(action_string, ",")
                    For Each action In actions
                        list.Add(CInt(action))
                    Next
                Next


            End Sub

            Sub Initialize(ByVal gamescreen As Screen)
                If Not IsNothing(area_points) Then
                    Allowed_Area = New Pony.RECT
                    Allowed_Area.Left = area_points(0) * 0.01 * gamescreen.WorkingArea.Width + gamescreen.WorkingArea.X
                    Allowed_Area.Top = area_points(1) * 0.01 * gamescreen.WorkingArea.Height + gamescreen.WorkingArea.Y
                    Allowed_Area.Right = Allowed_Area.Left + (area_points(2) * 0.01 * gamescreen.WorkingArea.Width)
                    Allowed_Area.Bottom = Allowed_Area.Top + (area_points(3) * 0.01 * gamescreen.WorkingArea.Height)
                End If
            End Sub

            Function isBlankRect(ByVal RECT As Pony.RECT) As Boolean

                If RECT.Bottom = 0 AndAlso RECT.Left = 0 AndAlso RECT.Top = 0 AndAlso RECT.Right = 0 Then
                    Return True
                Else
                    Return False
                End If

            End Function


            Sub Decide_On_Action(ByVal game As Game)

                Dim nearest_ball = get_nearest_ball(game.Active_Balls)

                If IsNothing(nearest_ball) Then
                    PerformAction(No_Ball_Actions, Nothing)
                    Exit Sub
                End If

                Dim screen_diagonal = Math.Sqrt((game.GameScreen.Bounds.Height) ^ 2 + (game.GameScreen.Bounds.Width) ^ 2)
                If nearest_ball_distance > screen_diagonal * (1 / 2) Then 'AndAlso _
                    PerformAction(Distant_Ball_Actions, nearest_ball)
                    Exit Sub
                End If

                If nearest_ball_distance <= (Player.current_behavior.current_image.Size.Width / 2) + 50 Then ' / 2 Then
                    PerformAction(Have_Ball_Actions, nearest_ball)
                    Exit Sub
                End If

                Dim BallOwner_Team = game.Get_Ball_LastHandler_Team(nearest_ball)

                If IsNothing(BallOwner_Team) Then
                    PerformAction(Neutral_Ball_Actions, nearest_ball)
                    Exit Sub
                End If

                If BallOwner_Team.Number = Me.Team_Number Then
                    PerformAction(Friendly_Ball_Actions, nearest_ball)
                    Exit Sub
                Else
                    PerformAction(Hostile_Ball_Actions, nearest_ball)
                    Exit Sub
                End If

            End Sub

            Sub PerformAction(ByVal action_list As List(Of PlayerActionType), ByVal ball As Ball)

                If Not IsNothing(Current_Action_Group) Then ' AndAlso Not ReferenceEquals(Current_Action_Group, Have_Ball_Actions) Then
                    If ReferenceEquals(action_list, Current_Action_Group) Then
                        'we are already doing an action from this list

                        'if we recently kicked the ball, don't do it for 2 seconds.
                        If ReferenceEquals(Current_Action_Group, Have_Ball_Actions) Then
                            If DateDiff(DateInterval.Second, Last_Kick_Time, Now()) <= 2 Then
                                If DateDiff(DateInterval.Second, Last_Kick_Time, Now()) > 1 AndAlso (Player.ManualControl_P1 OrElse Player.ManualControl_P2) Then
                                    Speak("Can't kick again so soon!")
                                End If

                                Exit Sub
                            End If
                        Else
                            'if it was any other action, don't reset it.
                            Exit Sub
                        End If
                    Else
                        Current_Action_Group = Nothing
                    End If
                End If

                Dim selection = Math.Round(Rnd() * (action_list.Count - 1))

                Dim selected_action As PlayerActionType = action_list(selection)

                Select Case selected_action

                    Case PlayerActionType.Not_Set
                        Throw New Exception("Can't do this action (reserved): " & selected_action)
                    Case PlayerActionType.Return_To_Start
                        SetFollowBehavior(Nothing, Nothing, True)
                    Case PlayerActionType.Chase_Ball
                        SetFollowBehavior(ball.Handler.Name, ball.Handler)
                    Case PlayerActionType.Lead_Ball
                        SetFollowBehavior(ball.Handler.Name, ball.Handler, False, True)
                    Case PlayerActionType.Avoid_Ball
                        Throw New Exception("Not implemented yet: action type " & selected_action)
                    Case PlayerActionType.Throw_Ball_ToGoal
                        If DateDiff(DateInterval.Second, Last_Kick_Time, Now()) <= 2 Then
                            'can't kick again so soon
                            Exit Sub
                        End If
                        'If ponies are being controlled, don't kick unless the action key (control - left or right) is being pushed
                        If Player.ManualControl_P1 AndAlso Not My.Forms.Main.PonyAction Then Exit Sub
                        If Player.ManualControl_P2 AndAlso Not My.Forms.Main.PonyAction_2 Then Exit Sub
                        Kick_Ball(ball, 10, Get_OtherTeam_Goal(), Nothing, Me, "*Kick*!")
                        Last_Kick_Time = Now()
                    Case PlayerActionType.Throw_Ball_ToTeammate
                        If DateDiff(DateInterval.Second, Last_Kick_Time, Now()) <= 2 Then
                            'can't kick again so soon
                            Exit Sub
                        End If

                        'don't pass if we are under control - kick instead.
                        If Player.ManualControl_P1 Then Exit Sub
                        If Player.ManualControl_P2 Then Exit Sub

                        Dim open_teammate = Get_Open_Teammate(Me.Team, Get_OtherTeam_Goal())
                        If IsNothing(open_teammate) Then
                            'no teammates to pass to, kick to goal instead, unless a player controlled pony.
                            If Player.ManualControl_P1 OrElse Player.ManualControl_P2 Then Exit Sub
                            Kick_Ball(ball, 10, Get_OtherTeam_Goal(), Nothing, Me, "*Kick*!")
                            Last_Kick_Time = Now()
                            Exit Sub
                        End If

                        Kick_Ball(ball, 10, Nothing, open_teammate, Me, "*Pass*!")
                        Last_Kick_Time = Now()

                    Case PlayerActionType.Throw_Ball_Reflect
                        If DateDiff(DateInterval.Second, Last_Kick_Time, Now()) <= 2 Then
                            'can't kick again so soon
                            Exit Sub
                        End If

                        If Player.ManualControl_P1 AndAlso Not My.Forms.Main.PonyAction Then Exit Sub
                        If Player.ManualControl_P2 AndAlso Not My.Forms.Main.PonyAction_2 Then Exit Sub

                        Bounce_Ball(ball, 7, Me, "*Ping*!")
                        Last_Kick_Time = Now()

                    Case PlayerActionType.Approach_Own_Goal
                        Dim goal = Get_Team_Goal()
                        SetFollowBehavior(goal.form.name, goal.form)

                    Case PlayerActionType.Approach_Target_Goal
                        Dim goal = Get_OtherTeam_Goal()
                        SetFollowBehavior(goal.form.name, goal.form)

                    Case PlayerActionType.Idle
                        If Current_Action = PlayerActionType.Idle Then Exit Sub
                        Player.current_behavior.follow_object = Nothing
                        Player.current_behavior.follow_object_name = ""

                        If Player.ManualControl_P1 Then Exit Sub
                        If Player.ManualControl_P2 Then Exit Sub

                        Player.SelectBehavior()
                        SetSpeed()

                    Case Else
                        My.Forms.Main.MoveTimer.Enabled = False
                        MsgBox("Invalid action type: " & selected_action)
                        Throw New Exception("Invalid action type: " & selected_action)
                End Select

                Current_Action = selected_action
                Current_Action_Group = action_list

            End Sub

            Function get_nearest_ball(ByVal balls As List(Of Ball)) As Ball

                Dim me_center_x As Integer = Player.Center.X
                Dim me_center_y As Integer = Player.Center.Y

                Dim nearest_ball As Ball = Nothing
                Dim nearest_ball_distance As Integer = -1

                For Each Ball In balls
                    Dim distance = Math.Sqrt((me_center_x - Ball.Center().X) ^ 2 + (me_center_y - Ball.Center().Y) ^ 2)
                    If distance < nearest_ball_distance OrElse nearest_ball_distance = -1 Then
                        nearest_ball_distance = distance
                        nearest_ball = Ball
                    End If
                Next

                If IsNothing(nearest_ball) Then Throw New Exception("No available balls found when checking distance!")

                Me.nearest_ball_distance = nearest_ball_distance
                Return nearest_ball

            End Function

            Function Get_OtherTeam_Goal() As Goal_Area

                For Each goal In Main.current_game.Goals
                    If goal.team_number <> Me.Team_Number Then
                        Return goal
                    End If
                Next

                Throw New Exception("Couldn't find a goal for another team.")

            End Function

            Function Get_Team_Goal() As Goal_Area

                For Each goal In Main.current_game.Goals
                    If goal.team_number = Me.Team_Number Then
                        Return goal
                    End If
                Next

                Throw New Exception("Couldn't find a goal for pony's team.")

            End Function

            Sub SetFollowBehavior(ByVal target_name As String, ByVal target As Object, Optional ByVal return_to_start As Boolean = False, Optional ByVal lead_target As Boolean = False)

                Player.current_behavior = Player.GetAppropriateBehavior(Pony.Allowed_Moves.All, True)
                Player.current_behavior.follow_object = Nothing
                Player.current_behavior.follow_object_name = ""

                SetSpeed()

                If return_to_start Then
                    Player.current_behavior.destination_xcoord = Start_Location.X
                    Player.current_behavior.destination_ycoord = Start_Location.Y
                    Exit Sub
                Else
                    Player.current_behavior.follow_object_name = target_name
                    Player.current_behavior.follow_object = target
                    Player.current_behavior.destination_xcoord = 0
                    Player.current_behavior.destination_ycoord = 0
                    If lead_target Then
                        Player.current_behavior.lead_target = True
                    End If

                    'If My.Forms.Main.current_game.Name = "Ping Pong Pony" Then
                    '    Player.current_behavior.speed = Player.current_behavior.original_speed * 1.5
                    'End If
                End If
            End Sub

            Sub Kick_Ball(ByVal ball As Ball, ByVal speed As Double, ByVal target_goal As Goal_Area, ByVal target_pony As Pony, ByVal kicker As Position, ByVal line As String)

                If Rnd() < 0.05 Then
                    Speak("Missed!")
                    Exit Sub
                End If

                speed = kicker.Player.GetScale() * speed

                Speak(line)

                Dim angle As Double = Nothing

                If IsNothing(target_goal) Then
                    angle = Get_Angle_To_Object(target_pony.Center)
                Else
                    angle = Get_Angle_To_Object(target_goal.Center)
                End If


                'add a bit of inaccuracy
                If Rnd() > 0.5 Then
                    angle += Rnd() * (Math.PI / 8)
                Else
                    angle -= Rnd() * (Math.PI / 8)
                End If

                ball.Kick(speed, angle, kicker)

            End Sub

            Sub Bounce_Ball(ByVal ball As Ball, ByVal speed As Double, ByVal kicker As Position, ByVal line As String)

                If My.Forms.Main.current_game.Name = "Ping Pong Pony" Then
                    'avoid boucing the ball back into our own goal.
                    If Not IsNothing(ball.Last_Handled_By) AndAlso ReferenceEquals(ball.Last_Handled_By, Me) Then
                        Exit Sub
                    End If
                End If

                Speak(line)

                Dim angle As Double = Nothing
                Dim right As Boolean = True
                Dim gamescreen = My.Forms.Main.current_game.GameScreen

                If ball.Handler.current_behavior.diagonal < (Math.PI / 2) OrElse ball.Handler.current_behavior.diagonal > (3 / 2) * Math.PI Then
                    'ball is going to the right, it will 'bounce' to the left.
                    angle = Math.PI
                    right = False
                Else
                    'ball is going to the left, and will 'bounce' right.
                    angle = 0
                    right = True
                End If

                Dim ball_center As Point = ball.Handler.Center
                Dim kicker_center As Point = kicker.Player.Center

                Dim y_difference = kicker_center.Y - ball_center.Y

                Dim kicker_height = kicker.Player.current_behavior.current_image.Height / 1.5

                If kicker.Player.Center.X < (gamescreen.Bounds.Width * 0.5) Then

                    If y_difference > 0 Then
                        angle += (Math.PI / 4) * (Math.Abs(y_difference) / kicker_height)
                    Else
                        angle -= (Math.PI / 4) * (Math.Abs(y_difference) / kicker_height)
                    End If

                Else
                    If y_difference > 0 Then
                        angle -= (Math.PI / 4) * (Math.Abs(y_difference) / kicker_height)
                    Else
                        angle += (Math.PI / 4) * (Math.Abs(y_difference) / kicker_height)
                    End If
                End If

                'add a bit of inaccuracy
                If Rnd() > 0.5 Then
                    angle += Rnd() * (Math.PI / 8)
                Else
                    angle -= Rnd() * (Math.PI / 8)
                End If

                'avoid high angels
                If angle >= Math.PI / 2 AndAlso angle < Math.PI * (2 / 3) Then
                    angle = Math.PI * (2 / 3)
                End If
                If angle <= Math.PI / 2 AndAlso angle > Math.PI * (1 / 3) Then
                    angle = Math.PI / 3
                End If
                If angle <= Math.PI * (3 / 2) AndAlso angle > Math.PI * (4 / 3) Then
                    angle = Math.PI * (4 / 3)
                End If
                If angle >= Math.PI * (3 / 2) AndAlso angle < Math.PI * (5 / 3) Then
                    angle = Math.PI * (5 / 3)
                End If

                ball.Kick(speed * kicker.Player.GetScale(), angle, kicker)

            End Sub

            'returns radians
            Function Get_Angle_To_Object(ByVal target As Point) As Double

                'opposite = y_distance
                Dim opposite = Player.Center.Y - target.Y

                'adjacent = x_distance
                Dim adjacent = Player.Center.X - target.X

                Dim hypotenuse As Double = Math.Sqrt(opposite ^ 2 + adjacent ^ 2)

                'sin(angle) = opposite / h
                'angle = asin(opposite / h)

                Dim angle = Math.Asin(Math.Abs(opposite) / hypotenuse)

                ' if the target is below, flip the angle to the 4th quadrant
                If target.Y > Player.Center.Y Then
                    angle = (2 * Math.PI) - angle
                    'if the target is to the left, flip the angle to 3rd quadrant
                    If target.X < Player.Center.X Then
                        angle = Math.PI - angle
                    End If
                Else
                    ' If the tartget is above and to the left, flip the angle to the 2nd quadrant.
                    If target.X < Player.Center.X Then
                        angle = Math.PI - angle
                    End If
                End If

                Return angle

            End Function

            Sub Speak(ByVal line As String)
                Dim new_line As New Pony.Behavior.Speaking_Line(Player.Name, "Kick", line, "", "", True, 0)
                Player.Pony_Speak(new_line)
            End Sub

            Sub PushBackOverlappingPonies(ByVal all_positions As List(Of Position))

                For Each otherposition As Position In all_positions
                    Dim otherpony = otherposition.Player
                    If ReferenceEquals(Me.Player, otherpony) Then Continue For


                    If DoesPonyOverlap(Me.Player, otherpony) Then
                        'Push overlapping ponies a bit apart
                        PonyPush(Me.Player, otherpony, Allowed_Area)
                        Exit Sub
                    End If


                Next

            End Sub

            Function DoesPonyOverlap(ByVal pony As Pony, ByVal otherpony As Pony) As Boolean

                Dim otherpony_area As New Rectangle(otherpony.location.X, _
                                                 otherpony.location.Y, _
                                                 otherpony.current_behavior.current_image.Size.Width * otherpony.GetScale,
                                                 otherpony.current_behavior.current_image.Size.Height * otherpony.GetScale)


                If otherpony_area.IntersectsWith(New Rectangle(pony.location, New Point(pony.GetScale * pony.current_behavior.current_image.Size.Width, _
                                                                                        pony.GetScale * pony.current_behavior.current_image.Size.Height))) Then
                    Return True
                Else
                    Return False
                End If

            End Function

            Sub PonyPush(ByVal pony1 As Pony, ByVal pony2 As Pony, ByVal allowed_area As Pony.RECT)

                Dim xchange = 1
                Dim ychange = 2

                Dim direction = pony1.Get_Destination_Direction(pony2.Center)

                If direction(0) = Pony.Directions.left Then
                    xchange = -1
                End If
                If direction(1) = Pony.Directions.top Then
                    ychange = -2
                End If

                Dim new_px = pony1.Precise_X_Location - xchange
                Dim new_py = pony1.Precise_Y_Location - ychange
                Dim new_location = New Point(new_px, new_py)

                Dim screenlist As New List(Of Screen)
                screenlist.Add(My.Forms.Main.current_game.GameScreen)

                If pony1.IsPonyOnScreen(new_location, screenlist) AndAlso pony1.IsPonyInBox(new_location, allowed_area) Then
                    pony1.Precise_X_Location -= xchange
                    pony1.Precise_Y_Location -= ychange
                    pony1.Location = New Point(pony1.Precise_X_Location, pony1.Precise_Y_Location)
                End If

            End Sub

            Function Friendly_Ponies_Around_Ball(ByVal ball As Ball, ByVal team As Team, ByVal min_distance As Integer) As List(Of Pony)

                Dim ponies As New List(Of Pony)

                For Each Position In team.Positions
                    Dim distance = Math.Sqrt((Position.Player.Center.X - ball.Center().X) ^ 2 + (Position.Player.Center.Y - ball.Center().Y) ^ 2)
                    If distance <= min_distance Then
                        ponies.Add(Position.Player)
                    End If
                Next

                Return ponies
            End Function

            'get a teammate that is not near any enemy players and is closer to the goal than we are.
            Function Get_Open_Teammate(ByVal team As Team, ByVal goal As Goal_Area) As Pony

                Dim open_teammates As New List(Of Pony)

                For Each Position In team.Positions

                    If ReferenceEquals(Position, Me) Then
                        Continue For
                    End If

                    Dim open = True
                    For Each other_position As Position In My.Forms.Main.current_game.AllPlayers
                        If other_position.Team.Name = Me.Team.Name Then
                            Continue For
                        End If

                        Dim distance = Math.Sqrt((Position.Player.Center.X - other_position.Player.Center().X) ^ 2 + (Position.Player.Center.Y - other_position.Player.Center().Y) ^ 2)
                        If distance <= 200 Then
                            open = False
                        End If
                    Next

                    Dim me_distance_to_goal = Math.Sqrt((Me.Player.Center.X - goal.Center().X) ^ 2 + (Me.Player.Center.Y - goal.Center().Y) ^ 2)
                    Dim teammate_distance_to_goal = Math.Sqrt((Position.Player.Center.X - goal.Center().X) ^ 2 + (Position.Player.Center.Y - goal.Center().Y) ^ 2)

                    If open = True AndAlso teammate_distance_to_goal <= me_distance_to_goal Then
                        open_teammates.Add(Position.Player)
                    End If
                Next

                If open_teammates.Count = 0 Then Return Nothing

                Return open_teammates(Rnd() * (open_teammates.Count - 1))

            End Function

            Sub SetSpeed()

                'set all ponies to be the same speed. (Prevent rainbow from always winning).
                If Player.current_behavior.speed = Player.current_behavior.original_speed * 2 * Player.GetScale() = True Then
                    If My.Forms.Main.current_game.Name = "Ping Pong Pony" Then
                        Player.current_behavior.speed = 8 * Player.GetScale()
                    Else
                        Player.current_behavior.speed = 5 * Player.GetScale()
                    End If
                Else
                    If My.Forms.Main.current_game.Name = "Ping Pong Pony" Then
                        Player.current_behavior.speed = 5 * Player.GetScale()
                    Else
                        Player.current_behavior.speed = 3 * Player.GetScale()
                    End If
                End If

            End Sub

        End Class
    End Class



End Module
