Imports System.Collections.Generic
Imports System.Drawing
Imports System.Drawing.Drawing2D
Imports System.Globalization
Imports System.Threading
Imports System.Windows.Forms
Namespace DesktopPonies
#Region "LinkedListExtensions class"
    ''' <summary>
    ''' Defines Sort extension methods for LinkedList.
    ''' </summary>
    Public Module LinkedListExtensions
        ''' <summary>
        ''' Sorts the elements in the entire System.Collections.Generic.LinkedList using the default comparer.
        ''' </summary>
        ''' <typeparam name="TSource">The type of the elements of source.</typeparam>
        ''' <param name="source">A list to sort.</param>
        <System.Runtime.CompilerServices.Extension()> _
        Public Sub Sort(Of TSource)(source As LinkedList(Of TSource))
            If Comparer(Of TSource).[Default] Is Nothing Then
                Throw New InvalidOperationException()
            End If

            Sort(source, Comparer(Of TSource).[Default])
        End Sub

        ''' <summary>
        ''' Sorts the elements in the entire System.Collections.Generic.LinkedList using the specified System.Comparison.
        ''' </summary>
        ''' <typeparam name="TSource">The type of the elements of source.</typeparam>
        ''' <param name="source">A list to sort.</param>
        ''' <param name="comparison">The System.Comparison to use when comparing elements.</param>
        <System.Runtime.CompilerServices.Extension()> _
        Public Sub Sort(Of TSource)(source As LinkedList(Of TSource), comparison As Comparison(Of TSource))
            If comparison Is Nothing Then
                Throw New ArgumentNullException("comparison")
            End If

            MergeSort(source, comparison)
        End Sub

        ''' <summary>
        ''' Sorts the elements in the entire System.Collections.Generic.LinkedList using the specified comparer.
        ''' </summary>
        ''' <typeparam name="TSource">The type of the elements of source.</typeparam>
        ''' <param name="source">A list to sort.</param>
        ''' <param name="comparer">The System.Collections.Generic.IComparer implementation to use when comparing elements.</param>
        <System.Runtime.CompilerServices.Extension()> _
        Public Sub Sort(Of TSource)(source As LinkedList(Of TSource), comparer As IComparer(Of TSource))
            If comparer Is Nothing Then
                Throw New InvalidOperationException()
            End If

            MergeSort(source, AddressOf comparer.Compare)
        End Sub

        ''' <summary>
        ''' Performs a merge sort on the elements in the entire System.Collections.Generic.LinkedList using the specified
        ''' System.Comparison.
        ''' </summary>
        ''' <typeparam name="T">The type of the elements of list.</typeparam>
        ''' <param name="list">A list to sort.</param>
        ''' <param name="comparison">The System.Comparison to use when comparing elements.</param>
        ''' <remarks>This method performs an in-place stable merge sort of the given list. The sort requires O(n log n) running time and
        ''' O(1) extra memory.</remarks>
        Private Sub MergeSort(Of T)(list As LinkedList(Of T), comparison As Comparison(Of T))
            Dim mergeSize As Integer = 1
            Dim merges As Integer
            Dim leftHead As LinkedListNode(Of T)
            Dim rightHead As LinkedListNode(Of T)
            Dim nextRightHeadPrevious As LinkedListNode(Of T) = list.First
            ' Perform a series of passes, in which all the pairs of lists are individually merged.
            Do
                ' Keep a count of the merges done this pass, if none were done then the list is sorted.
                merges = 0

                ' Set the initial positions of the list sections to sort.
                leftHead = list.First
                rightHead = nextRightHeadPrevious.[Next]

                ' Perform a series of merges considering mergeSize elements from each side, do this whilst a second list exists.
                While rightHead IsNot Nothing
                    ' Perform a merge on the next pair of lists.
                    merges += 1
                    Dim leftSize As Integer = mergeSize
                    Dim rightSize As Integer = mergeSize

                    ' Merge whilst there are elements remaining in either list.
                    While leftSize > 0 AndAlso rightSize > 0 AndAlso rightHead IsNot Nothing
                        If comparison.Invoke(leftHead.Value, rightHead.Value) <= 0 Then
                            ' The elements were already in the correct order, so just advance on the left list.
                            ' We also advance in the case of equality, in order to create a stable sort.
                            leftHead = leftHead.[Next]
                            leftSize -= 1
                        Else
                            ' The elements are in the wrong order, we must swap them.
                            Dim nextRightHead As LinkedListNode(Of T) = rightHead.[Next]
                            rightSize -= 1
                            list.Remove(rightHead)
                            list.AddBefore(leftHead, rightHead)
                            rightHead = nextRightHead
                        End If
                    End While

                    ' Advance the right head to the start of the next pair of lists.
                    While rightSize > 0 AndAlso rightHead IsNot Nothing
                        rightSize -= 1
                        rightHead = rightHead.[Next]
                    End While

                    ' Set up the start positions for the next pair of lists.
                    leftHead = rightHead
                    Dim i As Integer = 0
                    While i < mergeSize AndAlso rightHead IsNot Nothing
                        rightHead = rightHead.[Next]
                        i += 1
                    End While

                    ' If we just completed the first merge, we can save the position of the end of the last list, and advance from there
                    ' on the next pass to get the starting point for the right without having to go through the list to find it.
                    If merges = 1 Then
                        If leftHead IsNot Nothing Then
                            nextRightHeadPrevious = leftHead.Previous
                        Else
                            nextRightHeadPrevious = list.Last
                        End If
                    End If
                End While
                ' Double the size of the lists we are merging.
                mergeSize *= 2
            Loop While merges > 0
        End Sub
    End Module
#End Region

    ''' <summary>
    ''' Manages a transparent form that handles the display of a collection of IAnimatableInstances.
    ''' </summary>
    ''' <remarks>
    ''' This IInstanceAnimator provides for the management of a collection of IAnimatableInstances, and maintains a form to display them.
    ''' This class will call Start on any instances passed to it when required. This class guarantees to call Update and Draw once each per
    ''' tick. It will attempt to tick at a frequency in order to meet the desired MaximumFrameRate, but this is dependant on performance.
    ''' </remarks>
    Public Class AnimatorForm
        Implements IInstanceAnimator
        Implements IDisposable
#Region "Variables and Properties"
        ''' <summary>
        ''' The underlying form on which graphics are displayed.
        ''' </summary>
        Private ReadOnly form As New GraphicsForm()
        ''' <summary>
        ''' The context that allocates graphics buffers.
        ''' </summary>
        Private ReadOnly bufferedGraphicsContext As New BufferedGraphicsContext()
        ''' <summary>
        ''' The graphics buffer of the form to which drawing is done and from which rendering to the screen is performed.
        ''' </summary>
        Private bufferedGraphics As BufferedGraphics
        ''' <summary>
        ''' The graphics surface of the form (unbuffered), used to provide the pixel format required when allocating graphics buffers.
        ''' </summary>
        Private graphics As Graphics
        ''' <summary>
        ''' Indicates if any manual painting is taking place.
        ''' </summary>
        Private manualPainting As Boolean
        ''' <summary>
        ''' The PaintEventHandler that ensures the form is repainted when paused.
        ''' </summary>
        Private manualRender As PaintEventHandler
        ''' <summary>
        ''' The PaintEventHandler that ensures the form can be manually cleared.
        ''' </summary>
        Private manualClear As PaintEventHandler
        ''' <summary>
        ''' Synchronization object for use when the resources of the class must be locked.
        ''' </summary>
        Private resourcesObject As New Object()

        ''' <summary>
        ''' Main watch used to record the elapsed time for frame rendering, and to track the accuracy of frame intervals.
        ''' </summary>
        Private ReadOnly tickWatch As New System.Diagnostics.Stopwatch()
        ''' <summary>
        ''' Watch used to record the total elapsed time of animation since the form was started.
        ''' </summary>
        Private ReadOnly elapsedWatch As New System.Diagnostics.Stopwatch()
        ''' <summary>
        ''' Gets a value indicating whether animation has been started.
        ''' </summary>
        Public ReadOnly Property Started() As Boolean Implements IInstanceAnimator.Started
            Get
                Return m_Started
            End Get
        End Property
        Private m_Started As Boolean
        ''' <summary>
        ''' Gets or sets a value indicating whether animation is currently paused.
        ''' </summary>
        Public Property Paused() As Boolean
            Get
                Return Not elapsedWatch.IsRunning
            End Get
            Set(value As Boolean)
                If value Then
                    Pause(False)
                Else
                    [Resume]()
                End If
            End Set
        End Property
        ''' <summary>
        ''' Used to pause the thread running the main loop. Signals false whilst paused, otherwise signals true.
        ''' </summary>
        Private running As New ManualResetEvent(True)
        ''' <summary>
        ''' Gets the total elapsed time that animation has been running.
        ''' </summary>
        Public ReadOnly Property ElapsedTime() As TimeSpan
            Get
                Return elapsedWatch.Elapsed
            End Get
        End Property

        ''' <summary>
        ''' The collection of animatable instances.
        ''' </summary>
        Private m_instances As LinkedList(Of IAnimatableInstance)
        ''' <summary>
        ''' Gets the collection of animatable instances.
        ''' </summary>
        Protected ReadOnly Property Instances() As InstancesCollection
            Get
                Return m_InstancesCollection
            End Get
        End Property
        Private m_InstancesCollection As InstancesCollection
        ''' <summary>
        ''' Represents the area that becomes invalidated before updating.
        ''' This area needs to be cleared.
        ''' </summary>
        Private preUpdateInvalidRegion As New Region()
        ''' <summary>
        ''' Represents the area that becomes invalidated after updating.
        ''' This area needs to be drawn.
        ''' </summary>
        Private postUpdateInvalidRegion As New Region()
        ''' <summary>
        ''' The number of ponies below which region based invalidation is done to save on repainting the whole screen.
        ''' Above this, it is cheaper to just redraw everything than to calculate and perform hit testing on the region in the first place.
        ''' </summary>
        Private threshold As Integer

        ''' <summary>
        ''' Gets or sets the text associated with the form.
        ''' </summary>
        Public Property Text() As String
            Get
                Return form.Text
            End Get
            Set(value As String)
                form.Text = value
            End Set
        End Property
        ''' <summary>
        ''' Gets or sets the icon for the form.
        ''' </summary>
        Public Property Icon() As Icon
            Get
                Return form.Icon
            End Get
            Set(value As Icon)
                form.Icon = value
            End Set
        End Property
        ''' <summary>
        ''' Gets or sets a value indicating whether the form should be displayed as a topmost form.
        ''' This will cause a momentary graphical stutter if the form has already been started.
        ''' If animation is active, this will not take effect until the next tick begins.
        ''' </summary>
        ''' <value>True to display the form as a topmost form; otherwise, false.</value>
        Public Property Topmost() As Boolean
            Get
                Return form.TopMost
            End Get
            Set(value As Boolean)
                If value <> form.TopMost Then
                    If Not Started Then
                        form.TopMost = Not form.TopMost
                        form.ShowInTaskbar = Not form.TopMost
                    Else
                        ToggleTopmost(Paused)
                    End If
                End If
            End Set
        End Property
        ''' <summary>
        ''' Gets or sets the display bounds of the form.
        ''' </summary>
        Public Property DisplayArea() As Rectangle Implements IInstanceAnimator.DisplayArea
            Get
                Return form.DisplayRectangle
            End Get
            Set(value As Rectangle)
                If Started Then
                    Throw New InvalidOperationException("Cannot change the display area once the form has been started.")
                End If

                form.Location = value.Location
                form.Size = value.Size
            End Set
        End Property
        ''' <summary>
        ''' Gets the current location of the cursor.
        ''' </summary>
        Public ReadOnly Property CursorLocation() As Point Implements IInstanceAnimator.CursorLocation
            Get
                Return m_CursorLocation
            End Get
        End Property
        Private m_CursorLocation As Point

        ''' <summary>
        ''' The FontFamily to be used when drawing text to the screen.
        ''' </summary>
        Private ReadOnly fontFamily As FontFamily = fontFamily.GenericMonospace
        ''' <summary>
        ''' The full Font definition to be used when drawing text to the screen.
        ''' </summary>
        Private ReadOnly font As Font
        ''' <summary>
        ''' The brush used to draw the dark backdrop to the frame timings string.
        ''' </summary>
        Private ReadOnly darkTimingsBrush As Brush = New SolidBrush(Color.FromArgb(1, 1, 1))
        ''' <summary>
        ''' Gets or sets a value indicating whether the timings graph should be displayed.
        ''' </summary>
        Public Property ShowTimingsGraph() As Boolean
            Get
                Return m_ShowTimingsGraph
            End Get
            Set(value As Boolean)
                m_ShowTimingsGraph = value
            End Set
        End Property
        Private m_ShowTimingsGraph As Boolean
        ''' <summary>
        ''' Gets or sets a value indicating whether the clipping region should be displayed.
        ''' </summary>
        Public Property ShowClippingRegion() As Boolean
            Get
                Return m_ShowClippingRegion
            End Get
            Set(value As Boolean)
                m_ShowClippingRegion = value
            End Set
        End Property
        Private m_ShowClippingRegion As Boolean

        ''' <summary>
        ''' Stores the frame time of this loop, for recording at the start of the next loop.
        ''' </summary>
        Private frameTime As Single = Single.NaN
        ''' <summary>
        ''' Holds information about the performance of the form.
        ''' </summary>
        Private recorder As New FrameInfoRecorder(300)
        ''' <summary>
        ''' The area where the graph displayed by the recorder will be drawn.
        ''' </summary>
        Private recorderGraphArea As Rectangle
        ''' <summary>
        ''' The identity matrix.
        ''' </summary>
        Private identityMatrix As New Matrix()
        ''' <summary>
        ''' The minimum value for the interval of the timer.
        ''' </summary>
        Private minimumTickInterval As Single = 1000.0F / 60.0F
        ''' <summary>
        ''' Gets or sets the target frame rate of the animation.
        ''' </summary>
        Public Property MaximumFramesPerSecond() As Single
            Get
                Return 1000.0F / minimumTickInterval
            End Get
            Set(value As Single)
                If value <= 0.0F Then
                    Throw New ArgumentException("Value must be positive.", "value")
                End If
                If value > 120.0F Then
                    Throw New ArgumentException("Value should not exceed 120.", "value")
                End If
                minimumTickInterval = 1000.0F / value
            End Set
        End Property
#End Region

#Region "Expose Form Events"
        ''' <summary>
        ''' Occurs after the form is closed.
        ''' </summary>
        Public Custom Event FormClosed As FormClosedEventHandler
            AddHandler(ByVal value As FormClosedEventHandler)
                AddHandler form.FormClosed, value
            End AddHandler
            RemoveHandler(ByVal value As FormClosedEventHandler)
                RemoveHandler form.FormClosed, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
        ''' <summary>
        ''' Occurs before the form is closed.
        ''' </summary>
        Public Custom Event FormClosing As FormClosingEventHandler
            AddHandler(ByVal value As FormClosingEventHandler)
                AddHandler form.FormClosing, value
            End AddHandler
            RemoveHandler(ByVal value As FormClosingEventHandler)
                RemoveHandler form.FormClosing, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
        ''' <summary>
        ''' Occurs when a key is pressed while the control has focus.
        ''' </summary>
        Public Custom Event KeyDown As KeyEventHandler
            AddHandler(ByVal value As KeyEventHandler)
                AddHandler form.KeyDown, value
            End AddHandler
            RemoveHandler(ByVal value As KeyEventHandler)
                RemoveHandler form.KeyDown, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
        ''' <summary>
        ''' Occurs when a key is pressed while the control has focus.
        ''' </summary>
        Public Custom Event KeyPress As KeyPressEventHandler
            AddHandler(ByVal value As KeyPressEventHandler)
                AddHandler form.KeyPress, value
            End AddHandler
            RemoveHandler(ByVal value As KeyPressEventHandler)
                RemoveHandler form.KeyPress, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
        ''' <summary>
        ''' Occurs when a key is released while the control has focus.
        ''' </summary>
        Public Custom Event KeyUp As KeyEventHandler
            AddHandler(ByVal value As KeyEventHandler)
                AddHandler form.KeyUp, value
            End AddHandler
            RemoveHandler(ByVal value As KeyEventHandler)
                RemoveHandler form.KeyUp, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
        ''' <summary>
        ''' Occurs when the control loses or gains mouse capture.
        ''' </summary>
        Public Custom Event MouseCaptureChanged As EventHandler
            AddHandler(ByVal value As EventHandler)
                AddHandler form.MouseCaptureChanged, value
            End AddHandler
            RemoveHandler(ByVal value As EventHandler)
                RemoveHandler form.MouseCaptureChanged, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
        ''' <summary>
        ''' Occurs when the control is clicked by the mouse.
        ''' </summary>
        Public Custom Event MouseClick As MouseEventHandler
            AddHandler(ByVal value As MouseEventHandler)
                AddHandler form.MouseClick, value
            End AddHandler
            RemoveHandler(ByVal value As MouseEventHandler)
                RemoveHandler form.MouseClick, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
        ''' <summary>
        ''' Occurs when the control is double clicked by the mouse.
        ''' </summary>
        Public Custom Event MouseDoubleClick As MouseEventHandler
            AddHandler(ByVal value As MouseEventHandler)
                AddHandler form.MouseDoubleClick, value
            End AddHandler
            RemoveHandler(ByVal value As MouseEventHandler)
                RemoveHandler form.MouseDoubleClick, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
        ''' <summary>
        ''' Occurs when the mouse pointer is over the control and a mouse button is pressed.
        ''' </summary>
        Public Custom Event MouseDown As MouseEventHandler
            AddHandler(ByVal value As MouseEventHandler)
                AddHandler form.MouseDown, value
            End AddHandler
            RemoveHandler(ByVal value As MouseEventHandler)
                RemoveHandler form.MouseDown, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
        ''' <summary>
        ''' Occurs when the mouse pointer enters the control.
        ''' </summary>
        Public Custom Event MouseEnter As EventHandler
            AddHandler(ByVal value As EventHandler)
                AddHandler form.MouseEnter, value
            End AddHandler
            RemoveHandler(ByVal value As EventHandler)
                RemoveHandler form.MouseEnter, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
        ''' <summary>
        ''' Occurs when the mouse pointer rests on the control.
        ''' </summary>
        Public Custom Event MouseHover As EventHandler
            AddHandler(ByVal value As EventHandler)
                AddHandler form.MouseHover, value
            End AddHandler
            RemoveHandler(ByVal value As EventHandler)
                RemoveHandler form.MouseHover, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
        ''' <summary>
        ''' Occurs when the mouse pointer leaves the control.
        ''' </summary>
        Public Custom Event MouseLeave As EventHandler
            AddHandler(ByVal value As EventHandler)
                AddHandler form.MouseLeave, value
            End AddHandler
            RemoveHandler(ByVal value As EventHandler)
                RemoveHandler form.MouseLeave, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
        ''' <summary>
        ''' Occurs when the mouse pointer is moved over the control.
        ''' </summary>
        Public Custom Event MouseMove As MouseEventHandler
            AddHandler(ByVal value As MouseEventHandler)
                AddHandler form.MouseMove, value
            End AddHandler
            RemoveHandler(ByVal value As MouseEventHandler)
                RemoveHandler form.MouseMove, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
        ''' <summary>
        ''' Occurs when the mouse pointer is over the control and a mouse button is released.
        ''' </summary>
        Public Custom Event MouseUp As MouseEventHandler
            AddHandler(ByVal value As MouseEventHandler)
                AddHandler form.MouseUp, value
            End AddHandler
            RemoveHandler(ByVal value As MouseEventHandler)
                RemoveHandler form.MouseUp, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
        ''' <summary>
        ''' Occurs when the mouse wheel moves while the control has focus.
        ''' </summary>
        Public Custom Event MouseWheel As MouseEventHandler
            AddHandler(ByVal value As MouseEventHandler)
                AddHandler form.MouseWheel, value
            End AddHandler
            RemoveHandler(ByVal value As MouseEventHandler)
                RemoveHandler form.MouseWheel, value
            End RemoveHandler
            RaiseEvent()
            End RaiseEvent
        End Event
#End Region

#Region "GraphicsForm class"
        ''' <summary>
        ''' Transparent form that handles drawing and display of graphics.
        ''' </summary>
        Private Class GraphicsForm
            Inherits Form
            ''' <summary>
            ''' Initializes a new instance of the GraphicsForm class.
            ''' </summary>
            Public Sub New()
                ' Create the form.
                Name = [GetType]().Name
                Cursor = Cursors.Hand
                FormBorderStyle = FormBorderStyle.None
                AutoScaleMode = AutoScaleMode.None
                Visible = False

                ' Tell the OS we'll be handling the drawing.
                ' This causes the OnPaint and OnPaintBackground events to be raised for us.
                SetStyle(ControlStyles.UserPaint, True)
                ' Prevents the OS from raising the OnPaintBackground event on us, which prevents flicker.
                ' This event will now only occur when we raise it.
                SetStyle(ControlStyles.AllPaintingInWmPaint, True)

                If My.Forms.Main.OS_Is_Old Then
                    ' Certain versions of XP won't support transparency with the settings given so far. Using the
                    ' SupportsTransparentBackColor draws the form onto an all zeroed buffer. The flag is only meant to support transparent
                    ' controls on or above other controls and not to provide actual transparency on the desktop but we can abuse the all
                    ' zero buffer. With the buffer being transparent black as a result of being zeroed, we can abuse that fact and use
                    ' transparent black as the TransparencyKey.
                    SetStyle(ControlStyles.SupportsTransparentBackColor, True)
                    BackColor = Color.FromArgb(255, 0, 0, 0)
                    TransparencyKey = Color.FromArgb(0, 0, 0, 0)
                Else
                    ' Forms don't truly support a transparent background, however a transparency key can be used to tell it to treat a
                    ' certain color as transparent. So we'll try and use an uncommon color. This also has the bonus effect of making
                    ' interaction fall through the transparent areas. This means interaction with the desktop is possible.
                    BackColor = Color.FromArgb(254, 255, 254)
                    TransparencyKey = BackColor
                End If
            End Sub
        End Class
#End Region

#Region "InstancesCollection class"
        ''' <summary>
        ''' Provides a wrapper around a LinkedList of IAnimatableInstance so that outside access can be monitored and controlled.
        ''' Methods that modify the LinkedList are made thread-safe, so that ongoing enumerators are not invalidated.
        ''' </summary>
        Protected Class InstancesCollection
            Implements ICollection(Of IAnimatableInstance)
            ''' <summary>
            ''' Gets the form that displays the instances in the collection, and that is used when starting additional instances.
            ''' </summary>
            Public Property Form() As AnimatorForm
                Get
                    Return m_Form
                End Get
                Private Set(value As AnimatorForm)
                    m_Form = value
                End Set
            End Property
            Private m_Form As AnimatorForm

            ''' <summary>
            ''' The underlying collection of instances.
            ''' </summary>
            Private instances As LinkedList(Of IAnimatableInstance)

            ''' <summary>
            ''' Initializes a new instance of the InstancesCollection class.
            ''' </summary>
            ''' <param name="owner">The form that owns this collection, from which the initial parameters for new instances are created.
            ''' </param>
            ''' <param name="collection">The LinkedList of IAnimatableInstance that should be wrapped.</param>
            Public Sub New(owner As AnimatorForm, collection As LinkedList(Of IAnimatableInstance))
                If owner Is Nothing Then
                    Throw New ArgumentNullException("owner")
                End If

                If collection Is Nothing Then
                    Throw New ArgumentNullException("collection")
                End If

                Form = owner
                instances = collection
            End Sub

            ''' <summary>
            ''' Gets the number of instances contained in the InstancesCollection.
            ''' </summary>
            Public ReadOnly Property Count() As Integer Implements ICollection(Of IAnimatableInstance).Count
                Get
                    Return instances.Count
                End Get
            End Property

            ''' <summary>
            ''' Gets a value indicating whether the InstancesCollection is read-only.
            ''' </summary>
            Public ReadOnly Property IsReadOnly() As Boolean Implements ICollection(Of IAnimatableInstance).IsReadOnly
                Get
                    Return False
                End Get
            End Property

            ''' <summary>
            ''' Adds an IAnimatableInstance to the InstancesCollection.
            ''' </summary>
            ''' <param name="instance">The IAnimatableInstance to add to the InstancesCollection.</param>
            Public Sub Add(instance As IAnimatableInstance) Implements ICollection(Of IAnimatableInstance).Add
                If instance Is Nothing Then
                    Throw New ArgumentNullException("instance")
                End If

                SyncLock Me
                    instance.Start(Form.ElapsedTime, Form)
                    instances.AddLast(instance)
                End SyncLock
            End Sub

            ''' <summary>
            ''' Removes all instances from the InstancesCollection.
            ''' </summary>
            Public Sub Clear() Implements ICollection(Of IAnimatableInstance).Clear
                SyncLock Me
                    instances.Clear()
                End SyncLock
            End Sub

            ''' <summary>
            ''' Determines whether the InstancesCollection contains a specific IAnimatableInstance.
            ''' </summary>
            ''' <param name="instance">The IAnimatableInstance to locate in the InstancesCollection.</param>
            ''' <returns>True if instance is found in the InstancesCollection; otherwise, false.</returns>
            Public Function Contains(instance As IAnimatableInstance) As Boolean Implements ICollection(Of IAnimatableInstance).Contains
                Return instances.Contains(instance)
            End Function

            ''' <summary>
            ''' Copies the elements of the InstancesCollection to an System.Array, starting at a particular System.Array index.
            ''' </summary>
            ''' <param name="array">The one-dimensional System.Array that is the destination of the elements copied from
            ''' InstancesCollection. The System.Array must have zero-based indexing.</param>
            ''' <param name="index">The zero-based index in array at which copying begins.</param>
            Public Sub CopyTo(array As IAnimatableInstance(), index As Integer) Implements ICollection(Of IAnimatableInstance).CopyTo
                instances.CopyTo(array, index)
            End Sub

            ''' <summary>
            ''' Removes the first occurrence of a specific IAnimatableInstance from the InstancesCollection.
            ''' </summary>
            ''' <param name="instance">The IAnimatableInstance to remove from the InstancesCollection.</param>
            ''' <returns>True if instance was successfully removed from the InstancesCollection; otherwise, false. This method also returns
            ''' false if instance is not found in the original InstancesCollection.</returns>
            Public Function Remove(instance As IAnimatableInstance) As Boolean Implements ICollection(Of IAnimatableInstance).Remove
                SyncLock Me
                    Return instances.Remove(instance)
                End SyncLock
            End Function

            ''' <summary>
            ''' Returns an enumerator that iterates through the InstancesCollection.
            ''' </summary>
            ''' <returns>An System.Collections.Generic.IEnumerator that can be used to iterate through the collection.</returns>
            Public Function GetEnumerator() As IEnumerator(Of IAnimatableInstance) Implements IEnumerable(Of IAnimatableInstance).GetEnumerator
                Return instances.GetEnumerator()
            End Function

            ''' <summary>
            ''' Returns an enumerator that iterates through the InstancesCollection.
            ''' </summary>
            ''' <returns>An System.Collections.IEnumerator that can be used to iterate through the collection.</returns>
            Private Function System_Collections_IEnumerable_GetEnumerator() As System.Collections.IEnumerator Implements System.Collections.IEnumerable.GetEnumerator
                Return GetEnumerator()
            End Function

            ''' <summary>
            ''' Sorts the elements in the entire InstancesCollection using the default comparer.
            ''' </summary>
            Public Sub Sort()
                instances.Sort()
            End Sub

            ''' <summary>
            ''' Sorts the elements in the entire InstancesCollection using the specified System.Comparison.
            ''' </summary>
            ''' <param name="comparsion">The System.Comparison to use when comparing elements.</param>
            Public Sub Sort(comparsion As Comparison(Of IAnimatableInstance))
                instances.Sort(comparsion)
            End Sub

            ''' <summary>
            ''' Sorts the elements in the entire InstancesCollection using the specified comparer.
            ''' </summary>
            ''' <param name="comparer">The System.Collections.Generic.IComparer implementation to use when comparing elements, or null to
            ''' use the default comparer System.Collections.Generic.Comparer.Default.</param>
            Public Sub Sort(comparer As IComparer(Of IAnimatableInstance))
                instances.Sort(comparer)
            End Sub
        End Class
#End Region

#Region "FrameInfoRecorder class"
        ''' <summary>
        ''' Holds information about the rendering times, frame rate and garbage collections and provides methods to output and display this
        ''' data.
        ''' </summary>
        Private Class FrameInfoRecorder
            Implements IDisposable
#Region "FrameInfo struct"
            ''' <summary>
            ''' Holds information for one frame about its render time, interval and garbage collections performed in this interval.
            ''' </summary>
            Private Structure FrameInfo
                ''' <summary>
                ''' The time this frame took to update and draw, in milliseconds.
                ''' </summary>
                Public ReadOnly Time As Single
                ''' <summary>
                ''' The interval between the starting of this frame and the last frame, in milliseconds.
                ''' </summary>
                Public ReadOnly Interval As Single
                ''' <summary>
                ''' The cumulative number of generation zero garbage collections that had occurred by this frame.
                ''' </summary>
                Public ReadOnly Gen0Collections As Integer
                ''' <summary>
                ''' The cumulative number of generation one garbage collections that had occurred by this frame.
                ''' </summary>
                Public ReadOnly Gen1Collections As Integer
                ''' <summary>
                ''' The cumulative number of generation two garbage collections that had occurred by this frame.
                ''' </summary>
                Public ReadOnly Gen2Collections As Integer

                ''' <summary>
                ''' Initializes a new instance of the FrameInfo struct with the given frame time and interval,
                ''' and records the current garbage collection counts.
                ''' </summary>
                ''' <param name="time__1">The time the frame took to update and draw, in milliseconds.</param>
                ''' <param name="interval__2">The interval between frames (inclusive of frame rendering time).</param>
                Public Sub New(time__1 As Single, interval__2 As Single)
                    Time = time__1
                    Interval = interval__2
                    Gen0Collections = GC.CollectionCount(0)
                    Gen1Collections = GC.CollectionCount(1)
                    Gen2Collections = GC.CollectionCount(2)
                End Sub

                ''' <summary>
                ''' Determines the highest, if any, generation of garbage collection that was performed between two FrameInfos.
                ''' </summary>
                ''' <param name="a">First FrameInfo to compare for garbage collections.</param>
                ''' <param name="b">Second FrameInfo to compare for garbage collections.</param>
                ''' <returns>A integer indicating the highest garbage generation collected, or -1 if no collections occurred.</returns>
                Public Shared Function CollectionGenerationPerformed(a As FrameInfo, b As FrameInfo) As Integer
                    If a.Gen2Collections <> b.Gen2Collections Then
                        Return 2
                    ElseIf a.Gen1Collections <> b.Gen1Collections Then
                        Return 1
                    ElseIf a.Gen0Collections <> b.Gen0Collections Then
                        Return 0
                    Else
                        Return -1
                    End If
                End Function

                ''' <summary>
                ''' Gets a Brush related to the highest, if any, generation of garbage collection that was performed between two FrameInfos.
                ''' </summary>
                ''' <param name="a">First FrameInfo to compare for garbage collections.</param>
                ''' <param name="b">Second FrameInfo to compare for garbage collections.</param>
                ''' <returns>A Brush whose color is determined by the garbage collection performed, if any.
                ''' The color matches those used in the CLR Profiler.</returns>
                Public Shared Function CollectionGenerationPerformedBrush(a As FrameInfo, b As FrameInfo) As Brush
                    Select Case CollectionGenerationPerformed(a, b)
                        Case 2
                            Return Brushes.Blue
                        Case 1
                            Return Brushes.Green
                        Case 0
                            Return Brushes.Red
                        Case Else
                            Return Brushes.White
                    End Select
                End Function
            End Structure
#End Region

            ''' <summary>
            ''' Information about previous frames, this array is reused circularly with the current start index given by marker.
            ''' </summary>
            Private frameInfos As FrameInfo()
            ''' <summary>
            ''' The current location in the array where the new information is being written.
            ''' </summary>
            Private marker As Integer

            ''' <summary>
            ''' The area the graph is to occupy when drawn.
            ''' </summary>
            Private graphArea As Rectangle
            ''' <summary>
            ''' The width of a bar for each frame time.
            ''' </summary>
            Private barWidth As Integer
            ''' <summary>
            ''' The scale factor for the height of bars, where a factor of 1 results in 1 pixel per millisecond.
            ''' </summary>
            Private barHeightFactor As Single
            ''' <summary>
            ''' The brush used to paint the background of the graph area.
            ''' </summary>
            Private graphBackgroundBrush As Brush = New SolidBrush(Color.FromArgb(1, 1, 1))

            ''' <summary>
            ''' Initializes a new instance of the FrameInfoRecorder class to hold data about the given number of frames.
            ''' </summary>
            ''' <param name="count">The number of frames to hold data on.</param>
            Public Sub New(count As Integer)
                frameInfos = New FrameInfo(count - 1) {}
                SetGraphingAttributes(Point.Empty, 100, 1, 1.0F)
            End Sub

            ''' <summary>
            ''' Records the rendering of a new frame that took the given time, and records the garbage collections made.
            ''' </summary>
            ''' <param name="time">The time the frame took to update and draw, in milliseconds.</param>
            ''' <param name="interval">The interval between frames (inclusive of frame rendering time), in milliseconds.</param>
            Public Sub Record(time As Single, interval As Single)
                frameInfos(marker) = New FrameInfo(time, interval)
                marker += 1
                If marker >= frameInfos.Length Then
                    marker = 0
                End If
                If Count < frameInfos.Length Then
                    Count += 1
                End If
            End Sub

            ''' <summary>
            ''' Gets the number of values this recorder can hold.
            ''' </summary>
            Public ReadOnly Property Capacity() As Integer
                Get
                    Return frameInfos.Length
                End Get
            End Property

            ''' <summary>
            ''' Gets the number of recorded values in this recorder.
            ''' </summary>
            Public Property Count() As Integer
                Get
                    Return m_Count
                End Get
                Private Set(value As Integer)
                    m_Count = value
                End Set
            End Property
            Private m_Count As Integer

            ''' <summary>
            ''' Gets the recorded time value at this index.
            ''' </summary>
            ''' <param name="index">The index of the time value to get, where 0 is the most recently recorded value and Count - 1 is the
            ''' oldest recorded value.</param>
            ''' <returns>The time value at the given index.</returns>
            Default Public ReadOnly Property Item(index As Integer) As Single
                Get
                    If index >= Count Then
                        Throw New IndexOutOfRangeException()
                    End If
                    Dim targetIndex As Integer = marker - 1 - index
                    If targetIndex < 0 Then
                        targetIndex += frameInfos.Length
                    End If
                    Return frameInfos(targetIndex).Time
                End Get
            End Property

            ''' <summary>
            ''' Gets the lowest frame time on record.
            ''' </summary>
            Public ReadOnly Property MinTime() As Single
                Get
                    Dim minTime__1 As Single = Single.MaxValue
                    For i As Integer = 0 To Count - 1
                        If frameInfos(i).Time < minTime__1 Then
                            minTime__1 = frameInfos(i).Time
                        End If
                    Next
                    Return If(Count <> 0, minTime__1, 0)
                End Get
            End Property

            ''' <summary>
            ''' Gets the mean frame time of the whole record.
            ''' </summary>
            Public ReadOnly Property MeanTime() As Single
                Get
                    Dim sumTimes As Single = 0
                    For i As Integer = 0 To Count - 1
                        sumTimes += frameInfos(i).Time
                    Next
                    Return If(Count <> 0, sumTimes / CSng(Count), 0)
                End Get
            End Property

            ''' <summary>
            ''' Gets the highest frame time on record.
            ''' </summary>
            Public ReadOnly Property MaxTime() As Single
                Get
                    Dim maxTime__1 As Single = Single.MinValue
                    For i As Integer = 0 To Count - 1
                        If frameInfos(i).Time > maxTime__1 Then
                            maxTime__1 = frameInfos(i).Time
                        End If
                    Next
                    Return If(Count <> 0, maxTime__1, 0)
                End Get
            End Property

            ''' <summary>
            ''' Gets the lowest frame interval on record.
            ''' </summary>
            Public ReadOnly Property MinInterval() As Single
                Get
                    Dim minInterval__1 As Single = Single.MaxValue
                    For i As Integer = 0 To Count - 1
                        If frameInfos(i).Interval < minInterval__1 Then
                            minInterval__1 = frameInfos(i).Interval
                        End If
                    Next
                    Return If(Count <> 0, minInterval__1, 0)
                End Get
            End Property

            ''' <summary>
            ''' Gets the mean frame interval of the whole record.
            ''' </summary>
            Public ReadOnly Property MeanInterval() As Single
                Get
                    Dim sumIntervals As Single = 0
                    For i As Integer = 0 To Count - 1
                        sumIntervals += frameInfos(i).Interval
                    Next
                    Return If(Count <> 0, sumIntervals / CSng(Count), 0)
                End Get
            End Property

            ''' <summary>
            ''' Gets the highest frame interval on record.
            ''' </summary>
            Public ReadOnly Property MaxInterval() As Single
                Get
                    Dim maxInterval__1 As Single = Single.MinValue
                    For i As Integer = 0 To Count - 1
                        If frameInfos(i).Interval > maxInterval__1 Then
                            maxInterval__1 = frameInfos(i).Interval
                        End If
                    Next
                    Return If(Count <> 0, maxInterval__1, 0)
                End Get
            End Property

            ''' <summary>
            ''' Gets the current frames per second for the frames on record.
            ''' </summary>
            Public ReadOnly Property FramesPerSecond() As Single
                Get
                    Return If(Count <> 0, 1000.0F / MeanInterval, 0)
                End Get
            End Property

            ''' <summary>
            ''' Gets the achievable (i.e. if the frame rate was not limited) frames per second for the frames on record.
            ''' </summary>
            Public ReadOnly Property AchieveableFramesPerSecond() As Single
                Get
                    Return If(Count <> 0, 1000.0F / MeanTime, 0)
                End Get
            End Property

            ''' <summary>
            ''' Gets a printable string of the min, mean and max frame times.
            ''' </summary>
            Public ReadOnly Property FrameTimings() As String
                Get
                    Return "time: " & MinTime.ToString("0.0", CultureInfo.CurrentCulture) & "ms/" & MeanTime.ToString("0.0", CultureInfo.CurrentCulture) & "ms/" & MaxTime.ToString("0.0", CultureInfo.CurrentCulture) & "ms"
                End Get
            End Property

            ''' <summary>
            ''' Gets a printable string of the min, mean and max frame intervals.
            ''' </summary>
            Public ReadOnly Property FrameIntervals() As String
                Get
                    Return "interval: " & MinInterval.ToString("0.0", CultureInfo.CurrentCulture) & "ms/" & MeanInterval.ToString("0.0", CultureInfo.CurrentCulture) & "ms/" & MaxInterval.ToString("0.0", CultureInfo.CurrentCulture) & "ms"
                End Get
            End Property

            ''' <summary>
            ''' Gets a printable string of the garbage collections for generations 0, 1 and 2.
            ''' </summary>
            Public ReadOnly Property CollectionCounts() As String
                Get
                    Dim index As Integer = If(marker - 1 >= 0, marker - 1, Count - 1)
                    Return "Collections: " & frameInfos(index).Gen0Collections & "/" & frameInfos(index).Gen1Collections & "/" & frameInfos(index).Gen2Collections
                End Get
            End Property

            ''' <summary>
            ''' Sets up the desired position and bar sizes for the graph drawn by the DrawGraph() method.
            ''' Returns the resulting Rectangle drawing this graph will occupy.
            ''' </summary>
            ''' <param name="location">The top-left corner the graph should be drawn from.</param>
            ''' <param name="graphHeight">The desired height of the graph.</param>
            ''' <param name="individualBarWidth">The width of each bar for each frame time.</param>
            ''' <param name="barHeightScaleFactor">The scale factor for the height of bars.
            ''' A factor of 1 results in 1 pixel per millisecond if this fits within the desired height.</param>
            ''' <returns>The resulting Rectangle drawing this graph will occupy.</returns>
            Public Function SetGraphingAttributes(location As Point, graphHeight As Integer, individualBarWidth As Integer, barHeightScaleFactor As Single) As Rectangle
                ' Leave a 1 pixel padding on each side.
                graphArea.Width = individualBarWidth * frameInfos.Length + 2
                graphArea.Height = graphHeight
                graphArea.Location = location
                barWidth = individualBarWidth
                barHeightFactor = barHeightScaleFactor

                Return graphArea
            End Function

            ''' <summary>
            ''' Draws a graph and related text output onto the given graphics surface.
            ''' </summary>
            ''' <param name="g">The graphics surface on which to draw the graph.</param>
            ''' <param name="targetTime">The target frame time, in milliseconds. This will be drawn on the graph if specified.</param>
            Public Sub DrawGraph(g As Graphics, targetTime As System.Nullable(Of Double))
                ' Determine the bar height scale, this will adjust if the max time is too high to fit on the graph.
                Dim barHeightScale As Single = barHeightFactor
                Dim height As Single = CSng(graphArea.Height - 2)
                If MaxTime <> 0 AndAlso MaxInterval <> 0 Then
                    barHeightScale = Math.Min(barHeightFactor, Math.Min(height / MaxTime, height / MaxInterval))
                End If

                ' Fill the graph area.
                ' We are leaving a 1 pixel padding on all sides, so the component parts of the graph are often offset by 1.
                g.FillRectangle(graphBackgroundBrush, graphArea)

                ' Start at the oldest record.
                Dim barOffset As Integer = frameInfos.Length - Count
                Dim m As Integer = If(barOffset = 0, marker, 0)
                ' Iterate through the records and draw each time as a bar, and interval as a running line.
                For i As Integer = 0 To Count - 1
                    ' Determine bar color by comparing this record with the previous record.
                    ' The first bar has no previous record, so just get the default color.
                    ' The records array is circular, so we must join the first and last elements when required.
                    Dim timeBarBrush As Brush
                    If i = 0 Then
                        timeBarBrush = FrameInfo.CollectionGenerationPerformedBrush(frameInfos(m), frameInfos(m))
                    Else
                        timeBarBrush = FrameInfo.CollectionGenerationPerformedBrush(frameInfos(m), frameInfos(If(m <> 0, m - 1, Count - 1)))
                    End If

                    ' Determine bar sizes.
                    Dim intervalHeight As Single = frameInfos(m).Interval * barHeightScale
                    Dim timeHeight As Single = frameInfos(m).Time * barHeightScale
                    Dim barLeft As Integer = graphArea.Left + 1 + barOffset + i * barWidth

                    ' Draw the bars.
                    g.FillRectangle(Brushes.Gray, barLeft, graphArea.Bottom - 1 - intervalHeight, barWidth, intervalHeight - timeHeight)
                    g.FillRectangle(timeBarBrush, barLeft, graphArea.Bottom - 1 - timeHeight, barWidth, timeHeight)

                    ' Move to the next record.
                    m += 1
                    If m >= Count Then
                        m = 0
                    End If
                Next

                ' Draw the target time as a horizontal line, if set.
                If targetTime.HasValue Then
                    Dim targetLineHeight As Integer = graphArea.Bottom - 1 - CInt(targetTime * barHeightScale)
                    g.DrawLine(Pens.DarkBlue, graphArea.Left + 1, targetLineHeight, graphArea.Right - 2, targetLineHeight)
                End If

                ' Draw the mean interval time.
                Dim meanIntervalHeight As Integer = graphArea.Bottom - 1 - CInt(Math.Truncate(MeanInterval * barHeightScale))
                g.DrawLine(Pens.DarkGray, graphArea.Left + 1, meanIntervalHeight, graphArea.Right - 2, meanIntervalHeight)

                ' Draw the mean frame time.
                Dim meanTimeHeight As Integer = graphArea.Bottom - 1 - CInt(Math.Truncate(MeanTime * barHeightScale))
                g.DrawLine(Pens.LightGray, graphArea.Left + 1, meanTimeHeight, graphArea.Right - 2, meanTimeHeight)

                ' Draw axis markers.
                Dim markerThickness As Integer = If(barHeightScale < 0.5F, 1, 2)
                For i As Integer = 0 To (graphArea.Height - 1) / barHeightScale Step 10
                    Dim markerLineHeight As Integer = graphArea.Bottom - 1 - CInt(Math.Truncate(i * barHeightScale))
                    Dim markerWidth As Integer = If(i Mod 50 = 0, 12, 4)
                    g.FillRectangle(Brushes.Cyan, graphArea.Left + 1, markerLineHeight - 1, markerWidth, markerThickness)
                Next
            End Sub

            ''' <summary>
            ''' Releases all resources used by this object.
            ''' </summary>
            Public Sub Dispose() Implements IDisposable.Dispose
                graphBackgroundBrush.Dispose()
            End Sub
        End Class
#End Region

        ''' <summary>
        ''' Initializes a new instance of the AnimatorForm class handling the given collection of IAnimatableInstance instances.
        ''' </summary>
        ''' <param name="animatableInstances">The initial collection of IAnimatableInstance to be displayed by the form, which may be null.
        ''' </param>
        Public Sub New(animatableInstances As IEnumerable(Of IAnimatableInstance))
            ' Create a new collection of instances from the given enumeration.
            If animatableInstances Is Nothing Then
                m_instances = New LinkedList(Of IAnimatableInstance)()
            Else
                m_instances = New LinkedList(Of IAnimatableInstance)(animatableInstances)
            End If

            ' Create a wrapper around our collection, so we can safely expose the collection to derived classes.
            m_InstancesCollection = New InstancesCollection(Me, m_instances)

            ' Hook up to form events.
            AddHandler form.MouseMove, New MouseEventHandler(AddressOf GraphicsForm_MouseMove)
            AddHandler form.FormClosed, New FormClosedEventHandler(AddressOf GraphicsForm_FormClosed)

            ' Set the initial display area to the working area on the screen where the form is automatically placed.
            DisplayArea = Screen.GetWorkingArea(form)

            ' Create manual painting handlers.
            manualRender = New PaintEventHandler(Sub(sender As Object, e As PaintEventArgs) bufferedGraphics.Render(e.Graphics))
            manualClear = New PaintEventHandler(Sub(sender As Object, e As PaintEventArgs) e.Graphics.Clear(form.TransparencyKey))

            font = New Font(fontFamily, 18, GraphicsUnit.Pixel)
            postUpdateInvalidRegion.MakeEmpty()

            Topmost = True
            'ShowTimingsGraph = true;

            ' Do a garbage collection so we can free garbage caused by start-up.
            GC.Collect()
            GC.WaitForPendingFinalizers()
        End Sub

        ''' <summary>
        ''' Shows the form and begins animating. Start is called on behalf of all instances currently in the collection.
        ''' </summary>
        Public Overridable Sub Start() Implements IInstanceAnimator.Start
            If Started Then
                Throw New InvalidOperationException("Cannot start a form that has already been started.")
            End If

            ' Scale threshold with screen area.
            ' Larger screens need more ponies before they become sufficiently densely packed that there is payoff in clearing everything.
            threshold = DisplayArea.Width * DisplayArea.Height \ 40000

            ' Windows Forms provides easy ways to double buffer forms.
            ' Double buffering prevents flicker by drawing to a back buffer, then displaying it all at once.
            ' It can be quickly set with DoubleBuffered = true or SetStyle(ControlStyles.OptimizedDoubleBuffer, true).
            ' However this method creates a lot of garbage, so it's better to handle it manually.
            ' We create our own buffer surface, draw to that and render when done.
            ' Allocating a full size buffer will prevent the context from having to create a larger surface when resizing and creating
            ' garbage later. Instead it can reuse portions of the existing surface.
            graphics = form.CreateGraphics()
            graphics.SetClip(DisplayArea)
            bufferedGraphicsContext.MaximumBuffer = DisplayArea.Size
            bufferedGraphics = bufferedGraphicsContext.Allocate(graphics, DisplayArea)

            m_Started = True
            Dim runner As New Thread(New ThreadStart(AddressOf Run))
            For Each instance As IAnimatableInstance In m_instances
                instance.Start(elapsedWatch.Elapsed, Me)
            Next
            form.Show()
            m_CursorLocation = Cursor.Position
            runner.Start()
            elapsedWatch.Start()
        End Sub

        ''' <summary>
        ''' Pauses animation on the form, and optionally hides the form while paused.
        ''' </summary>
        ''' <param name="hide">True to hide the form, false to leave the form visible in the paused state.</param>
        Public Overridable Sub Pause(hide As Boolean)
            If Not Started Then
                Throw New InvalidOperationException("Cannot pause a form that has not been started.")
            End If

            If elapsedWatch.IsRunning Then
                running.Reset()
                tickWatch.[Stop]()
                elapsedWatch.[Stop]()

                If hide Then
                    form.Hide()
                End If
            End If
        End Sub

        ''' <summary>
        ''' Resumes animation on the form, and shows the form is it was hidden while paused.
        ''' </summary>
        Public Overridable Sub [Resume]()
            If Not Started Then
                Throw New InvalidOperationException("Cannot resume a form that has not been started.")
            End If

            If Not elapsedWatch.IsRunning Then
                ' Reset any manual painting that is taking place.
                If manualPainting Then
                    RemoveHandler form.Paint, manualRender
                    AddHandler form.Paint, manualClear
                    form.Invalidate()
                    form.Update()
                    RemoveHandler form.Paint, manualClear
                    form.Location = Point.Empty
                    manualPainting = False
                End If

                form.Show()

                elapsedWatch.Start()
                tickWatch.Start()
                running.[Set]()
            End If
        End Sub

        ''' <summary>
        ''' Runs frame cycles continuously and tracks animation timings.
        ''' </summary>
        Private Sub Run()
            ' Loop whilst not paused.
            While running IsNot Nothing AndAlso running.WaitOne()
                ' Measure the total time between frames. Then restart the watch so we can measure tick time.
                If Not Single.IsNaN(frameTime) Then
                    recorder.Record(frameTime, CSng(tickWatch.Elapsed.TotalMilliseconds))
                End If
                tickWatch.Reset()

                ' Lock to ensure our resources are not modified whilst the tick is in progress.
                SyncLock resourcesObject
                    ' Check to see if our resources still exist, if not then we can assume they were disposed before we acquired the lock.
                    ' If this is the case, it's time to stop animating.
                    If form.IsDisposed Then
                        Return
                    End If

                    ' Run an update and draw cycle for one frame.
                    Tick()
                End SyncLock

                frameTime = CSng(tickWatch.Elapsed.TotalMilliseconds)

                ' Sleep until the next tick should start.
                Dim nextInterval As Single = minimumTickInterval - frameTime
                If nextInterval > 0 Then
                    Thread.Sleep(CInt(Math.Truncate(Math.Ceiling(nextInterval))))
                End If
            End While
        End Sub

        ''' <summary>
        ''' Runs an update and draw cycle then displays the resulting frame.
        ''' </summary>
        Private Sub Tick()
            ' Save the invalid region from last frame.
            preUpdateInvalidRegion.MakeEmpty()
            preUpdateInvalidRegion.Union(postUpdateInvalidRegion)

            ' Reset the post-update clipping region.
            postUpdateInvalidRegion.MakeEmpty()

            ' Lock Instances to prevent the collection being modified until drawing is finished.
            SyncLock Instances
                ' Update all the instances.
                Update()

                ' Apply the clipping region with respect to the screen.
                If m_instances.Count < threshold Then
                    postUpdateInvalidRegion.Intersect(DisplayArea)
                Else
                    postUpdateInvalidRegion.Union(DisplayArea)
                End If

                Dim timingsArea As Rectangle = Rectangle.Empty
                Dim timingsInfo As String = Nothing
                If ShowTimingsGraph Then
                    ' Create info string.
                    timingsInfo = "fps: " & recorder.FramesPerSecond.ToString("0.0", CultureInfo.CurrentCulture) & "/" & MaximumFramesPerSecond.ToString("0.0", CultureInfo.CurrentCulture) & "/" & recorder.AchieveableFramesPerSecond.ToString("0.0", CultureInfo.CurrentCulture)
                    Dim timingsSize As Size = bufferedGraphics.Graphics.MeasureString(timingsInfo, font).ToSize()

                    ' Set location and get area of graph draw.
                    Dim offset As New Point(10, 10)
                    Dim graphLocation As New Point(offset.X, offset.Y + timingsSize.Height)
                    recorderGraphArea = recorder.SetGraphingAttributes(graphLocation, 150, 1, 1.5F)
                    postUpdateInvalidRegion.Union(recorderGraphArea)

                    ' Set location of info string draw.
                    timingsArea = New Rectangle(recorderGraphArea.Left, recorderGraphArea.Top - timingsSize.Height - 1, timingsSize.Width + 1, timingsSize.Height + 1)
                    postUpdateInvalidRegion.Union(timingsArea)
                End If

                ' Determine the bounds of the current draw.
                ' This needs to cover areas to be cleared and areas to be drawn.
                Dim boundsF As RectangleF = RectangleF.Union(preUpdateInvalidRegion.GetBounds(graphics), postUpdateInvalidRegion.GetBounds(graphics))
                ' See if the bounds differ from the current buffer.
                If bufferedGraphics.Graphics.ClipBounds <> boundsF Then
                    '#Region "Resize buffer area."
                    ' Reallocate the buffer at the new size. We'll need to clip the whole area, so it can be filled with the right color. 
                    ' Doing this doesn't have any real effect on our program, but it does help the desktop composition program out.
                    ' dwm.exe (Vista and later) maintains bitmaps of all running programs so it can do the fancy stuff for the Aero theme.
                    ' By redrawing a full size buffer every tick, it needs to update a large area and uses significant CPU time. By
                    ' redrawing a minimally sized buffer, we can reduce the pixel count significantly and thus it will use less CPU.
                    ' Whilst helpful, this is obviously limited by the size we can set the buffer to. For example, if there are graphics
                    ' in two opposite corners then we still need a full size buffer and thus no real savings results. On the plus side
                    ' even small reductions in width and height result in big savings on area (e.g. 90% width and 90% height is only 81%
                    ' of the area), and the saved CPU indirectly benefits our frame time.
                    bufferedGraphics.Dispose()
                    bufferedGraphics = bufferedGraphicsContext.Allocate(graphics, Rectangle.Truncate(boundsF))
                    bufferedGraphics.Graphics.SetClip(DisplayArea)

                    ' Set the quality of drawing.
                    ' As a result of having to use a transparency key, we can't use anti-aliasing functions.
                    ' If we did, semi-transparent pixels would be blended with the form background color.
                    bufferedGraphics.Graphics.InterpolationMode = InterpolationMode.NearestNeighbor
                    '#End Region
                    bufferedGraphics.Graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.SingleBitPerPixelGridFit
                Else
                    ' We can reuse the existing buffer, and so reduce the area needing to be cleared.
                    bufferedGraphics.Graphics.SetClip(preUpdateInvalidRegion, CombineMode.Replace)
                End If

                ' Clear the area to remove old graphics.
                bufferedGraphics.Graphics.Clear(form.TransparencyKey)

                ' Set the clipping area to the region we'll be drawing in for this frame.
                bufferedGraphics.Graphics.SetClip(postUpdateInvalidRegion, CombineMode.Replace)

                If ShowClippingRegion Then
                    ' Get the clipping region as a series of non-overlapping rectangles.
                    Dim invalidRectangles As RectangleF() = postUpdateInvalidRegion.GetRegionScans(identityMatrix)

                    ' Display the clipping rectangles.
                    For Each invalidRectangle As RectangleF In invalidRectangles
                        bufferedGraphics.Graphics.DrawRectangle(Pens.Blue, invalidRectangle.X, invalidRectangle.Y, invalidRectangle.Width - 1, invalidRectangle.Height - 1)
                    Next
                End If

                ' Draw all the instances.
                Draw()

                If ShowTimingsGraph AndAlso recorder.Count <> 0 Then
                    ' Display a graph of frame times and garbage collections.
                    recorder.DrawGraph(bufferedGraphics.Graphics, minimumTickInterval)

                    ' Display how long this frame took.
                    bufferedGraphics.Graphics.DrawString(timingsInfo, font, darkTimingsBrush, timingsArea.Left + 1, timingsArea.Top + 1)
                    bufferedGraphics.Graphics.DrawString(timingsInfo, font, Brushes.White, timingsArea.Left, timingsArea.Top)
                End If
            End SyncLock

            ' Render our buffer to the screen.
            bufferedGraphics.Render()
        End Sub

        ''' <summary>
        ''' Updates the instances.
        ''' </summary>
        Protected Overridable Sub Update()
            For Each instance As IAnimatableInstance In m_instances
                instance.Update(ElapsedTime, Me)

                ' Add the post-update region, we'll need to draw it.
                If m_instances.Count < threshold Then
                    postUpdateInvalidRegion.Union(instance.DrawArea)
                End If
            Next
        End Sub

        ''' <summary>
        ''' Draws the instances.
        ''' </summary>
        Protected Overridable Sub Draw()
            For Each instance As IAnimatableInstance In m_instances
                instance.Draw(bufferedGraphics.Graphics)
            Next
        End Sub

        ''' <summary>
        ''' Raised when the mouse moves.
        ''' Tracks the location of the cursor.
        ''' </summary>
        ''' <param name="sender">The object that raised the event.</param>
        ''' <param name="e">Data about the event.</param>
        Private Sub GraphicsForm_MouseMove(sender As Object, e As MouseEventArgs)
            m_CursorLocation = e.Location
        End Sub

        ''' <summary>
        ''' Toggles the topmost display of the form and recreates the graphics surfaces since they become invalidated.
        ''' </summary>
        ''' <param name="beginManualPainting">If true, indicates the paint method of the form should be hooked on to in order to ensure the
        ''' graphics get redrawn.</param>
        Private Sub ToggleTopmost(beginManualPainting As Boolean)
            ' Lock the resources to ensure a tick does not begin until the revised buffers are ready.
            SyncLock resourcesObject
                ' Toggle the topmost setting.
                form.Invoke(New System.Action(Sub()
                                                  form.TopMost = Not form.TopMost
                                                  form.ShowInTaskbar = Not form.TopMost

                                              End Sub))

                ' Recreate the buffers as they will have become invalid.
                graphics.Dispose()
                graphics = form.CreateGraphics()
                graphics.SetClip(DisplayArea)
                bufferedGraphics.Dispose()
                bufferedGraphics = bufferedGraphicsContext.Allocate(graphics, DisplayArea)
                bufferedGraphics.Graphics.SetClip(DisplayArea)
                bufferedGraphics.Graphics.InterpolationMode = InterpolationMode.NearestNeighbor
                bufferedGraphics.Graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.SingleBitPerPixelGridFit

                If beginManualPainting AndAlso Not manualPainting Then
                    ' Hook up to form painting so our buffer still gets drawn.
                    ' We must also offset the form as rendering the buffer does not account for its transparent edges.
                    form.Location = Point.Truncate(postUpdateInvalidRegion.GetBounds(bufferedGraphics.Graphics).Location)
                    AddHandler form.Paint, manualRender
                    manualPainting = True
                End If
            End SyncLock
        End Sub

        ''' <summary>
        ''' Ends animation and closes the form.
        ''' </summary>
        Public Overridable Sub Close() Implements IInstanceAnimator.Close
            SyncLock resourcesObject
                form.Close()
            End SyncLock
        End Sub

        ''' <summary>
        ''' Raised when the form has been closed.
        ''' Triggers disposal of our resources.
        ''' </summary>
        ''' <param name="sender">The object that raised the event.</param>
        ''' <param name="e">Data about the event.</param>
        Private Sub GraphicsForm_FormClosed(sender As Object, e As FormClosedEventArgs)
            ' Dispose of resources, we need to do this from the event since the form could be closed by an OS event and not just our code.
            ' If the OS did close us, this ensures our resources still get released.
            Dispose()
        End Sub

        ''' <summary>
        ''' Releases all resources used by this object.
        ''' </summary>
        Public Sub Dispose() Implements IDisposable.Dispose
            ' Lock to prevent a tick progressing until we have disposed of everything, otherwise we may dispose of resources whilst a draw
            ' operation is still in progress.
            SyncLock resourcesObject
                form.Dispose()
                graphics.Dispose()
                bufferedGraphics.Dispose()
                bufferedGraphicsContext.Dispose()
                'running.Dispose()
                running = Nothing

                preUpdateInvalidRegion.Dispose()
                postUpdateInvalidRegion.Dispose()

                For Each disposableInstance As IDisposable In m_instances
                    disposableInstance.Dispose()
                Next

                recorder.Dispose()
                identityMatrix.Dispose()
                fontFamily.Dispose()
                font.Dispose()
                darkTimingsBrush.Dispose()

                GC.Collect()
                GC.WaitForPendingFinalizers()
            End SyncLock
        End Sub
    End Class

    ''' <summary>
    ''' Defines an object that can animate a collection of IAnimatableInstances.
    ''' </summary>
    ''' <remarks>
    ''' Notes to Implementers:
    ''' A class implementing this interface should provide some method of accepting a collection of IAnimatableInstances to be drawn. This
    ''' would minimally be provided by at least one constructor overload, or by exposing methods to modify an internal collection.
    ''' <list type="bullet">
    ''' <item><term>DisplayArea</term><description>DisplayArea should be the area on the screen in which IAnimatableInstances can be drawn,
    ''' this should be the value passed to the Update method of each IAnimatableInstance for drawBounds.</description></item>
    ''' <item><term>Started</term><description>Started should indicate if Start has been called.
    ''' </description></item>
    ''' <item><term>Start</term><description>Start should begin all animation. Any instances passed to the class must have Start called on
    ''' them. Implementations are required to call Update at least once on any instance before they Draw it, and calls to Update must been 
    ''' done with a monotonically increasing elapsedTime value. Correct drawBounds and cursorPosition are expected to be given, should a 
    ''' IAnimatableInstance wish to make use of them.</description></item>
    ''' <item><term>Close</term><description>Close should permanently cease all animation. This is intended to be called once the class is
    ''' no longer needed, thus implementations are advised to call cleanup code and Dispose if they implement IDisposable.
    ''' </description></item>
    ''' </list>
    ''' </remarks>
    Public Interface IInstanceAnimator
        ''' <summary>
        ''' Gets or sets the display area.
        ''' </summary>
        Property DisplayArea() As Rectangle
        ''' <summary>
        ''' Gets the location of the cursor.
        ''' </summary>
        ReadOnly Property CursorLocation() As Point
        ''' <summary>
        ''' Gets a value indicating whether animation has been started.
        ''' </summary>
        ReadOnly Property Started() As Boolean
        ''' <summary>
        ''' Starts animation of any given IAnimatableInstances.
        ''' </summary>
        Sub Start()
        ''' <summary>
        ''' Ends animation of all IAnimatableInstances, and closes the IInstanceAnimator.
        ''' </summary>
        Sub Close()
    End Interface

    ''' <summary>
    ''' Defines an object that can be drawn and animated by an IInstanceAnimator.
    ''' </summary>
    ''' <remarks>
    ''' Notes to Implementers:
    ''' <list type="bullet">
    ''' <item><term>Position</term><description>Position may be implementation defined, and is used in the GetClosestInstance method.
    ''' </description></item>
    ''' <item><term>DrawArea</term><description>DrawArea should cover all the regions that drawing this instance will affect. If it does
    ''' not, visual artifacts may result from incorrect clipping of the drawing area.</description></item>
    ''' <item><term>Start</term><description>Start should be called by the class drawing the instance. The given time is the start time
    ''' of the instance, which should act as the zero point for the instance in future update calls. The drawing bounds are provided should
    ''' the instance wish to make use of this information.</description></item>
    ''' <item><term>Update</term><description>Update should be called by the class drawing the instance. Implementations should update
    ''' themselves so their state best represents the given elapsed time, in preparation to be drawn. The class handling drawing may call
    ''' this as it sees fit, and implementations should handle multiple calls before drawing, or changes of frequency in calls, without
    ''' error.</description></item>
    ''' <item><term>Draw</term><description>Draw should be called by the class drawing the instance. The instance should draw itself to the
    ''' given graphics target. The class handling drawing may wish to call this method multiple times (to different targets, for example),
    ''' so implementations should be able to handle this without error.</description></item>
    ''' </list>
    ''' </remarks>
    Public Interface IAnimatableInstance
        ''' <summary>
        ''' Gets the position of the instance.
        ''' </summary>
        Property Position() As Point
        ''' <summary>
        ''' Gets the region that will be affected when this instance is drawn.
        ''' </summary>
        Property DrawArea() As Region
        ''' <summary>
        ''' Starts an instance.
        ''' </summary>
        ''' <param name="startTime">The time the instance is starting from.</param>
        ''' <param name="animator">The IInstanceAnimator that started this instance.</param>
        Sub Start(startTime As TimeSpan, animator As IInstanceAnimator)
        ''' <summary>
        ''' Updates an instance to the given instant in time.
        ''' </summary>
        ''' <param name="elapsedTime">The instant in time to which the instance should set its state.</param>
        ''' <param name="animator">The IInstanceAnimator that raised this update.</param>
        Sub Update(elapsedTime As TimeSpan, animator As IInstanceAnimator)
        ''' <summary>
        ''' Draws the instance to the given graphics surface.
        ''' </summary>
        ''' <param name="target">The graphics surface to draw the instance to.</param>
        Sub Draw(target As Graphics)
    End Interface
End Namespace
