package mechanics;

import game.Root;
import graphics.*;

import io.IOH;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.Queue;

import javax.media.opengl.GL2;
import javax.media.opengl.GLAutoDrawable;
import javax.media.opengl.GLEventListener;

import audio.Audio;

import com.jogamp.newt.event.KeyEvent;
import com.jogamp.opengl.util.texture.Texture;
import com.jogamp.opengl.util.texture.TextureData;

import calc.Calc;
import calc.V3D;
import main.Main;
/**Class managing sorted instance list, global game loop/frame rate limiter and openGL viewport.*/
public class Scene implements GLEventListener
	{
	/**The width and height of the scene view on the screen*/
	public static int width,height;
	
	public static double gravity = 0.5;
	public static double friction = 0.75;
	public static double air = 0.0025;
	
	public static final int PARTICLE_DEPTH = -65536;
	
	//public static GLPbuffer render;
	public static Texture renderTexture,workTexture;
	
	/**Time after rendering(lastDisplayCompleteTime) and time after sleeping(lastDisplayEndTime)*/
	private static long lastDisplayCompleteTime,lastDisplayEndTime;
	/**Increases with Main.time each step. Range: [0,2>. Is guaranteed to be 0 upon reset.*/
	public static double blink;
	/**Increases with Main.time each step. Range: [0,4>. Is guaranteed to be 0 upon reset.*/
	public static double fourstage;
	/**Increases with Main.time each step. Range: [0,8>. Is guaranteed to be 0 upon reset.*/
	public static double eightstage;
	
	/**A list of all instances in the scene, sorted on depth with binary search insertion*/
	private ArrayList<Entity> instances;
	/**Used to allow adding/removing instances during the step event.*/
	private Queue<Entity> stepQueue;
	
	public static int x;
	public static int y;
	
	private static boolean resized = false;

	private static Scene me;

	
	
	public Scene(int w, int h)
		{
		if(me!=null)
			throw new RuntimeException("Duplicate scene!");
		me = this;
		width=w;
		height=h;
		instances = new ArrayList<Entity>();
		lastDisplayCompleteTime = System.currentTimeMillis();
		lastDisplayEndTime = lastDisplayCompleteTime;
		}
	@Override
	public void init(GLAutoDrawable drawable)
		{
		if(GFX.gl != null)
			throw new RuntimeException("Duplicate Scene initialization");
		blink=0;
		fourstage=0;
		eightstage=0;
		//Main.animator.pause();
		GL2 gl = (GL2)drawable.getGL();
		drawable.swapBuffers();
		
		if((!gl.isNPOTTextureAvailable() || !gl.isExtensionAvailable("GL_ARB_texture_non_power_of_two")) && !Main.forcePOTs)
			{
			System.out.println("Non-Power-Of-Two textures not supported by driver. Disabling NPOTs.");
			Main.forcePOTs = true;
			}
		if(!gl.isExtensionAvailable("GL_ARB_vertex_buffer_object") || 
				!(gl.isFunctionAvailable("glGenBuffersARB") &&
				  gl.isFunctionAvailable("glBindBufferARB") &&
				  gl.isFunctionAvailable("glBufferDataARB") &&
				  gl.isFunctionAvailable("glDeleteBuffersARB")))
			{
			System.out.println("VBOs not supported by driver. Disabling VBOs.");
			Main.useVBO = false;
			}
		
		//This must be here because it's the first access to a GL instance:
		Main.gfx = new GFX(gl);
		makeWorkTextures(drawable);
		//This is here for the hell of it
		Main.mtl = new MTL();
		//Add the root instance	
		//new game.derpy_branch.DerpyRoot(new V3D(0,0,0));
        new Root(new V3D(0,0,0));
        
        //Main.frame.setVisible(true);
        if(Main.loadFrame != null)
        	{
        	Main.loadFrame.setVisible(false);
        	Main.loadFrame.dispose();
        	Main.loadFrame = null;
        	Main.loadImage.flush();
        	Main.loadImage = null;
        	}
        Main.animator.start();
        //Main.animator.getThread().setDaemon(true);
        Main.setFullScreen(Main.isFullscreen);
        }

	private void makeWorkTextures(GLAutoDrawable drawable)
		{
		if(Main.bloom!=Main.BLOOM_OFF)
			{
			if(renderTexture!=null)
				renderTexture.destroy(GFX.gl);
			if(workTexture!=null)
				workTexture.destroy(GFX.gl);
			//if(render!=null)
			//	render.destroy();
			
			//if(!drawable.getFactory().canCreateGLPbuffer(Main.device))
			//	System.err.println("No PBuffers! OMG! WTF?");
			//render = drawable.getFactory().createGLPbuffer(Main.device, Main.caps, null, width, height, drawable.getContext());
			int w = (int)Math.pow(2,Math.ceil(Calc.log2(width)));
			int h = (int)Math.pow(2,Math.ceil(Calc.log2(height)));
			renderTexture = new Texture(GFX.gl, new TextureData(Main.profile, GL2.GL_RGB, w,h, 0, GL2.GL_RGB, GL2.GL_UNSIGNED_BYTE, false, false, false, ByteBuffer.allocate(w*h*4), null));
			renderTexture.setTexParameteri(GFX.gl, GL2.GL_TEXTURE_WRAP_S, GL2.GL_REPEAT);
			renderTexture.setTexParameteri(GFX.gl, GL2.GL_TEXTURE_WRAP_T, GL2.GL_REPEAT);
			
			w = (int)Math.pow(2,Math.ceil(Calc.log2(width/2+8)));
			h = (int)Math.pow(2,Math.ceil(Calc.log2(height/2+8)));
			workTexture = new Texture(GFX.gl, new TextureData(Main.profile, GL2.GL_RGB, w,h, 0, GL2.GL_RGB, GL2.GL_UNSIGNED_BYTE, false, false, false, ByteBuffer.allocate(w*h*4), null));
			workTexture.setTexParameteri(GFX.gl, GL2.GL_TEXTURE_WRAP_S, GL2.GL_REPEAT);
			workTexture.setTexParameteri(GFX.gl, GL2.GL_TEXTURE_WRAP_T, GL2.GL_REPEAT);
			
			}
		}
	@Override
	public synchronized void display(GLAutoDrawable drawable)
		{
		GFX.gl = (GL2)drawable.getGL();
		
		if(resized)
			{
			int oldWidth = width;
			int oldHeight = height;
			if(height==0) height = 1;
			double ratio = (double)width/height;
			if(ratio>4d/3)
				width= (int)(height*4d/3);
			else
				height= (int)(width*3d/4);
			x+=(oldWidth-width)/2;
			y+=(oldHeight-height)/2;
			drawable.getGL().glViewport(x, y, width, height);
			makeWorkTextures(drawable);
			resized = false;
			}
		
		beginStep();
		step();
		endStep();
		render();
		
	
		long currentTime = System.currentTimeMillis();
		String fpsStr;
		if(currentTime==lastDisplayCompleteTime)
			fpsStr = "N/A";
		else
			fpsStr = String.valueOf(1000/(currentTime-lastDisplayCompleteTime));

		if(IOH.checkKey(KeyEvent.VK_F2))
			System.out.println(fpsStr+", "+(1000/Main.FPS)+", "+(currentTime-lastDisplayEndTime));
		//Overlay.debug = /*Runtime.getRuntime().freeMemory()+*/"FPS: "+fpsStr+"/30";
		//System.out.println(Math.max(0, 16-(currentTime-lastTime)));
		try {Thread.sleep(Math.max(0, (1000/Main.FPS)-(currentTime-lastDisplayEndTime)));}
		catch (InterruptedException e){}
		lastDisplayCompleteTime = currentTime;
		lastDisplayEndTime = System.currentTimeMillis();
		}
	
	private synchronized void add(Entity e)
		{
		int position = findPositionFor(e.getDepth());
		//Shift above instances up
		for(int i = position; i<instances.size(); i++)
			instances.get(i).shiftListPosition(1);
		//Add to list;
		instances.add(position,e);
		
		if (e.getListPosition()==-1 && stepQueue!=null)//Fresh instance
			{stepQueue.offer(e);}//Add to step queue
		//Set position
		e.setListPosition(position);
		}
	
	public static void addInstance(Entity e)
		{
		if(Thread.currentThread() == Audio.getLine())
			throw new RuntimeException("Cannot instantiate in audio thread!");
		me.add(e);
		}
	
	private synchronized void remove(Entity e)
		{
		int position = e.getListPosition();
		//Shift above instances down
		for(int i = position+1; i<instances.size(); i++)
			instances.get(i).shiftListPosition(-1);
		//Remove from list
		instances.remove(position);
		//instances.remove(e);
		}
	
	public static void removeInstance(Entity e)
		{
		if(Thread.currentThread() == Audio.getLine())
			throw new RuntimeException("Cannot destroy in audio thread!");
		me.remove(e);
		}
	
	private synchronized void reinsert(Entity e)
		{
		if(!e.isDestroyed())
			{
			removeInstance(e);
			addInstance(e);
			}
		}
	public static synchronized void depthChanged(Entity e)
		{
		if(Thread.currentThread() == Audio.getLine())
			throw new RuntimeException("Cannot change depth in audio thread!");
		me.reinsert(e);
		}

	private int findPositionFor(int depth)
		{
		//returns position to insert an element into the list sorted on depth
		int size = instances.size();
		int minimum = 0;
		int maximum = size;
		int it=0;
		int its;
		boolean found=false;
		if(size==0)
		    its = 0;
		else
		    its = (int) (Math.log(size)/Math.log(2));//This could be more effective with bit twiddling but whatever
		int pos=0;
		while(it<its && !found)//This is a modified binary search in case you can't see that
		    {
		    pos=(int)((minimum+maximum)/2);//Floor
		    int currentDepth = instances.get(pos).getDepth();
		    if (currentDepth == depth)
		        found=true;
		    else
		        {
		        if(currentDepth<depth)
		            maximum=pos;
		        else
		            minimum=pos;
		        }
		    it+=1;
		    }
		if(!found && size!=0)
		    {
		    pos=(int)((minimum+maximum)/2);//Floor
		    if(instances.get(pos).getDepth()>depth)
		        pos+=1;
		    }
		return pos;
		}
	private synchronized void render()
		{
		GFX.gl.glClearColor(1,1,1,1);
		GFX.gl.glClear(GL2.GL_COLOR_BUFFER_BIT | 
				  GL2.GL_DEPTH_BUFFER_BIT );
		GFX.gl.setSwapInterval(0);
		
		//It is illegal to add or remove instances during rendering.
		for(int i=0; i<instances.size(); i++)
			{
			Entity e = instances.get(i);
			if(e.visible) 
				e.render();
			}
		}

	private synchronized void step()
		{
		fourstage+=1;
		if(fourstage>=4)
			fourstage=0;
		eightstage+=1;
		if(eightstage>=8)
			eightstage=0;
		blink+=1;
		if(blink>=2)
			blink=0;
		
		int error = GFX.gl.glGetError();
		if(error!=GL2.GL_NO_ERROR)
			{
			System.err.println(GFX.glu.gluErrorString(error));
			}
		IOH.step();
		
		//Instances should be processed in a fixed order to avoid duplicate events due to depth changes.
		stepQueue = new LinkedList<Entity>();
		for(int i=0; i<instances.size(); i++)
			{
			stepQueue.offer(instances.get(i));
			}
		
		while (!stepQueue.isEmpty())
			{
			Entity e = stepQueue.poll();
			if(!e.isDestroyed())
				e.step();
			}
		stepQueue = null;
		}
	private synchronized void beginStep()
		{
		
		// BEGIN STEP:
		stepQueue = new LinkedList<Entity>();
		
		for(int i=0; i<instances.size(); i++)
			{
			stepQueue.offer(instances.get(i));
			}
		
		while (!stepQueue.isEmpty())
			{
			Entity e = stepQueue.poll();
			if(!e.isDestroyed())
				e.beginStep();
			}
		stepQueue = null;
		}
	private synchronized void endStep()
		{
		
		// END STEP:
		stepQueue = new LinkedList<Entity>();
		
		for(int i=0; i<instances.size(); i++)
			{
			stepQueue.offer(instances.get(i));
			}
		
		while (!stepQueue.isEmpty())
			{
			Entity e = stepQueue.poll();
			if(!e.isDestroyed())
				e.endStep();
			}
		if (IOH.checkKeyPressed(KeyEvent.VK_F4))
			Main.setFullScreen(!Main.isFullscreen);
		if (IOH.checkKeyPressed(KeyEvent.VK_ESCAPE))
			System.exit(0);
		IOH.endStep();
		stepQueue = null;
		}
	@Override
	public void reshape(GLAutoDrawable drawable, int x, int y, int width, int height)
		{
		Scene.x = x;
		Scene.y = y;
		Scene.width=width;
		Scene.height=height;
		Scene.resized = true;
		}

	private synchronized void programEnd()
		{
		//Instances should be processed in a temporarily fixed order to avoid duplicate or randomly missing events
		Queue<Entity> q = new LinkedList<Entity>();
		for(int i=0; i<instances.size(); i++)
			{
			q.offer(instances.get(i));
			}
		while (!q.isEmpty())
			{
			Entity e = q.poll();
			if(!e.isDestroyed())
				e.programEnd();
			}
		}
	@Override
	public void dispose(GLAutoDrawable drawable)
		{
		programEnd();
		}
	/**Get all instances*/
	private synchronized Entity[] getAll()
		{
		if(Thread.currentThread() == Audio.getLine())
			throw new RuntimeException("Cannot get instances in audio thread!");
		Entity[] r = new Entity[instances.size()];
		instances.toArray(r);
		return r;
		}
	/**Get all instances*/
	public static Entity[] getInstances()
		{
		return me.getAll();
		}
	/**Destroy everything.*/
	private synchronized void destroyAll()
		{
		if(Thread.currentThread() == Audio.getLine())
			throw new RuntimeException("Cannot destroyAll in audio thread!");
		Queue<Entity> q = new LinkedList<Entity>();
		for(Entity e:instances)
			{
			q.offer(e);
			}
		while(!q.isEmpty())
			{
			q.poll().destroy();
			}
		}
	/**Destroy everything. No exceptions.*/
	public static void destroyInstances()
		{
		me.destroyAll();
		}
	}
