package mechanics;

/**
 * This is an abstract class defining the basic capabilities of objects in a 3D world
 */

import java.util.ArrayList;

import javax.media.opengl.GL2;

import calc.*;
import graphics.*;


public abstract class Entity
	{
	private static long currentID;
	//Do not change these directly(it will break the game)
	private long id;
	/**The current depth(must use get/set)*/
	private int depth;
	/**The current position in the instance list(do not meddle)*/
	private int listPosition;
	/**A flag for identifying instances that were destroyed during step and shouldn't be processed*/
	private boolean destroyed;
	/**A list of instances that should be notified upon destruction of this instance.*/
	private ArrayList<DestructionListener> destructionListeners;
	/**Image transform*/
	public float xScale,yScale;
	/**Angle in radians*/
	public float angle;
	
	//Object properties:
	public V3D pos,speed,lastPos;
	public Tex tex;
	public float[] alarm, color;
	public float image,imageSpeed;
	
	public int orientation;
	
	public BlendMode bm;
	/**If this is false then the render method is not called at all. */
	public boolean visible;
	
	public Entity(V3D pos)
		{
		this.pos=pos;
		orientation = GFX.CENTER;
		id=currentID;
		currentID++;
		lastPos=pos.clone();
		destroyed=false;
		speed = new V3D(0,0,0);
		xScale=1;
		yScale=1;
		angle=0;
		tex = null;
		image = 0;
		imageSpeed=1;
		color = GFX.WHITE.clone();
		alarm = new float[16];
		visible=true;
		listPosition=-1;//Not added yet
		destructionListeners = new ArrayList<DestructionListener>();
		Scene.addInstance(this);
		}
	public void step()
		{
		lastPos.set(pos);
		pos.add(speed);
		if (tex==null)
			image=0;
		else
			{
			image+=imageSpeed;
			boolean called=false;
			while (image>=tex.length)
				{image-=tex.length;
				if(!called)
					{animationEnd();
					called=true;}}
			while (image<0)
				{image+=tex.length;
				if(!called)
					{animationEnd();
					called=true;}}
			}

		for(int i=0; i<alarm.length; i++)
			{
			if(alarm[i]!=0)
				{
				alarm[i]-=1;
				if(alarm[i]<=0)
					{
					alarm[i]=0;
					alarm(i);
					}
				}
			}
		}
	public void endStep()
		{}
	public void alarm(int index)
		{}
	public void animationEnd()
		{}
	public final long getID()
		{
		return id;
		}
	
	public void render()
		{
		if(tex!=null)
			{while (image>=tex.length)
				image-=tex.length;
			while (image<0)
				image+=tex.length;
			if(bm!=null)
				bm.set();
			
			GL2 gl = GFX.gl;
			
			
			gl.glColor4fv(color,0);
			GFX.drawTexture(tex,(int)image,(float)pos.x(),(float)pos.y(),(float)pos.z(),orientation,xScale,yScale,angle);
			gl.glColor4fv(GFX.WHITE,0);
			if(bm!=null)
				BlendMode.NORMAL.set();
			}
		}
	public void programEnd()
		{
		//Program closing
		}
	public void destroy()
		{
		if(!destroyed)
			{
			Scene.removeInstance(this);//Remove from instance list
			destroyed=true;
			for(DestructionListener d: destructionListeners)//Notify
				d.instanceDestroyed(this);
			}
		}
	
	public final void setDepth(int newDepth)
		{
		if(depth!=newDepth)
			{
			depth=newDepth;
			if(listPosition!=-1)
				Scene.depthChanged(this);
			}
		}
	public final int getDepth()
		{
		return depth;
		}
	//Begin list getters/setters
	public final void setListPosition(int position)
		{
		listPosition = position;
		}
	public final int getListPosition()
		{
		return listPosition;
		}
	public final void shiftListPosition(int shift)
		{
		listPosition+=shift;
		}
	//End list getters/setters
	public final boolean isDestroyed()
		{
		return destroyed;
		}
	/**Notify some instance when this is destroyed*/
	public void addDestructionListener(DestructionListener listener)
		{
		destructionListeners.add(listener);
		}
	public void removeDestructionListener(DestructionListener listener)
		{
		destructionListeners.remove(listener);
		}
	public void beginStep()
		{}
	}
