package graphics;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Queue;

import javax.media.opengl.GL;
import javax.media.opengl.GL2;

import main.Main;

import calc.Calc;
import calc.FaceKey;
import calc.V3D;

import com.jogamp.common.nio.Buffers;

public class Model
	{
	private static HashMap<String,Model> models = new HashMap<String,Model>();
	/**This method looks for the model with the given file name.
	 * If it is already in memory, the first instance that was loaded is returned.
	 * Otherwise, it is loaded from disk, and bound to the supplied armature.
	 * If multiple instances are required, use softCopy on the returned instance.*/
	public static Model get(String fileName,Armature armature)
		{
		if(models.containsKey(fileName))
			return models.get(fileName);
		else
			{
			Model m = new Model(fileName,armature);
			models.put(fileName, m);
			return m;
			}
		}
	
	String objName;
	String mtllib;
	HashMap<String,Integer> groupMap;//Useful when parsing the bvh files
	String[] groupNames;//Useful for debug 
	int[] groupIndices;//Mapping obj file indices to armature file indices
	
	float[] recoverVertices;
	float[] vertices;
	float[] transformedVertices;
	int  [][] vertexGroups;//These are not flattened.
	float[][] vertexGroupInfluence;//Normalize in post-processing
	
	float[] normals;//Normalize during reading
	float[] transformedNormals;
	
	float[] texCoords;
	
	int  [] faceTuples;
	
	Material [] faceMaterials;
	int      [] faceMaterialIndices;
	
	Armature armature;
	
    /**Pointer to the indices of the VBOs, if any*/
    int vertexVbo;
	int normalVbo;
	int texCoordVbo;
	int indexVbo;
	/**The buffers containing the data*/
    FloatBuffer vertexData;
    FloatBuffer normalData;
    FloatBuffer texCoordData;
    IntBuffer indexData;

    /**The total space of a single vertex with all its data*/
    int vertexStride;
    /**The position of the color data in the vertex data*/
    int colorPointer;
    /**The position of the coordinate data in the vertex data*/
	int vertexPointer;
	/**The position of the texture data in the vertex data*/
	int texCoordPointer;
	
	/**Flag set to true when the VBO needs to be reloaded due to a deformation(no effect on VA implementation)*/
	private boolean vboOutdated;
	/**Whether to render with materials*/
	public boolean useMaterials;
	/**Provides a hint as to whether this instance is being used at the moment.
	 * Strive to set this to false when the user is destroyed.*/
	public boolean isInUse;
	public int image = 0;
	
	/**Load the obj from the file in the model directory. 
	 * If an armature is supplied, it attempts to bind the vertices to the bones.
	 * Vertex group data must be supplied in the obj file through my modified obj exporter.*/
	private Model(String fname, Armature a)
		{
		isInUse = false;
		useMaterials=true;
		armature = a;
		groupMap = new HashMap<String,Integer>();
		//Initialize temporary data structures to ease parsing
		Queue<String> groupNameQ = new LinkedList<String>(); 
		int groupIndex=0;
		
		//This is an arraylist because we need look up in it for calculating missing normals:
		ArrayList<float[]> vertexQ = new ArrayList<float[]>();
		Queue<int[]> vertexGroupQ = new LinkedList<int[]>();
		Queue<float[]> vertexGroupInfluenceQ = new LinkedList<float[]>();
		vertexQ.add(new float[]{9999,9999,9999});//Note: All face definitions have vertex indices. This is not used.);
		vertexGroupQ.add(null);//This is just for consistency
		vertexGroupInfluenceQ.add(null);
		int vertexIndex=1;
		
		Queue<float[]> normalQ = new LinkedList<float[]>();
		normalQ.offer(new float[]{0,0,1});//This is not used. Normals are calculated instead.
		int normalIndex=1;
		
		Queue<float[]> texCoordQ = new LinkedList<float[]>();
		texCoordQ.offer(new float[]{0,0,0});//This may be used.
		int texCoordIndex=1;
		
		Queue<int[]> faceVertexQ = new LinkedList<int[]>();
		Queue<int[]> faceTexCoordQ = new LinkedList<int[]>();
		Queue<int[]> faceNormalQ = new LinkedList<int[]>();
		int faceIndex=0;
		int numQuads =0;//Remember number of quads we have triangulated.
		
		Queue<Material> faceMaterialQ = new LinkedList<Material>();
		Queue<Integer> faceMaterialIndexQ = new LinkedList<Integer>(); 
		int faceMaterialIndex=0;
		
		File file = new File("res/models/"+fname);
		FileReader fr = null;
		BufferedReader br = null;
		try//Setup readers
			{
			fr = new FileReader(file);
			br = new BufferedReader(fr);
			
			
			String line;
			while((line=br.readLine())!=null)
				{
				if(line.startsWith("o"))
					{
					//Hack: Vertex group names
					line = MTL.strip(line,1);
					objName=line;
					}
				else if(line.startsWith("#@N"))
					{
					//Hack: Vertex group names
					line = MTL.strip(line,3);
					groupMap.put(line, groupIndex);
					groupNameQ.offer(line);
					groupIndex++;
					}
				else if(line.startsWith("vt"))
					{
					//TexCoord definitions
					line = MTL.strip(line,2);
					String[] split = line.split(" ");
					float[] tc = new float[split.length];
					for(int i=0; i<split.length; i++)
						{
						tc[i] = Float.parseFloat(split[i]);
						}
					tc[1]=1f-tc[1];//It's upside down for no apparent reason
					texCoordQ.offer(tc);
					texCoordIndex++;
					}
				else if(line.startsWith("vn"))
					{
					//Normal definitions
					line = MTL.strip(line,2);
					String[] split = line.split(" ");
					float[] n = new float[split.length];
					for(int i=0; i<split.length; i++)
						{
						n[i] = Float.parseFloat(split[i]);
						}
					normalQ.offer(n);
					normalIndex++;
					}
				else if(line.startsWith("f"))
					{
					//face definitions
					line = MTL.strip(line,1);
					String[] split = line.split(" ");
					if(split.length==4)//crappiest triangulation(creates a double entry, remember number of double entries):
						{
						split = new String[]{split[0],split[1],split[2],split[2],split[3],split[0]};
						numQuads++;
						}
						
					int[] fv = new int[split.length];
					int[] ftc = new int[split.length];
					int[] fn = new int[split.length];
					
					int fabricatedNormalIndex=-1;
					
					for(int i=0; i<split.length; i++)
						{
						
						if(i==0 || i==3)
							fabricatedNormalIndex=-1;
						
						String[] v = split[i].split("/",-1);
						for(int j=0; j<3; j++)
							{
							int index;
							if(j>=v.length || v[j].equals(""))
								index = 0;//Default value
							else
								index = Integer.parseInt(v[j]); //Remember: this is 1-indexed by definition.
								//This is OK because 0 is reserved for a default value!
							
							if (j==0)
								{
								//Vertex index
								fv[i]=index;
								}
							else if (j==1)
								{
								//TexCoord index
								ftc[i] = index;
								}
							else if (j==2)
								{
								//Normal index
								//Note: The calculation only works if all vertices and normals are defined first.
								if(index==0)
									{
									if(fabricatedNormalIndex==-1)//Reserve an index
										{
										fabricatedNormalIndex = normalIndex; 
										normalIndex++;
										}
									//Note: the actual normal must be computed below, after the vertices are parsed.
									index=fabricatedNormalIndex;
									}
								if(fabricatedNormalIndex!=-1 && (i==2 || i==5))//Actually calculate the normal if required
									{
									V3D normal =          new V3D(vertexQ.get(fv[i]),0).add(new V3D(vertexQ.get(fv[i-2]),0), -1);
									normal.setCross(normal, new V3D(vertexQ.get(fv[i]),0).add(new V3D(vertexQ.get(fv[i-1]),0), -1));
									normalQ.offer(normal.toFloat());//It's unitized later, so whatever
									}
								fn[i] = index;
								}
							}
						}
					
					faceVertexQ.offer(fv);
					faceTexCoordQ.offer(ftc);
					faceNormalQ.offer(fn);
					faceIndex++;
					}
				else if (line.startsWith("v"))
					{
					//Vertex definition
					
					//Check for a vertex group definition(my personal hack)
					boolean added = false;
					int groupDefIndex = line.indexOf("#@");
					if (groupDefIndex!=-1)
						{
						String groupDef = MTL.strip(line,groupDefIndex+2);
						if(!groupDef.equals(""))
							{
							String[] split = groupDef.split(" ");
							int[] vg = new int[split.length/2];
							float[] vgi = new float[split.length/2];
							for(int i=0; i<split.length; i+=2)
								{
								vg[i/2] = Integer.parseInt(split[i]);
								vgi[i/2] = Float.parseFloat(split[i+1]);
								}
							vertexGroupQ.offer(vg);
							vertexGroupInfluenceQ.offer(vgi);
							added=true;
							}
						}
					if(!added)//Must add something
						{
						vertexGroupQ.offer(null);
						vertexGroupInfluenceQ.offer(null);
						}
					line = MTL.strip(line,1);//This also strips the hack from the line.
					String[] split = line.split(" ");
					float[] v = new float[split.length];
					for(int i=0; i<split.length; i++)
						{
						v[i] = Float.parseFloat(split[i]);
						}
					vertexQ.add(v);
					vertexIndex++;
					}
				else if (line.startsWith("mtllib"))
					{
					mtllib = MTL.strip(line, 6);
					}
				else if (line.startsWith("usemtl"))
					{
					Material mtl = MTL.getMaterial(mtllib,MTL.strip(line, 6));
					if(mtl==null)
						System.err.println("WARNING: material "+mtllib+", "+MTL.strip(line, 6)+" not found!");
					//System.out.println(mtllib+", "+MTL.strip(line, 6));
					faceMaterialQ.offer(mtl);
					faceMaterialIndexQ.offer((faceIndex+numQuads)*3);//number of triangles.
					faceMaterialIndex++;
					}
				}
			}
		catch (IOException e)//Print info for debugging if anything fails
			{
			System.err.println("Model loader fail: "+file);
			e.printStackTrace();
			}
		catch (NumberFormatException e)
			{
			System.err.println("Model loader fail: "+file);
			e.printStackTrace();
			}
		finally//Close readers if they were opened at all
			{
			try {br.close();
				fr.close();}
			catch (Exception e){}
			}
		
		
		//Post-processing:
		int c;
		groupNames = new String[groupIndex];
		groupIndices = new int[groupIndex];
		for(int i=0; i<groupIndex; i++)
			{
			groupNames[i] = groupNameQ.poll();
			
			Bone b = null; 
			if(armature!=null)
				{
				b = armature.getBone(groupNames[i]);
				}	
			if(b==null)
				groupIndices[i]=-1;
			else
				groupIndices[i] = b.index;
			}
		
		//These values are NOT FINAL. 
		//They must be redefined later down because OpenGL does not support multiple IBOs on one mesh:
		vertices = new float[vertexIndex*3];
		vertexGroups = new int[vertexIndex][];
		vertexGroupInfluence = new float[vertexIndex][];
		c=0;
		for(int i=0; i<vertexIndex; i++)
			{
			float[] v = vertexQ.get(i);
			
			for(int j=0; j<3; j++)
				{
				vertices[c] = v[j];
				c++;
				}
			
			int[] vg = vertexGroupQ.poll();
			if(vg==null)
				vg = new int[0];
			vertexGroups[i] = vg;
			float[] vgi = vertexGroupInfluenceQ.poll();
			if(vgi==null)
				vgi = new float[0];
			vertexGroupInfluence[i] = vgi;
			
			if (vertexGroups[i].length>0 && armature!=null)//Normalization, if appliccable
				{
				float total = 0;
				for(int j=0; j<vertexGroups[i].length; j++)
					{
					if(armature.getBone(groupNames[vertexGroups[i][j]])==null)
						vertexGroupInfluence[i][j]=0;//Missing bone in armature
					else
						total+=vertexGroupInfluence[i][j];
					}
				for(int j=0; j<vertexGroups[i].length; j++)
					vertexGroupInfluence[i][j]/=total;
				}
			}
		vertexQ = new ArrayList<float[]>();//Empty it
		vertexQ.add(new float[]{9999,9999,9999});//Readd default values
		vertexGroupQ.add(null);
		vertexGroupInfluenceQ.add(null);
		
		normals = new float[normalIndex*3];
		c=0;
		for(int i=0; i<normalIndex; i++)
			{
			float[] n = Calc.normalize(normalQ.poll());
			for(int j=0; j<3; j++)
				{
				normals[c] = n[j];
				c++;
				}
			}
		normalQ.offer(new float[]{0,0,1});//Readd the default normal
		
		texCoords = new float[texCoordIndex*2];
		c=0;
		for(int i=0; i<texCoordIndex; i++)
			{
			float[] t = texCoordQ.poll();
			for(int j=0; j<2; j++)
				{
				texCoords[c] = t[j];
				c++;
				}
			}
		texCoordQ.offer(new float[]{0,0,0});//Readd the default texCoord
		
		//Get material indexing
		faceMaterials = new Material[faceMaterialIndex];
		faceMaterialIndices = new int[faceMaterialIndex];
		for(int i=0; i<faceMaterialIndex; i++)
			{
			faceMaterials[i] = faceMaterialQ.poll();
			if(faceMaterials[i]==null)
				System.err.println("Material error!");
			faceMaterialIndices[i] = faceMaterialIndexQ.poll();
			}
		
		int uniqueTuples=1;
		c=0;
		faceTuples = new int[(faceIndex+numQuads)*3];//Keep track of entries 
		HashMap<FaceKey,Integer> faces = new HashMap<FaceKey,Integer>();//Keep track of which tuples we have and where they are
		//int nextMaterial = 0;
		for(int i=0; i<faceIndex; i++)
			{
			//These are 1-indexed, with 0 for default value
			
			/*We need to deal with indexing:
			 * Each unique vertex/normal/texCoord/material tuple needs to have its own index 
			 * We have everything listed above.
			 * We are going to readd them to their queues whenever a new tuple is discovered.
			 * */
			
			int[] fv = faceVertexQ.poll();
			int[] ft = faceTexCoordQ.poll();
			int[] fn = faceNormalQ.poll();
			for(int j=0; j<fv.length; j++)
				{
				int v = fv[j];
				int n = fn[j];
				int t = ft[j];
				FaceKey key = new FaceKey(v,n,t);
				Integer position = faces.get(key);
				
				if(position==null)//If new
					{
					position = uniqueTuples;
					faces.put(key, position);
					vertexQ.add(new float[]{vertices[v*3],vertices[v*3+1],vertices[v*3+2]});
					vertexGroupQ.add(vertexGroups[v]);
					vertexGroupInfluenceQ.add(vertexGroupInfluence[v]);
					normalQ.offer(new float[]{normals[n*3],normals[n*3+1],normals[n*3+2]});
					texCoordQ.offer(new float[]{texCoords[t*2],texCoords[t*2+1]});
					uniqueTuples++;
					}
				faceTuples[c] = position;
				
				c++;
				}
			}
		
		//The following have a default value at position 0. These values were added before.
		int vertexCnt = vertexQ.size();
		vertices = new float[vertexCnt*3];
		normals = new float[vertexCnt*3];
		texCoords = new float[vertexCnt*2];
		vertexGroups = new int[vertexCnt][];
		vertexGroupInfluence = new float[vertexCnt][];
		
		int d=0;
		c=0; 
		for(int i=0; i<vertexCnt; i++)
			{
			float[] v = vertexQ.get(i);
			float[] n = normalQ.poll();
			float[] t = texCoordQ.poll();

			for(int j=0; j<3; j++)
				{
				vertices[c] = v[j];
				normals[c] = n[j];
				if(j<2)
					{texCoords[d] = t[j];
					d++;}
				c++;
				}
			
			int[] vg = vertexGroupQ.poll();
			if(vg==null)
				vg = new int[0];
			vertexGroups[i] = vg;
			float[] vgi = vertexGroupInfluenceQ.poll();
			if(vgi==null)
				vgi = new float[0];
			vertexGroupInfluence[i] = vgi;//(already normalized.)
			}
		transformedVertices = vertices.clone();
		transformedNormals = normals.clone();
		setupBuffers();
        bindBuffers();
        vboOutdated=false;
		}
	
	private Model()
		{}
	
	public Model softCopy(Armature armature)
		{
		Model r = new Model();
		
		r.objName = objName;
		r.mtllib = mtllib;
		r.groupMap = groupMap;
		r.groupNames = groupNames; 
		r.groupIndices = groupIndices;
		r.vertices = vertices;
		r.transformedVertices = transformedVertices.clone();
		r.vertexGroups = vertexGroups;
		r.vertexGroupInfluence = vertexGroupInfluence;
		r.normals = normals;
		r.transformedNormals = transformedNormals.clone();
		r.texCoords = texCoords;
		r.faceTuples = faceTuples;
		r.faceMaterials = faceMaterials;
		r.faceMaterialIndices = faceMaterialIndices;
		r.armature = armature;
    	r.vertexStride = vertexStride;
    	r.colorPointer = colorPointer;
		r.vertexPointer = vertexPointer;
		r.texCoordPointer = texCoordPointer;
		r.useMaterials = useMaterials;
		r.setupBuffers();
		r.bindBuffers();
        r.vboOutdated=false;
        r.isInUse = false;
		return r;
		}
	
	private void setupBuffers()
		{
		GL2 gl = GFX.gl;
		// generate VBO handles
		if(Main.useVBO)
			{
			IntBuffer buf = Buffers.newDirectIntBuffer(4);
			gl.glGenBuffers(4, buf);
			vertexVbo = buf.get();
			normalVbo = buf.get();
			texCoordVbo = buf.get();
			indexVbo = buf.get();
			}
	    // generate data buffers
	    vertexData = FloatBuffer.wrap(transformedVertices);
	    normalData = FloatBuffer.wrap(transformedNormals);
	    texCoordData = FloatBuffer.wrap(texCoords);
	    indexData = IntBuffer.wrap(faceTuples);
		}
		
	private void bindBuffers()
		{
		if(Main.useVBO)
			{
			GL2 gl = GFX.gl;
			// transfer data to VBO
	        gl.glBindBuffer(GL.GL_ARRAY_BUFFER, vertexVbo);
	        gl.glBufferData(GL.GL_ARRAY_BUFFER, vertexData.capacity()*Calc.bytesPerFloat, vertexData, GL.GL_DYNAMIC_DRAW);
	        gl.glBindBuffer(GL.GL_ARRAY_BUFFER, 0);
	        
	        gl.glBindBuffer(GL.GL_ARRAY_BUFFER, normalVbo);
	        gl.glBufferData(GL.GL_ARRAY_BUFFER, normalData.capacity()*Calc.bytesPerFloat, normalData, GL.GL_STATIC_DRAW);
	        gl.glBindBuffer(GL.GL_ARRAY_BUFFER, 0);
	        
	        gl.glBindBuffer(GL.GL_ARRAY_BUFFER, texCoordVbo);
	        gl.glBufferData(GL.GL_ARRAY_BUFFER, texCoordData.capacity()*Calc.bytesPerFloat, texCoordData, GL.GL_STATIC_DRAW);
	        gl.glBindBuffer(GL.GL_ARRAY_BUFFER, 0);
	        
	    	gl.glBindBuffer(GL.GL_ELEMENT_ARRAY_BUFFER, indexVbo);
	    	gl.glBufferData(GL.GL_ELEMENT_ARRAY_BUFFER, indexData.capacity()*Calc.bytesPerInt, indexData, GL.GL_STATIC_DRAW);
	    	gl.glBindBuffer(GL.GL_ELEMENT_ARRAY_BUFFER, 0);
			}
		}
	/**Get the reference to the  array of untransformed vertices. This is a flat array, 3 components per vertex.
	 * Changing this array will result in the VBO being updated on the next armature update call, but is generally not advised.*/
	public float[] getVertices()
		{
		return vertices;
		}

	/**Get the reference to the  array of indices. This is a flat array, 3 components per triangle.
	 * Changing this array will result in the VBO being updated on the next armature update call, but is generally not advised.*/
	public int[] getTriangleIndices()
		{
		return faceTuples;
		}
	
	/**
	 * Creates a transformed model from the original model by applying the model's armature.
	 * The armature should be supplied with the constructor.
	 * You should update the armature first in order for this to have any effect.
	 * */
	public void updateToArmature()
		{
		if(armature!=null)
			{
			float[] n = new float[3];
			float[] v = new float[3];
			
			for(int i=0; i<transformedVertices.length; i+=3)
				{
				transformedVertices[i]=0;
				transformedVertices[i+1]=0;
				transformedVertices[i+2]=0;
				transformedNormals[i]=0;
				transformedNormals[i+1]=0;
				transformedNormals[i+2]=0;
				
				for(int j=0; j<vertexGroups[i/3].length; j++)
					{
					if(vertexGroupInfluence[i/3][j]>0)
						{
						Bone b = armature.bones[groupIndices[vertexGroups[i/3][j]]];
						if(b!=null)
							{
							/*float[] n = Calc.rotationMatrixPostMultiply(
								b.units,
								new float[]{normals[i],normals[i+1],normals[i+2]}
								);
							float[] v = Calc.rotationMatrixPostMultiply(
								b.units,
								new float[]{vertices[i]-b.undeformedPosition[0],vertices[i+1]-b.undeformedPosition[1],vertices[i+2]-b.undeformedPosition[2]}
								);*/
							n[0]=normals[i];
							n[1]=normals[i+1];
							n[2]=normals[i+2];
							Calc.rotationMatrixPostMultiplyOpt(b.units,n);
							
							v[0]=vertices[i]-b.undeformedPosition[0];
							v[1]=vertices[i+1]-b.undeformedPosition[1];
							v[2]=vertices[i+2]-b.undeformedPosition[2];
							Calc.rotationMatrixPostMultiplyOpt(b.units,v);

							transformedVertices[i]  +=(b.currentPosition[0]+v[0])*vertexGroupInfluence[i/3][j];
							transformedVertices[i+1]+=(b.currentPosition[1]+v[1])*vertexGroupInfluence[i/3][j];
							transformedVertices[i+2]+=(b.currentPosition[2]+v[2])*vertexGroupInfluence[i/3][j];
							
							transformedNormals[i]  +=n[0]*vertexGroupInfluence[i/3][j];
							transformedNormals[i+1]+=n[1]*vertexGroupInfluence[i/3][j];
							transformedNormals[i+2]+=n[2]*vertexGroupInfluence[i/3][j];
							}
						}
					}
				}
			}
		else
			{
			for(int i=0; i<transformedVertices.length; i+=1)
				transformedVertices[i] = vertices[i];
			}
		vboOutdated = true;
		}
	/**Extend the model along surface normals. Use only when smooth normals exist...*/
	public void extendAlongNormals(float amount)
		{
		//if(armature!=null)
			{
			for(int i=0; i<transformedVertices.length; i+=3)
				{
				transformedVertices[i]+=amount*transformedNormals[i];
				transformedVertices[i+1]+=amount*transformedNormals[i+1];
				transformedVertices[i+2]+=amount*transformedNormals[i+2];
				}
			vboOutdated=true;
			}
		}
	/**
	 * Render the model as it is now.
	 */
	public void render()
		{
		GL2 gl = GFX.gl;
		if(Main.useVBO)
			{
			if(vboOutdated)
				{
				gl.glBindBuffer(GL.GL_ARRAY_BUFFER, vertexVbo);
				gl.glBufferData(GL.GL_ARRAY_BUFFER, transformedVertices.length*Calc.bytesPerFloat, vertexData, GL.GL_DYNAMIC_DRAW);
				gl.glBindBuffer(GL.GL_ARRAY_BUFFER, 0);
				gl.glBindBuffer(GL.GL_ARRAY_BUFFER, normalVbo);
				gl.glBufferData(GL.GL_ARRAY_BUFFER, transformedNormals.length*Calc.bytesPerFloat, normalData, GL.GL_DYNAMIC_DRAW);
				gl.glBindBuffer(GL.GL_ARRAY_BUFFER, 0);
				vboOutdated=false;
				}
			
			gl.glBindBuffer(GL2.GL_ARRAY_BUFFER, texCoordVbo);
			gl.glTexCoordPointer(2,GL.GL_FLOAT, 2*Calc.bytesPerFloat, 0);
			gl.glBindBuffer(GL2.GL_ARRAY_BUFFER, normalVbo);
			gl.glNormalPointer(GL.GL_FLOAT, 3*Calc.bytesPerFloat, 0);
			gl.glBindBuffer(GL2.GL_ARRAY_BUFFER, vertexVbo);
			gl.glVertexPointer(3, GL.GL_FLOAT, 3*Calc.bytesPerFloat, 0);
			
			gl.glBindBuffer(GL2.GL_ELEMENT_ARRAY_BUFFER, indexVbo);
			gl.glEnableClientState(GL2.GL_VERTEX_ARRAY);
			gl.glEnableClientState(GL2.GL_NORMAL_ARRAY);
			gl.glEnableClientState(GL2.GL_TEXTURE_COORD_ARRAY);
			// This is the actual draw command
			int start, end;
			for(int i=0; i<faceMaterials.length; i++)
				{
				start=faceMaterialIndices[i];
				end=(i+1==faceMaterials.length)?faceTuples.length:faceMaterialIndices[i+1];
				if(useMaterials)
					faceMaterials[i].enable(image);
				gl.glDrawElements(GL2.GL_TRIANGLES, end-start, GL2.GL_UNSIGNED_INT, start*Calc.bytesPerInt);
				if(useMaterials)
					faceMaterials[i].disable();
				}
			if(useMaterials)
				faceMaterials[faceMaterials.length-1].disable();
			
			gl.glDisableClientState(GL2.GL_VERTEX_ARRAY);
			gl.glDisableClientState(GL2.GL_NORMAL_ARRAY);
			gl.glDisableClientState(GL2.GL_TEXTURE_COORD_ARRAY);
			
			gl.glBindBuffer(GL.GL_ARRAY_BUFFER, 0);
			gl.glBindBuffer(GL.GL_ELEMENT_ARRAY_BUFFER, 0);
			}
		else
			{
			vboOutdated=false;
			
			gl.glTexCoordPointer(2,GL.GL_FLOAT, 2*Calc.bytesPerFloat, texCoordData);
			gl.glNormalPointer(GL.GL_FLOAT, 3*Calc.bytesPerFloat, normalData);
			gl.glVertexPointer(3, GL.GL_FLOAT, 3*Calc.bytesPerFloat, vertexData);
			
			gl.glEnableClientState(GL2.GL_VERTEX_ARRAY);
			gl.glEnableClientState(GL2.GL_NORMAL_ARRAY);
			gl.glEnableClientState(GL2.GL_TEXTURE_COORD_ARRAY);
			// This is the actual draw command
			int start, end;
			for(int i=0; i<faceMaterials.length; i++)
				{
				start=faceMaterialIndices[i];
				end=(i+1==faceMaterials.length)?faceTuples.length:faceMaterialIndices[i+1];
				int length = end-start;
				IntBuffer currentBuffer = Buffers.newDirectIntBuffer(end-start);
				currentBuffer.put(faceTuples,start,length);
				currentBuffer.rewind();
				
				if(useMaterials)
					faceMaterials[i].enable(0);
				gl.glDrawElements(GL2.GL_TRIANGLES,length, GL2.GL_UNSIGNED_INT, currentBuffer);
				if(useMaterials)
					faceMaterials[i].disable();
				}
			indexData.rewind();
			if(useMaterials)
				faceMaterials[faceMaterials.length-1].disable();
			
			gl.glDisableClientState(GL2.GL_VERTEX_ARRAY);
			gl.glDisableClientState(GL2.GL_NORMAL_ARRAY);
			gl.glDisableClientState(GL2.GL_TEXTURE_COORD_ARRAY);
			}
		}
	
	public void dispose(GL gl) 
	   {
	   if(Main.useVBO)
		   {
		   gl.glDeleteBuffers(1, new int[] { vertexVbo }, 0);
		   gl.glDeleteBuffers(1, new int[] { normalVbo }, 0);
		   gl.glDeleteBuffers(1, new int[] { texCoordVbo }, 0);
		   gl.glDeleteBuffers(1, new int[] { indexVbo }, 0);
		   }
	   }

	/**Set the armature. The armature must have completely identical bones, but may have different animation data.*/
	public void setArmature(Armature a)
		{
		armature = a;
		}

	public void corrupt()
		{
		if(recoverVertices == null)
			recoverVertices = vertices.clone();
		Calc.corruptFloats(vertices,4);
		this.updateToArmature();
		}

	public void recover()
		{
		if(recoverVertices != null)
			System.arraycopy(recoverVertices,0,vertices,0,vertices.length);
		this.updateToArmature();
		}
	}