package graphics;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;

import calc.Calc;

/**
 * This class loads material files and keeps an index. 
 * GFX handles the textures, they're not here.
 */
public class MTL
	{
	private static HashMap<String,ArrayList<Material>> mtllibsList;
	private static HashMap<String,Material> mtllibsStr;
	
	public MTL()
		{
		mtllibsList = new HashMap<String,ArrayList<Material>>();
		mtllibsStr = new HashMap<String,Material>();
		loadDir("res/materials");
		}
	
	private void loadDir(String dirName) 
		{
		File dir = new File(dirName);
		System.out.print("Load Material dir: "+dirName+" - ");
		if(dir.isDirectory()) 
			{
			String filenames[] = dir.list();
			for (String file : filenames)
				{
				if(file.endsWith(".mtl"))
					loadLib(dirName,file);
				}
			}
		System.out.println("... done.");
		}
	
	private void loadLib(String dirName, String name)
		{
		//load and insert into HashMap
		System.out.print(name + " ");
		File file = new File(dirName+"/"+name);
		FileReader fr = null;
		BufferedReader br = null;
		try//Setup readers
			{
			fr = new FileReader(file);
			br = new BufferedReader(fr);
			
			//Create list(0-indexed)
			ArrayList<Material> list = new ArrayList<Material>();
			//Put list in HashMap
			mtllibsList.put(name, list);
			
			//Start with nulls
			int materialIndex=0;
			boolean textureOverride = true;
			Material mtl = null;
			
			String line;
			while((line=br.readLine())!=null)
				{
				if(line.startsWith("#@ENABLECOLORTEXTURE"))
					textureOverride = false;
				else if (line.startsWith("newmtl"))
					{
					line = strip(line,6);
					materialIndex++;
					mtl = new Material(name,line,materialIndex);
					mtl.textureOverride = textureOverride;
					mtllibsStr.put(name+", "+line, mtl);
					list.add(mtl);
					}
				else if (line.startsWith("map_Kd"))
					{
					line = strip(line,6);
					int lastDot = line.lastIndexOf('.');
					if(lastDot!=-1)
						line = line.substring(0,lastDot);
					mtl.diffuseTex = GFX.tex(line);
					}
				else if (line.startsWith("Ka"))
					{
					line = strip(line,2);
					mtl.ambientColor = Calc.getFloatArray(line);
					mtl.setAlpha(mtl.alpha);
					}
				else if (line.startsWith("Kd"))
					{
					line = strip(line,2);
					mtl.diffuseColor = Calc.getFloatArray(line);
					mtl.setAlpha(mtl.alpha);
					}
				else if (line.startsWith("Ks"))
					{
					line = strip(line,2);
					mtl.specularColor = Calc.getFloatArray(line);
					mtl.setAlpha(mtl.alpha);
					}
				else if (line.startsWith("Ke"))
					{
					line = strip(line,2);
					mtl.emitColor = Calc.getFloatArray(line);
					mtl.setAlpha(mtl.alpha);
					}
				else if (line.startsWith("d"))
					{
					line = strip(line,1);
					float a = Float.parseFloat(line);
					mtl.setAlpha(a);
					}
				else if (line.startsWith("Tr"))
					{
					line = strip(line,1);
					float a = Float.parseFloat(line);
					mtl.setAlpha(a);
					}
				}
			}
		catch (IOException e)//Print info for debugging if anything fails
			{
			System.err.println("Material loader fail: "+file);
			e.printStackTrace();
			}
		catch (NumberFormatException e)
			{
			System.err.println("Material loader fail: "+file);
			e.printStackTrace();
			}
		finally//Close readers if they were opened at all
			{
			try {br.close();
				fr.close();}
			catch (Exception e){}
			}
		}

	/**
	 * Strips .obj comments and whitespaces from a string and the first letters
	 * @param line the string to modify
	 * @param start the number of letters to strip from the start
	 */
	public static String strip(String line,int start)
		{
		line = line.substring(start);//Strip away the indicated start
		int commentIndex = line.indexOf("#");//Find remaining comment, if any
		if (commentIndex!=-1)
			{
			line = line.substring(0, commentIndex);
			}
		return line.trim();//Trim away whitespaces
		}
	
	/**
	 * Get a material from a loaded mtllib. 
	 * @param mtllib the file name of the mtllib(including extension)
	 * @param index the index of the material in the lib(1-indexed!)
	 */
	public static Material getMaterial(String mtllib, int index)
		{
		return mtllibsList.get(mtllib).get(index-1);
		}

	public static Material getMaterial(String mtllib, String mtlName)
		{
		return mtllibsStr.get(mtllib+", "+mtlName);
		}
	}
