package graphics;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.image.BufferedImage;
import java.awt.image.WritableRaster;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.imageio.ImageIO;
import javax.media.opengl.GL;
import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;

import com.jogamp.common.nio.Buffers;

import main.Main;

import calc.Calc;

/**
 * This class handles centralized storage of textures and useful drawing routines
 */
public class GFX
	{
	public static GFX gfx;
	public static GL2 gl;//This is set to the current gl instance before drawing anything
	public static GLU glu;
	private static HashMap<String,Tex> textures;
	
	private static int currentCullFace;
	private static float[] projMatrix;
	
	/**reusable HPRs*/
	public static HPR unitSquare;
	public static HPR unitSphere1;
	public static HPR unitSphere2;
	public static HPR unitCube;
	public static HPR unitCylinder1;
	public static HPR unitCylinder2;
	public static HPR unitSquareTesselated2;
	public static HPR unitSquareTesselated4;
	public static HPR unitSquareTesselated8;
	
	/**Colors used when defaulting the shading/material*/
	public static float[] ambientColor = {0.92f, 0.92f, 0.92f, 1.0f};
	public static float[] specularColor = {0f, 0f, 0f, 0f};
	public static float[] emitColor = {0f, 0f, 0f, 1f};
	public static int specularWeight = 32;
	
	/**Lighting*/
	public final static float LIGHT_DIRECTIONAL = 0;//secret GL constants. Sort of.
	public final static float LIGHT_POSITIONAL = 1;
	private static boolean[] lightEnabled;
	private static float[] lightVector;
	private static int zFar;
	private static int zNear;
	
	/**Origins*/
	public final static int CENTER 		= -1;
	public final static int EAST 		= 0;
	public final static int NORTHEAST 	= 1;
	public final static int NORTH 		= 2;
	public final static int NORTHWEST 	= 3;
	public final static int WEST 		= 4;
	public final static int SOUTHWEST 	= 5;
	public final static int SOUTH 		= 6;
	public final static int SOUTHEAST 	= 7;
	
	public static final float[] 
        BLACK = 	new float[]{0,0,0,1},
		WHITE = 	new float[]{1,1,1,1},
		DKGRAY = 	new float[]{0.25f,0.25f,0.25f,1},
		GRAY = 		new float[]{0.5f,0.5f,0.5f,1},
		LTGRAY = 	new float[]{0.75f,0.75f,0.75f,1},
		RED = 		new float[]{1,0,0,1},
		GREEN = 	new float[]{0,1,0,1},
		BLUE = 		new float[]{0,0,1,1},
		YELLOW = 	new float[]{1,1,0,1},
		CYAN = 		new float[]{0,1,1,1},
		MAGENTA = 	new float[]{1,0,1,1},
		ORANGE = 	new float[]{1,0.5f,0,1};
	
	public GFX(GL gl)
		{
		GFX.gl = (GL2)gl;
		
		String[] str = {"GL_EXT_copy_texture",
						"GL_EXT_texture3D", 
						"GL_NV_texgen_reflection",
						"WGL_EXT_pixel_format",
						"GL_ARB_multitexture",
						"GL_ARB_multisample",
						"GL_ARB_vertex_buffer_object",
						"GL_ARB_texture_non_power_of_two",
						"GL_ARB_texture_rectangle",
						"GL_ARB_framebuffer_object",
						"WGL_ARB_pixel_format"};
		for(String s: str)
			System.out.println((s)+": "+gl.isExtensionAvailable(s));
		
		glu = GLU.createGLU(gl);
		textures = new HashMap<String,Tex>();
		gfx = this;
		
		splitStrips("res/strips","res/textures",false);
		splitStrips("res/stripsT","res/textures",true);
		loadDir("res/textures");
		
		lightEnabled = new boolean[8];
		lightVector = new float[8*4];
		defaultShading();
		enableLight(0);
		setLight(0, -0.25f, -0.25f, 0.15f, LIGHT_DIRECTIONAL);
		GFX.setLightColor(0, WHITE);
		
		enableLight(1);
		setLight(1, 0.33f, 0.66f, -0.05f, LIGHT_DIRECTIONAL);
		GFX.setLightColor(1, WHITE);
		
		setLights();
		generateShapes();
		}
	
	public static void corruptTextures()
		{
		Iterator<Map.Entry<String,Tex>> it = textures.entrySet().iterator();
		while (it.hasNext()) 
			{
        	Map.Entry<String,Tex> pairs = (Map.Entry<String,Tex>)it.next();
    		Tex t = pairs.getValue();
    		if(t.isCorruptable)
    			{
    			if(t.recoverData == null)
    				{
	    			t.recoverData = new byte[t.data.length][];
	    			for(int i=0; i<t.data.length; i++)
	    				{
	    				t.recoverData[i] = t.data[i].clone();
	    				}
    				}
    			
	    		for(int i=0; i<t.tex.length; i++)
	    			{
	    			int texture = t.tex[i];
	    			byte[] data = t.data[i];
	    			
	    			Calc.corruptBytes(data,data.length/32,data.length/16,1+(int)(Math.random()*4));
	    			
	    			ByteBuffer buf = ByteBuffer.wrap(data);
	                gl.glBindTexture(GL2.GL_TEXTURE_2D, texture);
	    			gl.glTexSubImage2D(GL2.GL_TEXTURE_2D, 0, 0, 0, t.getWidth(), t.getHeight(), GL2.GL_RGBA, GL2.GL_UNSIGNED_BYTE, buf);
	    			}
    			}
			}
		}
	
	public static void recoverTextures()
		{
		Iterator<Map.Entry<String,Tex>> it = textures.entrySet().iterator();
		while (it.hasNext()) 
			{
        	Map.Entry<String,Tex> pairs = (Map.Entry<String,Tex>)it.next();
    		Tex t = pairs.getValue();
    		if(t.isCorruptable && t.recoverData != null)
    			{
	    		for(int i=0; i<t.tex.length; i++)
	    			{
	    			int texture = t.tex[i];
	    			byte[] data = t.recoverData[i];
	    			
	    			ByteBuffer buf = ByteBuffer.wrap(data);
	                gl.glBindTexture(GL2.GL_TEXTURE_2D, texture);
	    			gl.glTexSubImage2D(GL2.GL_TEXTURE_2D, 0, 0, 0, t.getWidth(), t.getHeight(), GL2.GL_RGBA, GL2.GL_UNSIGNED_BYTE, buf);
	    			}
    			}
			}
		}

	
	/**
	 * Creates an interpolation between the two color arrays
	 */
	public static float[] mix(float[] color1, float[] color2, float factor)
		{
		float invFactor = 1f-factor;
		return new float[]{
				color1[0]*invFactor+color2[0]*factor,
				color1[1]*invFactor+color2[1]*factor,
				color1[2]*invFactor+color2[2]*factor,
				color1[3]*invFactor+color2[3]*factor};
		}
	/**
	 * Returns a new color from the given color with the given alpha component
	 */
	public static float[] alpha(float[] color, float alpha)
		{
		return new float[]{
				color[0],
				color[1],
				color[2],
				alpha};
		}
	/**
	 * Splits all strips in src into single frames and put them in dst
	 * @param lowerLeftTransparent 
	 */
	private void splitStrips(String src,String dst, boolean lowerLeftTransparent)
		{
		File srcDir = new File(src);
		File dstDir = new File(src);
		System.out.print("Split strips from "+src+" to "+dst+" ... ");
		if(srcDir.isDirectory() && dstDir.isDirectory()) 
			{
			String filenames[] = srcDir.list();
			for (String file : filenames)
				{
				splitStrip(src+"/"+file,dst,lowerLeftTransparent);
				}
			}
		System.out.println("... done.");
		}
	
	/**
	 * Loads all textures we can find in the given directory
	 */
	private void loadDir(String dirName)
		{
		File dir = new File(dirName);
		System.out.print("Load graphics dir: "+dirName+" ... ");
		if(dir.isDirectory()) 
			{
			String filenames[] = dir.list();
			for (String file : filenames)
				{
				loadTexture(dirName+"/"+file);
				}
			}		
		System.out.println("... done.");
		}
	/**
	 * Split the strip in the given file and put the results in dst
	 * @param lowerLeftTransparent 
	 */
	public void splitStrip(String src,String dst, boolean lowerLeftTransparent)
		{
		String format = "png";
		String name = new File(src).getName();
		int dot = name.lastIndexOf('.');//Find the last dot
		if (dot==-1)
			return;//don't even try, it has no extension
		String extension = name.substring(dot+1).toLowerCase();//Get the extension
		if (	!extension.equals("png")  && 
				!extension.equals("gif")  &&
				!extension.equals("jpg")  &&
				!extension.equals("jpeg") &&
				!extension.equals("bmp")  &&
				!extension.equals("tif")  &&
				!extension.equals("tiff"))
			return;//don't try
		
		name = name.substring(0,dot);//Strip the extension from the name
		int underscore = name.lastIndexOf('_');//Find the last underscore, if any
		int subImages = 1;//Default is 1
		if (underscore>=0)
			{
			try {
				subImages = Integer.parseInt(name.substring(underscore+1));//Find the number of subimages from the file, if specified
				name = name.substring(0,underscore);//Strip the image number from the name.
				}
			catch (NumberFormatException e) {/*Not a number, continue peacefully*/}
			}
        //Check if the file already exists in dst
		File srcFile = new File(src);
		File firstDstFile = new File(dst+"/"+name+"_"+subImages+"_"+"0."+format);
		if (firstDstFile.exists())
			System.out.print("(re)");
        try {
            BufferedImage img = ImageIO.read(srcFile);
            int w = img.getWidth()/subImages;
            int h = img.getHeight();
            System.out.print("splitting "+name+"("+subImages+") ");
            
            for(int i=0; i<subImages; i++)
            	{
            	BufferedImage srcImage,dstImage;
            	srcImage = img.getSubimage(i*w, 0, w, h);
            	if(lowerLeftTransparent)
            		{
            		dstImage=makeARGBCopy(srcImage);
            		makeTransparent(dstImage,dstImage.getRGB(0,dstImage.getHeight()-1));
            		}
            	else
            		dstImage=srcImage;
            	ImageIO.write(dstImage, format, new File(dst+"/"+name+"_"+subImages+"_"+i+"."+format));
            	}
        	}
        catch (IOException exc) 
        	{exc.printStackTrace();
            System.err.println("Loading strip "+name+" failed miserably");}
		}
	
	private void makeTransparent(BufferedImage image, int color)
		{
		WritableRaster wr = image.getRaster();
		int[] rgb = new int[image.getWidth()*image.getHeight()]; 
		wr.getDataElements(0, 0, image.getWidth(), image.getHeight(), rgb);
		for(int i=image.getWidth()*image.getHeight()-1; i>=0; i--)
			{
			if (rgb[i]==color)
				rgb[i]=0;
			}
		wr.setDataElements(0, 0, image.getWidth(), image.getHeight(), rgb);
		}

	public static BufferedImage makeARGBCopy(Image bi) 
		{
		BufferedImage bia = new BufferedImage(bi.getWidth(null),bi.getHeight(null),BufferedImage.TYPE_INT_ARGB);
		Graphics2D g = bia.createGraphics();
		g.drawImage(bi, 0, 0, null);
		g.dispose();
		return bia;
		}
	public static BufferedImage makeByteBGRCopy(Image bi) 
		{
		BufferedImage bia = new BufferedImage(bi.getWidth(null),bi.getHeight(null),BufferedImage.TYPE_3BYTE_BGR);
		Graphics2D g = bia.createGraphics();
		g.drawImage(bi, 0, 0, null);
		g.dispose();
		return bia;
		}
	
	/**
	 * Loads the texture in the given file
	 */
	public void loadTexture(String fname)
		{
		String name = new File(fname).getName();
		int dot = name.lastIndexOf('.');//Find the last dot
		if (dot==-1)
			return;//don't even try, it has no extension
		String extension = name.substring(dot+1).toLowerCase();//Get the extension
		if (	!extension.equals("png")  && 
				!extension.equals("gif")  &&
				!extension.equals("jpg")  &&
				!extension.equals("jpeg") &&
				!extension.equals("bmp")  &&
				!extension.equals("tif")  &&
				!extension.equals("tiff"))
			return;//don't try
		
		name = name.substring(0,dot);//Strip the extension from the name
		int underscore = name.lastIndexOf('_');//Find the last underscore, if any
		int currentImage = 0;//Default is 0
		if (underscore>=0)
			{
			try {
				currentImage = Integer.parseInt(name.substring(underscore+1));//Find the number of this subimage, if specified
				name = name.substring(0,underscore);//Strip the number from the name.
				}
			catch (NumberFormatException e) {/*Not a number, continue peacefully*/}
			}
		underscore = name.lastIndexOf('_');//Find the last underscore, if any
		int subImages = 1;//Default is 1
		if (underscore>=0)
			{
			try {
				subImages = Integer.parseInt(name.substring(underscore+1));//Find the number of subimages from the file, if specified
				name = name.substring(0,underscore);//Strip the image number from the name.
				}
			catch (NumberFormatException e) 
				{
				//Not a number, meaning there is only a single subimage.
				currentImage=0;
				}
			}
        // Load texture.
        try {
        	System.out.print(name+"("+currentImage+"/"+subImages+") ");
            InputStream stream = new FileInputStream(fname);
            
            BufferedImage bi = ImageIO.read(stream);
            int width = bi.getWidth();
            int height = bi.getHeight();
            
            int textureWidth,textureHeight;
            if(Main.forcePOTs)
            	{//Calculate nearest power of two above
            	textureWidth = (int)Math.pow(2,Math.ceil(Calc.log2(width)));
            	textureHeight = (int)Math.pow(2,Math.ceil(Calc.log2(height)));
            	}
            else
	            {//Use original size
	            textureWidth = width;
	            textureHeight = height;
	            }
            if(bi.getType()!=BufferedImage.TYPE_4BYTE_ABGR)
            	{
            	BufferedImage old = bi;
            	bi = new BufferedImage(old.getWidth(),old.getHeight(),BufferedImage.TYPE_4BYTE_ABGR);
            	Graphics g = bi.createGraphics();
            	g.drawImage(old, 0, 0, null);
            	g.dispose();
            	old.flush();
            	System.out.print("->RGBA8 ");
            	}
            byte[] bytes = new byte[bi.getWidth()*bi.getHeight()*4];
            bi.getRaster().getDataElements(0, 0, bi.getWidth(), bi.getHeight(), bytes);
            ByteBuffer buf = ByteBuffer.wrap(bytes);
            
            Tex tex;
            int[] array;
            byte[][] dataArray;
            if (textures.containsKey(name))
            	{
            	tex = textures.get(name);
            	array = tex.tex;
            	dataArray = tex.data;
            	}
            else
            	{
            	dataArray = new byte[subImages][];
            	array = new int[subImages];
            	tex = new Tex(name,dataArray,array,width,height);
            	textures.put(name, tex);
            	}
            int[] ints = new int[1];
            gl.glGenTextures(1, ints, 0);
            array[currentImage] = ints[0];//TextureIO.newTexture(gl,data);
        	if(dataArray!=null)
        		dataArray[currentImage] = buf.array();
            gl.glBindTexture(GL2.GL_TEXTURE_2D, array[currentImage]);
            if(Main.forcePOTs)
            	{
            	System.out.print("->"+textureWidth+"x"+textureHeight+" ");
            	//Allocate empty data
            	gl.glTexImage2D(GL2.GL_TEXTURE_2D, 0, GL2.GL_RGBA8, textureWidth, textureHeight, 0, GL2.GL_RGBA, GL2.GL_UNSIGNED_BYTE, Buffers.newDirectByteBuffer(textureWidth*textureHeight*4));
            	//Transfer to relevant part of texture
            	gl.glTexSubImage2D(GL2.GL_TEXTURE_2D, 0, 0, 0, width, height, GL2.GL_RGBA, GL2.GL_UNSIGNED_BYTE, buf);
            	tex.xMult = (float)width/textureWidth;
            	tex.yMult = (float)height/textureHeight;
            	}
            else
            	gl.glTexImage2D(GL2.GL_TEXTURE_2D, 0, GL2.GL_RGBA8, textureWidth, textureHeight, 0, GL2.GL_RGBA, GL2.GL_UNSIGNED_BYTE, buf);
            
            gl.glTexParameteri(GL2.GL_TEXTURE_2D,GL2.GL_TEXTURE_WRAP_S, GL2.GL_CLAMP_TO_EDGE);
            gl.glTexParameteri(GL2.GL_TEXTURE_2D,GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_EDGE);
            
            gl.glTexParameteri(GL2.GL_TEXTURE_2D,GL2.GL_TEXTURE_MIN_FILTER, Main.bilinearMin?GL2.GL_LINEAR:GL2.GL_NEAREST);
            gl.glTexParameteri(GL2.GL_TEXTURE_2D,GL2.GL_TEXTURE_MAG_FILTER, Main.bilinearMag?GL2.GL_LINEAR:GL2.GL_NEAREST);
        	}
        catch (IOException exc) 
        	{exc.printStackTrace();
            System.err.println("Loading texture "+name+" failed miserably");}
		}
	/**
	 * Find the Texture array associated with the given name.
	 * */
	public static Tex tex(String textureName)
		{
		Tex t = textures.get(textureName);
		if(t==null)
			System.err.println("Texture "+textureName+" not found!");
		return t;
		}
	/**
	 * Generate some predefined shapes for faster drawing.
	 */
	private void generateShapes()
		{
		unitSquare = genSquare(1);
		unitSquareTesselated2 = genSquareTesselated(2,1);
		unitSquareTesselated4 = genSquareTesselated(4,1);
		unitSquareTesselated8 = genSquareTesselated(8,1);
		
		unitCube = genCube(1);
		
		unitSphere1 = genSphere(8,6);
		unitSphere2 = genSphere(20,16);
		
		unitCylinder1 = genCylinder(6);
		unitCylinder2 = genCylinder(16);
		//Sample code:
		/*square = gl.glGenLists(1);
		gl.glNewList(square, GL.GL_COMPILE);
		
		gl.glBegin(GL.GL_TRIANGLE_STRIP);
		gl.glTexCoord2f(0, 0);
		gl.glVertex2f(0, 0);
		gl.glTexCoord2f(1, 0);
		gl.glVertex2f(1, 0);
		gl.glTexCoord2f(0, 1);
		gl.glVertex2f(0, 1);
		gl.glTexCoord2f(1, 1);
		gl.glVertex2f(1, 1);
		gl.glEnd();
		
		gl.glEndList();*/
		//Display lists are deprecated
		}
	public static HPR genSquare(int textureRepeat)
		{
		HPR o = genHPR(GL2.GL_STATIC_DRAW, GL2.GL_TRIANGLE_STRIP, true, false, false, true, false);
		o.addVertex(new Vertex(
				new float[]{-1,1,0},
				new float[]{0,0,1},
				null,
				new float[]{0,textureRepeat},
				null
				));
		o.addVertex(new Vertex(
				new float[]{-1,-1,0},
				new float[]{0,0,1},
				null,
				new float[]{0,0},
				null
				));
		o.addVertex(new Vertex(
				new float[]{1,1,0},
				new float[]{0,0,1},
				null,
				new float[]{textureRepeat,textureRepeat},
				null
				));
		o.triangle(false);
		o.addVertex(new Vertex(
				new float[]{1,-1,0},
				new float[]{0,0,1},
				null,
				new float[]{textureRepeat,0},
				null
				));
		o.addIndex(o.getIndex()-1);
		o.end();
		return o;
		}
	public static HPR genSquareTesselated(int tesselation,int textureRepeat)
		{
		HPR o = genHPR(GL2.GL_STATIC_DRAW, GL2.GL_TRIANGLES, true, false, false, true, false);
		for(float y=-1; y<=1; y+=2d/tesselation)
			{
			for(float x=-1; x<=1; x+=2d/tesselation)
				{
				o.addVertex(new Vertex(
				new float[]{x,y,0},
				new float[]{0,0,1},
				null,
				new float[]{(x+1)/2*textureRepeat,(y+1)/2*textureRepeat},
				null
				));
				if(x<1 && y<1)
					{
					int i = (int)((x+1)*(tesselation)/2);
					int j = (int)((y+1)*(tesselation)/2);
					o.addIndex(i+(j+1)*(tesselation+1));
					o.addIndex(i+(j)*(tesselation+1));
					o.addIndex(i+1+(j)*(tesselation+1));
					
					o.addIndex(i+1+(j+1)*(tesselation+1));
					o.addIndex(i+(j+1)*(tesselation+1));
					o.addIndex(i+1+(j)*(tesselation+1));
					}
				}
			}
		o.end();
		return o;
		}

	public static HPR genCube(int textureRepeat)
		{
		HPR o = genHPR(GL2.GL_STATIC_DRAW, GL2.GL_TRIANGLES, true, false, false, true, false);
		
		o.addVertex(new Vertex(
				new float[]{-1,-1,-1},
				new float[]{-1,0,0},
				null,new float[]{0,textureRepeat},
				null));
		o.addVertex(new Vertex(
				new float[]{-1,-1,1},
				new float[]{-1,0,0},
				null,new float[]{0,0},
				null));
		o.addVertex(new Vertex(
				new float[]{-1,1,-1},
				new float[]{-1,0,0},
				null,new float[]{textureRepeat,textureRepeat},
				null));
		o.addVertex(new Vertex(
				new float[]{-1,1,1},
				new float[]{-1,0,0},
				null,new float[]{textureRepeat,0},
				null));
		o.quad(false);
		
		o.addVertex(new Vertex(
				new float[]{1,-1,-1},
				new float[]{1,0,0},
				null,new float[]{textureRepeat,textureRepeat},
				null));
		o.addVertex(new Vertex(
				new float[]{1,-1,1},
				new float[]{1,0,0},
				null,new float[]{textureRepeat,0},
				null));
		o.addVertex(new Vertex(
				new float[]{1,1,-1},
				new float[]{1,0,0},
				null,new float[]{0,textureRepeat},
				null));
		o.addVertex(new Vertex(
				new float[]{1,1,1},
				new float[]{1,0,0},
				null,new float[]{0,0},
				null));
		o.quad(true);
		
		o.addVertex(new Vertex(
				new float[]{-1,-1,-1},
				new float[]{0,-1,0},
				null,new float[]{0,textureRepeat},
				null));
		o.addVertex(new Vertex(
				new float[]{-1,-1,1},
				new float[]{0,-1,0},
				null,new float[]{0,0},
				null));
		o.addVertex(new Vertex(
				new float[]{1,-1,-1},
				new float[]{0,-1,0},
				null,new float[]{textureRepeat,textureRepeat},
				null));
		o.addVertex(new Vertex(
				new float[]{1,-1,1},
				new float[]{0,-1,0},
				null,new float[]{textureRepeat,0},
				null));
		o.quad(true);
		
		o.addVertex(new Vertex(
				new float[]{-1,1,-1},
				new float[]{0,1,0},
				null,new float[]{textureRepeat,textureRepeat},
				null));
		o.addVertex(new Vertex(
				new float[]{-1,1,1},
				new float[]{0,1,0},
				null,new float[]{textureRepeat,0},
				null));
		o.addVertex(new Vertex(
				new float[]{1,1,-1},
				new float[]{0,1,0},
				null,new float[]{0,textureRepeat},
				null));
		o.addVertex(new Vertex(
				new float[]{1,1,1},
				new float[]{0,1,0},
				null,new float[]{0,0},
				null));
		o.quad(false);
		
		o.addVertex(new Vertex(
				new float[]{-1,-1,1},
				new float[]{0,0,1},
				null,new float[]{0,0},
				null));
		o.addVertex(new Vertex(
				new float[]{1,-1,1},
				new float[]{0,0,1},
				null,new float[]{textureRepeat,0},
				null));
		o.addVertex(new Vertex(
				new float[]{-1,1,1},
				new float[]{0,0,1},
				null,new float[]{0,textureRepeat},
				null));
		o.addVertex(new Vertex(
				new float[]{1,1,1},
				new float[]{0,0,1},
				null,new float[]{textureRepeat,textureRepeat},
				null));
		o.quad(false);
		
		o.addVertex(new Vertex(
				new float[]{-1,-1,-1},
				new float[]{0,0,-1},
				null,new float[]{0,0},
				null));
		o.addVertex(new Vertex(
				new float[]{1,-1,-1},
				new float[]{0,0,-1},
				null,new float[]{textureRepeat,0},
				null));
		o.addVertex(new Vertex(
				new float[]{-1,1,-1},
				new float[]{0,0,-1},
				null,new float[]{0,textureRepeat},
				null));
		o.addVertex(new Vertex(
				new float[]{1,1,-1},
				new float[]{0,0,-1},
				null,new float[]{textureRepeat,textureRepeat},
				null));
		o.quad(true);
		o.end();
		return o;
		}

	public static HPR genCylinder(int u)
		{
		HPR o;
		o = genHPR(GL2.GL_STATIC_DRAW, GL2.GL_TRIANGLES, true, false, false, true, false);
		for(int i=0; i<=u; i++)
			{
			float x,y;
			x=(float) Math.cos((double)i*Math.PI*2/u);
			y=(float)-Math.sin((double)i*Math.PI*2/u);
			o.addVertex(new Vertex(
					new float[]{x,y,1},
					new float[]{x,y,1},
					null,
					new float[]{(float)i/u,0},
					null
				));
			o.addVertex(new Vertex(
					new float[]{x,y,-1},
					new float[]{x,y,-1},
					null,
					new float[]{(float)i/u,1},
					null
				));
			}
		int top = o.getIndex();
		o.addVertex(new Vertex(
				new float[]{0,0,1},
				new float[]{0,0,1},
				null,
				new float[]{0.5f,0},
				null
				));
		int bottom = o.getIndex();
		o.addVertex(new Vertex(
				new float[]{0,0,-1},
				new float[]{0,0,-1},
				null,
				new float[]{0.5f,1},
				null
				));
		
		for(int i=0; i<u; i++)
			{
			o.triangle(i*2,top,(i+1)*2);
			o.triangle(bottom,i*2+1,i*2+3);
			o.quad(i*2+1,i*2+3,
					i*2,(i+1)*2);
			}
		o.end();
		
		return o;
		}
	public static HPR genSphere(int u,int v)
		{
		HPR o;
		o = genHPR(GL2.GL_STATIC_DRAW, GL2.GL_TRIANGLES, true, false, false, true, false);
		for(int j=1; j<v; j++)
			{
			for(int i=0; i<=u; i++)
				{
				float x,y,z,w;
				z=(float)Math.cos((double)j*Math.PI/v);
				w=(float) Math.sin((double)j*Math.PI/v);
				x=(float) Math.cos((double)i*Math.PI*2/u)*w;
				y=(float)-Math.sin((double)i*Math.PI*2/u)*w;
				o.addVertex(new Vertex(
				new float[]{x,y,z},
				new float[]{x,y,z},
				null,
				new float[]{(float)i/u,(float)j/v},
				null
				));
				}
			}
		int top = o.getIndex();
		o.addVertex(new Vertex(
				new float[]{0,0,1},
				new float[]{0,0,1},
				null,
				new float[]{0.5f,0},
				null
				));
		int bottom = o.getIndex();
		o.addVertex(new Vertex(
				new float[]{0,0,-1},
				new float[]{0,0,-1},
				null,
				new float[]{0.5f,1},
				null
				));
		
		for(int j=0; j<v; j++)
			{
			for(int i=0; i<u; i++)
				{
				if(j==0)
					o.triangle(i,top,i+1);
				else if (j==v-1)
					o.triangle(bottom,(j-1)*(u+1)+i,(j-1)*(u+1)+i+1);
				else
					o.quad((j)*(u+1)+i,(j)*(u+1)+i+1,
									 (j-1)*(u+1)+i,(j-1)*(u+1)+i+1);
				}
			}
		o.end();
		
		return o;
		}

	public static HPR genHPR(int updateMode, int faceMode, boolean useNormal, boolean useColor, boolean useAlpha, boolean useTexCoord,boolean use3DTexCoord)
		{
		if(Main.useVBO && (!Main.alwaysUseDisplayLists || updateMode != GL2.GL_STATIC_DRAW))
			return new VBO(updateMode,faceMode,useNormal,useColor,useAlpha,useTexCoord,use3DTexCoord);
		else
			{
			if(updateMode == GL2.GL_STATIC_DRAW)
				return new DL(faceMode,useNormal,useColor,useAlpha,useTexCoord,use3DTexCoord);
			else
				return new VA(faceMode,useNormal,useColor,useAlpha,useTexCoord,use3DTexCoord);
			}
		}

	public static HPR genHPRFrom(HPR model, int initialFaceMode)
		{
		if (model instanceof VBO)
			return new VBO((VBO)model,initialFaceMode);
		else if(model instanceof VA)
			return new VA((VA)model,initialFaceMode);
		else
			return new DL((DL)model,initialFaceMode);
		}

	/**
	 * Draw the given rectangle with the given subImage of the texture associated with textureName
	 */
	public static void drawRectangle(float x1,float y1, float x2, float y2, String textureName, int subImage)
		{
		Tex tex = null;
		if (textureName!=null)
			{
			//Find the texture
			tex = tex(textureName);
			}
		drawRectangle(x1,y1,x2,y2,tex,subImage);
		}

	public static void drawRectangle(float x1, float y1, float x2, float y2, Tex tex, int subImage)
		{
		if(tex!=null)
			{
			tex.enable(subImage);
			}
		/*gl.glPushMatrix();
		gl.glScalef(x2-x1,y2-y1,1);
		gl.glTranslatef(x1/(x2-x1),y1/(y2-y1), 0);
		gl.glCallList(square);
		gl.glPopMatrix();*/
		
		//Immediate mode is faster when drawing a single rectangle.
		gl.glBegin(GL.GL_TRIANGLE_STRIP);
		gl.glTexCoord2f(0, tex.yMult);
		gl.glVertex2f(x1, y2);
		gl.glTexCoord2f(tex.xMult, tex.yMult);
		gl.glVertex2f(x2, y2);
		gl.glTexCoord2f(0, 0);
		gl.glVertex2f(x1, y1);
		gl.glTexCoord2f(tex.xMult, 0);
		gl.glVertex2f(x2, y1);
		gl.glEnd();
        if (tex!=null)
        	tex.disable();
		}
	
	public static void drawRectangle(float x1, float y1, float x2, float y2)
		{		
		//Immediate mode is faster when drawing a single rectangle.
		gl.glBegin(GL.GL_TRIANGLE_STRIP);
		gl.glVertex2f(x1, y2);
		gl.glVertex2f(x2, y2);
		gl.glVertex2f(x1, y1);
		gl.glVertex2f(x2, y1);
		gl.glEnd();
		}
	
	public static void drawRectangle(float x1, float y1, float x2, float y2, float tx1, float ty1, float tx2, float ty2, Tex tex, int subImage)
		{
		tex.enable(subImage);
		//Immediate mode is faster when drawing a single rectangle.
		gl.glBegin(GL.GL_TRIANGLE_STRIP);
		gl.glTexCoord2f(tex.xMult*tx1, tex.yMult*ty2);
		gl.glVertex2f(x1, y2);
		gl.glTexCoord2f(tex.xMult*tx2, tex.yMult*ty2);
		gl.glVertex2f(x2, y2);
		gl.glTexCoord2f(tex.xMult*tx1, tex.yMult*ty1);
		gl.glVertex2f(x1, y1);
		gl.glTexCoord2f(tex.xMult*tx2, tex.yMult*ty1);
		gl.glVertex2f(x2, y1);
		gl.glEnd();
        tex.disable();
		}
	
	public static void drawRectangle(float x1, float y1, float x2, float y2, float tx1, float ty1, float tx2, float ty2)
		{
		//Immediate mode is faster when drawing a single rectangle.
		gl.glBegin(GL.GL_TRIANGLE_STRIP);
		gl.glTexCoord2f(tx1, ty2);
		gl.glVertex2f(x1, y2);
		gl.glTexCoord2f(tx2, ty2);
		gl.glVertex2f(x2, y2);
		gl.glTexCoord2f(tx1, ty1);
		gl.glVertex2f(x1, y1);
		gl.glTexCoord2f(tx2, ty1);
		gl.glVertex2f(x2, y1);
		gl.glEnd();
		}
	
	/**@param angle In radians*/
	public static void drawTexture(Tex tex, int subImage, float x, float y, float z, int orientation, float xscale, float yscale, float angle)
		{
		//System.out.println("render texture "+tex+" at "+x+", "+y);
		Point o = tex.getOrigin(orientation);
		int l,t,r,b;
		l = o.x;
		t = o.y;
		r = tex.getWidth()-l;
		b = tex.getHeight()-t;
		
		boolean invert = (xscale<0)^(yscale<0);
		
		float h1 = (float)  Math.cos(angle);
		float v1 = (float) -Math.sin(angle);
		float h2 = -v1;
		float v2 =  h1;
		h1*=xscale;
		v1*=xscale;
		h2*=yscale;
		v2*=yscale;
		
		tex.enable(subImage);
		//Immediate mode is faster when drawing a single rectangle.
		gl.glBegin(GL.GL_TRIANGLE_STRIP);
		
		gl.glNormal3f(0,0,1);
		if(!invert)
			{
			gl.glTexCoord2f(0, 0);
			gl.glVertex3f(x-l*h1-t*h2, y-l*v1-t*v2, z);
		
			gl.glTexCoord2f(tex.xMult, 0);
			gl.glVertex3f(x+r*h1-t*h2, y+r*v1-t*v2 ,z);
			}
		gl.glTexCoord2f(0, tex.yMult);
		gl.glVertex3f(x-l*h1+b*h2, y-l*v1+b*v2 ,z);
		
		gl.glTexCoord2f(tex.xMult, tex.yMult);
		gl.glVertex3f(x+r*h1+b*h2, y+r*v1+b*v2 ,z);
		if(invert)
			{
			gl.glTexCoord2f(0, 0);
			gl.glVertex3f(x-l*h1-t*h2, y-l*v1-t*v2, z);
		
			gl.glTexCoord2f(tex.xMult, 0);
			gl.glVertex3f(x+r*h1-t*h2, y+r*v1-t*v2 ,z);
			}
		gl.glEnd();
		
        tex.disable();
		}

	public static void swapCullFace()
		{
		if(currentCullFace == GL.GL_FRONT)
			{
			currentCullFace = GL2.GL_BACK;
			gl.glCullFace(currentCullFace);
			}
		else
			{
			currentCullFace = GL2.GL_FRONT;
			gl.glCullFace(currentCullFace);
			}
		}
	/**implementation-independent z offset*/
	public static void loadOffsetMatrix(float delta, float pz)
		{
		projMatrix = new float[16];
		gl.glGetFloatv(GL2.GL_PROJECTION_MATRIX, projMatrix, 0);
		gl.glMatrixMode(GL2.GL_PROJECTION);
		float epsilon = -2f*zFar*zNear*delta/((zFar+zNear)*pz*(pz+delta));
		float temp = projMatrix[10];
		projMatrix[10] *= 1f+epsilon;
		gl.glLoadMatrixf(projMatrix,0);
		projMatrix[10] = temp;
		gl.glMatrixMode(GL2.GL_MODELVIEW);
		}
	
	public static void resetOffsetMatrix()
		{
		gl.glMatrixMode(GL2.GL_PROJECTION);
		gl.glLoadMatrixf(projMatrix,0);
		gl.glMatrixMode(GL2.GL_MODELVIEW);
		}
	
	/**Enable the light with the given index(0-7).*/
	public static void enableLight(int lightIndex)
		{
		lightEnabled[lightIndex]=true;
		gl.glEnable(GL2.GL_LIGHT0+lightIndex);
		}
	/**Disable the light with the given index(0-7).*/
	public static void disableLight(int lightIndex)
		{
		lightEnabled[lightIndex]=false;
		gl.glDisable(GL2.GL_LIGHT0+lightIndex);
		}
	/**Set the position or direction and light type for the light with the given index(0-7). Takes effect when using setLights()
	 * @param lightType LIGHT_DIRECTIONAL (0.0f) or LIGHT_POSITIONAL (1.0f).
	 * */
	public static void setLight(int lightIndex, float x, float y, float z, float lightType)
		{
		lightVector[lightIndex*4]  =x;
		lightVector[lightIndex*4+1]=y;
		lightVector[lightIndex*4+2]=z;
		lightVector[lightIndex*4+3]=lightType;
		}
		/**Set the color for the light with the given index(0-7).
		 * This method sets diffuse and specular to the given color and ambient to black.*/
	public static void setLightColor(int lightIndex, float[] color)
		{
        gl.glLightfv(GL2.GL_LIGHT0+lightIndex, GL2.GL_AMBIENT, BLACK,0);
        gl.glLightfv(GL2.GL_LIGHT0+lightIndex, GL2.GL_DIFFUSE, color,0);
        gl.glLightfv(GL2.GL_LIGHT0+lightIndex, GL2.GL_SPECULAR, color,0);
		}
	public static void setLightAttenuation(int lightIndex, float constant, float linear, float quadratic)
		{
		gl.glLightf(GL2.GL_LIGHT0+lightIndex, GL2.GL_CONSTANT_ATTENUATION, constant);
        gl.glLightf(GL2.GL_LIGHT0+lightIndex, GL2.GL_LINEAR_ATTENUATION, linear);
        gl.glLightf(GL2.GL_LIGHT0+lightIndex, GL2.GL_QUADRATIC_ATTENUATION, quadratic);
		}
	/**
	 * Call this after setting the camera to fix the light positions. 
	 */
	public static void setLights()
		{
		for(int i=0; i<8; i++)
			{
			if(lightEnabled[i])
				gl.glLightfv(GL2.GL_LIGHT0+i, GL2.GL_POSITION, lightVector ,i*4);
			}
		}

	public static void imageToByteBufferRGB(String string, ByteBuffer buf)
		{
		File f = new File(string);
		try
			{
			BufferedImage b = ImageIO.read(f);
			b = GFX.makeByteBGRCopy(b);
			int w=b.getWidth(),h=b.getHeight();
			byte[] data = new byte[w*h*3];
			b.getRaster().getDataElements(0,0,w,h, data);
			for(int i=0; i<data.length; i++)
				buf.put(data[i]);
			b.flush();
			}
		catch (IOException e)
			{System.err.println("Image loading failure: "+string);
			e.printStackTrace();}
		
		}

	public static void imageToByteBufferRGBA(String string, ByteBuffer buf)
		{
		File f = new File(string);
		try
			{
			BufferedImage b = ImageIO.read(f);
			b = GFX.makeARGBCopy(b);
			int w=b.getWidth(),h=b.getHeight();
			int[] data = new int[w*h];
			b.getRaster().getDataElements(0,0,w,h, data);
			for(int i=0; i<data.length; i++)
				{
				buf.put((byte)(((data[i]&0x00FF0000))>>>16));
				buf.put((byte)(((data[i]&0x0000FF00))>>>8));
				buf.put((byte)(((data[i]&0x000000FF))));
				buf.put((byte)(((data[i]&0xFF000000))>>>24));
				}
			b.flush();
			}
		catch (IOException e)
			{System.err.println("Image loading failure: "+string);
			e.printStackTrace();}
		
		}
	

	public static void defaultSetup()
		{
		//Culling
		gl.glEnable(GL.GL_CULL_FACE);
		gl.glCullFace(GL.GL_FRONT);
		currentCullFace = GL.GL_FRONT;
		
		//Alpha function(normally disabled)
		gl.glAlphaFunc(GL.GL_GREATER, 0);
		
		
		//Perspective
		gl.glEnable(GL.GL_DEPTH_TEST);
        gl.glDepthFunc(GL.GL_LEQUAL);
        gl.glHint(GL2.GL_PERSPECTIVE_CORRECTION_HINT, GL.GL_NICEST);
        gl.glClearDepth(1.0f);
		}
	public static void defaultShading()
		{
		//Blend modes
        gl.glShadeModel(GL2.GL_SMOOTH);
        
        //Lighting
        gl.glColorMaterial(GL.GL_FRONT_AND_BACK, GL2.GL_AMBIENT_AND_DIFFUSE );//Specifies what glColor does
        gl.glEnable(GL2.GL_COLOR_MATERIAL );
        gl.glEnable(GL2.GL_NORMALIZE);
        
        //Material
        gl.glMaterialfv(GL.GL_FRONT_AND_BACK, GL2.GL_SPECULAR, specularColor,0);
        gl.glMateriali(GL.GL_FRONT_AND_BACK,GL2.GL_SHININESS,specularWeight);
        gl.glMaterialfv( GL.GL_FRONT_AND_BACK, GL2.GL_EMISSION, emitColor,0);
        
        
        gl.glLightModelfv(GL2.GL_LIGHT_MODEL_AMBIENT, ambientColor,0); 
        gl.glLightModelf(GL2.GL_LIGHT_MODEL_LOCAL_VIEWER, 1.0f); 
        gl.glLightModeli(GL2.GL_LIGHT_MODEL_COLOR_CONTROL, GL2.GL_SEPARATE_SPECULAR_COLOR);
        }

	}
