package graphics;

import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.util.ArrayList;

import javax.media.opengl.*;

import calc.Calc;
import calc.V3D;



/**
 * This object can be used to build and store Display Lists.
 * Is an implementation of High Performance Renderable.
 */
public class DL implements HPR
	{
	private ArrayList<Vertex> vertices;
	private ArrayList<Integer> indices;
	private int index;
	/**Face type*/
	ArrayList<Integer> faceModes;
	ArrayList<Integer> faceModeIndices;
	/**Number of vertices*/
    int totalNumVerts;
    /**Number of indices*/
    int totalNumInds;
    float[] vData;
    /**Pointers to the display list*/
    int list;
    
    int[] iData;
    IntBuffer iBuf;
    /**The total space of a single vertex with all its data*/
    int vertexStride;
    /**Whether we're using another VA for vertex data*/
    private boolean remoteData;
    
    /**The position of the coordinate data in the vertex data*/
	int vertexPointer;
    /**The position of the normal data in the vertex data*/
    int normalPointer;
    boolean useNormal;
    /**The position of the color data in the vertex data*/
    int colorPointer;
    boolean useColor;
    boolean useAlpha;
    int numColorComponents;
	/**The position of the texture data in the vertex data*/
	int texCoordPointer;
	boolean useTexCoord;
	boolean use3DTexCoord;
	int numTexCoordComponents;
	private boolean arrayOutdated;
	/**Construct a new object which uses the vertex data from dataSrc
	 * Index data must be specified.*/
	public DL(DL dataSrc,int faceMode)
		{
		remoteData = true;
		faceModes = new ArrayList<Integer>();
		faceModeIndices = new ArrayList<Integer>();
		faceModes.add(faceMode);
		this.useNormal  = dataSrc.useNormal;
		this.useColor   = dataSrc.useColor;
		this.useAlpha   = dataSrc.useAlpha;
		this.useTexCoord= dataSrc.useTexCoord;
		this.use3DTexCoord = dataSrc.use3DTexCoord;
		
		vertices = null;
		indices  = new ArrayList<Integer>();
		
		vertexPointer  = dataSrc.vertexPointer;
		normalPointer  = dataSrc.normalPointer;
		colorPointer   = dataSrc.colorPointer;
		numColorComponents = dataSrc.numColorComponents;
		texCoordPointer= dataSrc.texCoordPointer;
		numTexCoordComponents=dataSrc.numTexCoordComponents;

	    totalNumVerts = dataSrc.totalNumVerts;
		vData = dataSrc.vData;
		list = 0;
		
		vertexStride=dataSrc.vertexStride;
		arrayOutdated=false;
		}
	/**Construct an empty object with the given settings
	 * Note: updateMode currently applies only to vertex data, not index data.*/
	public DL(int faceMode, boolean useNormal, boolean useColor, boolean useAlpha, boolean useTexCoord,boolean use3DTexCoord) 
		{
		remoteData = false;
		faceModes = new ArrayList<Integer>();
		faceModeIndices = new ArrayList<Integer>();
		faceModes.add(faceMode);
		this.useNormal  = useNormal;
		this.useColor   = useColor;
		this.useAlpha   = useAlpha;
		this.useTexCoord= useTexCoord;
		this.use3DTexCoord = use3DTexCoord;
		
		vertices = new ArrayList<Vertex>();
		indices  = new ArrayList<Integer>();
		
		list = 0;
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#addVertex(graphics.Vertex)
	 */
	@Override
	public void addVertex(Vertex v)
		{
		vertices.add(v);
		index++;
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#triangle(boolean)
	 */
	@Override
	public void triangle(boolean reverse)
		{
		if(reverse)
			{
			indices.add(index-1);
			indices.add(index-2);
			indices.add(index-3);
			}
		else
			{
			indices.add(index-3);
			indices.add(index-2);
			indices.add(index-1);
			}
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#triangle(int, int, int)
	 */
	@Override
	public void triangle(int i1, int i2, int i3)
		{
		indices.add(i1);
		indices.add(i2);
		indices.add(i3);
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#quad(boolean)
	 */
	@Override
	public void quad(boolean reverse)
		{
		if(reverse)
			{
			if(faceModes.get(faceModeIndices.size())==GL.GL_TRIANGLES)
				{
				indices.add(index-2);
				indices.add(index-3);
				indices.add(index-4);
				indices.add(index-3);
				indices.add(index-2);
				indices.add(index-1);
				}
			else
				{
				indices.add(index-1);
				indices.add(index-2);
				indices.add(index-3);
				indices.add(index-4);
				}
			}
		else
			{
			if(faceModes.get(faceModeIndices.size())==GL.GL_TRIANGLES)
				{
				indices.add(index-4);
				indices.add(index-3);
				indices.add(index-2);
				indices.add(index-1);
				indices.add(index-2);
				indices.add(index-3);
				}
			else
				{
				indices.add(index-4);
				indices.add(index-3);
				indices.add(index-2);
				indices.add(index-1);
				}
			}
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#quad(int, int, int, int)
	 */
	@Override
	public void quad(int i1, int i2, int i3, int i4)
		{
		if(faceModes.get(faceModeIndices.size())==GL.GL_TRIANGLES)
			{
			indices.add(i2);
			indices.add(i3);
			indices.add(i4);
			indices.add(i3);
			indices.add(i2);
			indices.add(i1);
			}
		else
			{
			indices.add(i1);
			indices.add(i2);
			indices.add(i3);
			indices.add(i4);
			}
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#addIndex(int)
	 */
	@Override
	public void addIndex(int index)
		{
		indices.add(index);
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#getIndex()
	 */
	@Override
	public int getIndex()
		{
		return index;
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#requestUpdate()
	 */
	@Override
	public void requestUpdate()
		{
		arrayOutdated = true;
		}
	
	/* (non-Javadoc)
	 * @see graphics.HPR#addSequence(int)
	 */
	@Override
	public void addSequence(int newFaceMode)
		{
		faceModes.add(newFaceMode);
		faceModeIndices.add(indices.size());
		}
	
	/* (non-Javadoc)
	 * @see graphics.HPR#end()
	 */
	@Override
	public void end()
		{
		faceModeIndices.add(indices.size());
		//Calculate component offsets and stuff
		if(!remoteData)
			{
			vertexPointer  = 0;
			vertexStride   = 3;
			normalPointer  = vertexStride*Calc.bytesPerFloat;
			if(useNormal)
				vertexStride+=3;
			colorPointer   = vertexStride*Calc.bytesPerFloat;
			if(useColor)
				vertexStride+=3;
			numColorComponents = 3;
			if(useAlpha)
				{vertexStride+=1;
				numColorComponents++;}
			texCoordPointer= vertexStride*Calc.bytesPerFloat;
			if(useTexCoord)
				vertexStride+=2;
			numTexCoordComponents=2;
			if(use3DTexCoord)
				{vertexStride+=1;
				numTexCoordComponents++;}
		
			//Create buffers and interleave data
		    totalNumVerts = vertices.size();
			vData = new float[totalNumVerts*vertexStride];
			FloatBuffer vBuf = FloatBuffer.wrap(vData);
			
			for(int i=0; i<totalNumVerts; i++)
				{
				Vertex v = vertices.get(i);
					
				vBuf.put(v.vertex);
				if (useNormal)
					vBuf.put(v.normal);
				if (useColor)
					{//Accomodate for missing alpha and vica versa
					if(useAlpha || v.color.length==3)
						{
						vBuf.put(v.color);
						if(useAlpha && v.color.length==3)
							vBuf.put(1);
						}
					else
						vBuf.put(new float[]{v.color[0],v.color[1],v.color[2]});
					}
				if (useTexCoord)
					vBuf.put(v.texCoord);
				}
			vBuf.rewind();
			
			vertices=null;//don't need it anymore
			vertexStride*=Calc.bytesPerFloat;
			}
			
		totalNumInds = indices.size();
		iData = new int[totalNumInds];
	    iBuf = IntBuffer.wrap(iData);
		for(int i=0; i<totalNumInds; i++)
			{
			iBuf.put(indices.get(i));
			}
		iBuf.rewind();
		indices=null;//don't need it anymore
		
		
		genList();
		arrayOutdated=false;
		}
	
	private void genList()
		{
		GL2 gl = GFX.gl;
		if(list!=0)
			gl.glDeleteLists(list, 1);
		list = gl.glGenLists(1);
		gl.glNewList( list, GL2.GL_COMPILE );
		
		int end = 0;
		for(int i=0; i<faceModeIndices.size(); i++)
			{
			int currentFaceMode = faceModes.get(i);
			int start = end;
			end = faceModeIndices.get(i);
			gl.glBegin(currentFaceMode);
			
			for(int j=start; j<end; j++)
				{
				int k = iData[j];
				
				if(useNormal)
					gl.glNormal3fv(vData, (vertexStride*k+normalPointer)/Calc.bytesPerFloat);
				if(useColor)
					{
					if(useAlpha)
						gl.glColor4fv(vData, (vertexStride*k+colorPointer)/Calc.bytesPerFloat);
					else
						gl.glColor3fv(vData, (vertexStride*k+colorPointer)/Calc.bytesPerFloat);
					}
				if(useTexCoord)
					{
					if(use3DTexCoord)
						gl.glTexCoord3fv(vData, (vertexStride*k+texCoordPointer)/Calc.bytesPerFloat);
					else
						gl.glTexCoord2fv(vData, (vertexStride*k+texCoordPointer)/Calc.bytesPerFloat);
					}
				gl.glVertex3fv(vData, (vertexStride*k+vertexPointer)/Calc.bytesPerFloat);
				}
			
			gl.glEnd();
			}
		gl.glEndList();
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#getVData()
	 */
	@Override
	public float[] getVData()
		{
		return vData;
		}
	
	
	/* (non-Javadoc)
	 * @see graphics.HPR#render()
	 */
	@Override
	public void render() 
		{
		GL2 gl = GFX.gl;
		
		
		if(arrayOutdated)
			{
			genList();
			arrayOutdated=false;
			}
		gl.glCallList(list);
		}

   /* (non-Javadoc)
 * @see graphics.HPR#dispose(javax.media.opengl.GL)
 */
@Override
public void dispose(GL gl) 
	   {
	   }
   /* (non-Javadoc)
 * @see graphics.HPR#render(calc.V3D, double)
 */
	@Override
	public void render(V3D center, double scale)
		{
		GFX.gl.glPushMatrix();
		GFX.gl.glTranslated(center.x(),center.y(),center.z());
		GFX.gl.glScaled(scale,scale,scale);
		render();
		GFX.gl.glPopMatrix();
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#render(calc.V3D, double, double, double)
	 */
	@Override
	public void render(V3D center, double xScale, double yScale, double zScale)
		{
		GFX.gl.glPushMatrix();
		GFX.gl.glTranslated(center.x(),center.y(),center.z());
		GFX.gl.glScaled(xScale,yScale,zScale);
		render();
		GFX.gl.glPopMatrix();
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#getVStride()
	 */
	@Override
	public int getVStride()
		{
		return vertexStride/Calc.bytesPerFloat;
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#getVertexPointer()
	 */
	@Override
	public int getVertexPointer()
		{
		return vertexPointer/Calc.bytesPerFloat;
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#getNormalPointer()
	 */
	@Override
	public int getNormalPointer()
		{
		return normalPointer/Calc.bytesPerFloat;
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#getColorPointer()
	 */
	@Override
	public int getColorPointer()
		{
		return colorPointer/Calc.bytesPerFloat;
		}
	/* (non-Javadoc)
	 * @see graphics.HPR#getTexCoordPointer()
	 */
	@Override
	public int getTexCoordPointer()
		{
		return texCoordPointer/Calc.bytesPerFloat;
		}
	}
