package graphics;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Queue;
import java.util.Stack;

import calc.Calc;

/**
 * Class for holding an armature animation loaded from a bvh file 
 */
public class Armature
	{
	private static HashMap<String,Armature> armatures = new HashMap<String,Armature>();
	/**This method looks for the armature with the given file name.
	 * If it is already in memory, the first instance that was loaded is returned.
	 * Otherwise, it is loaded from disk.
	 * If multiple instances of the armature are required, use softCopy on the returned instance.*/
	public static Armature get(String fileName)
		{
		if(armatures.containsKey(fileName))
			return armatures.get(fileName);
		else
			{
			Armature a = new Armature(fileName);
			armatures.put(fileName, a);
			return a;
			}
		}
	
	HashMap<String,Bone> boneMap;
	public Bone[] bones;
	public int frames;
	public float frameTime;
	/**Provides a hint as to whether this instance is being used at the moment.
	 * Strive to set this to false when the user is destroyed.*/
	public boolean isInUse;
	
	private Armature(String filename)
		{
		isInUse=false;
		//Bones will be topologically sorted since it's a tree.
		Stack<Bone> parents = new Stack<Bone>();
		Queue<Bone> boneQ = new LinkedList<Bone>();
		boneMap = new HashMap<String,Bone>();
		Bone bone = null;
		int boneIndex=0;
		@SuppressWarnings("unused")
		int frameIndex=0;
		//Flag set to true when encountering "End Site"
		//The endsite information is not needed.
		boolean endSite = false;
		//Flag set when entering the frame data phase
		boolean frameData = false;
		
		File file = new File("res/armature/"+filename);
		FileReader fr = null;
		BufferedReader br = null;
		try//Setup readers
			{
			fr = new FileReader(file);
			br = new BufferedReader(fr);
			
			String line;
			while((line=br.readLine())!=null)
				{
				line = MTL.strip(line, 0);
				
				if (line.startsWith("ROOT") || line.startsWith("JOINT"))
					{
					line = line.substring(line.indexOf(' ')+1);
					//System.out.println(line+" parent to "+((bone==null)?"null":(bone.name)));
					
					
					parents.push(bone);
					
					bone = new Bone(this, bone, line, boneIndex);
					boneMap.put(line, bone);
					boneQ.offer(bone);
					boneIndex++;
					}
				else if (line.startsWith("CHANNELS"))
					{
					line = line.substring(line.indexOf(' ')+1);
					String numStr = line.substring(0,line.indexOf(' '));
					int channels = Integer.parseInt(numStr);
					if (channels > 0)
						bone.enableRotData();
					if (channels > 3)
						bone.enablePosData();
					if (channels > 6)
						bone.enableScaData();//not impl
					}
				else if (line.startsWith("}"))
					{
					if(!endSite)
						{
						bone = parents.pop();//Go back up a level
						}
					endSite=false;
					}
				else if (line.startsWith("End Site"))
					{
					endSite=true;
					}
				else if (line.startsWith("MOTION"))
					{
					frameData=true;
					}
				else if (line.startsWith("Frames: "))
					{
					frames=Integer.parseInt(line.substring(8));
					//Build array.
					bones = new Bone[boneIndex];
					for(int i=0; i<boneIndex; i++)
						{
						bones[i]=boneQ.poll();
						bones[i].findMirror();
						bones[i].setFrameNumber(frames);
						}
					}
				else if (line.startsWith("Frame Time: "))
					{
					frameTime=Float.parseFloat(line.substring(12));
					}
				else if (line.startsWith("OFFSET") && !endSite)
					{
					line = line.substring(line.indexOf(' ')+1);
					bone.setOffsetFromParent(Calc.getFloatArray(line));
					}
				else if (frameData)
					{
					float[] data = Calc.getFloatArray(line);
					int position=0;
					for(int i=0; i<boneIndex; i++)
						{
						bones[i].addData(data, position);
						position+=bones[i].getBlockSize();
						}
					frameIndex++;
					}
				}
			}
		catch (IOException e)//Print info for debugging if anything fails
			{
			System.err.println("Armature loader fail: "+file);
			e.printStackTrace();
			}
		catch (NumberFormatException e)
			{
			System.err.println("Armature loader fail: "+file);
			e.printStackTrace();
			}
		finally//Close readers if they were opened at all
			{
			try {br.close();
				fr.close();}
			catch (Exception e){}
			}
		}
	private Armature()
		{}
	/**Compute matrices and vectors for the given frame.
	 * @param interpolate 0 means the old deform is kept. 1 means the old deform is completely overwritten.*/
	public void computeFrame(int frame,float interpolate, boolean mirror)
		{
		for (int i=0; i<bones.length; i++)
			{
			bones[i].compute(frame,interpolate,mirror);
			}
		}
	@Override
	public String toString()
		{
		String str = "Armature: ";
		for(int i=0; i<bones.length; i++)
			{
			str+=bones[i].toString();
			}
		str+=" End.";
		return str;
		}
	/**Get a copy that uses the same data arrays*/
	public Armature softCopy()
		{
		Armature result = new Armature();
		result.frameTime = frameTime;
		result.boneMap = new HashMap<String,Bone>();
		result.bones = new Bone[bones.length];
		for(int i=0; i<bones.length; i++)
			{
			Bone b = new Bone(result,bones[i]);
			result.bones[i] = b;
			result.boneMap.put(bones[i].name, b);
			}
		for(int i=0; i<bones.length; i++)
			{
			if(bones[i].parent!=null)
				result.bones[i].setParent(result.getBone(bones[i].parent.name));
			result.bones[i].findMirror();
			}
		result.isInUse=false;
		return result;
		}

	public Bone getBone(String boneName)
		{
		Bone b = boneMap.get(boneName);
		//if(b==null)
		//	System.err.println(boneName+" not found!");
		return b;
		}
	/**Create a new bvh file containing the animation data from all source files.
	 * The files should definitely have identical bone structure.
	 * The only excuse to use this method is that blender's bvh exporter is unbearably slow.
	 * @return The frame numbers of the source bvhs*/
	public static int[] joinBVH(String output, String... sources)
		{
		try
			{
			int[] lengths = new int[sources.length];
			//Create output
			File out = new File("res/armature/"+output);
			PrintWriter bo = new PrintWriter(out);
			//data structure
			Queue<String> data = new LinkedList<String>();
			//Total frame count
			int frames=0;
			float frameTime = 0;
			//Read all source files
			for(int i=0; i<sources.length; i++){
				//Open source file
				File in = new File("res/armature/"+sources[i]);
				BufferedReader bi = new BufferedReader(new FileReader(in));
				//Scan until frame number
				String str;
				while(!(str=bi.readLine()).startsWith("Frames: "))
					{
					//Write to out, only for first file
					if(i==0)
						bo.println(str);
					}
				//Add to frame count
				lengths[i] = Integer.parseInt(str.substring(8)); 
				frames+=lengths[i];
				//Read frame count
				str=bi.readLine();
				if(i==0)
					frameTime=Float.parseFloat(str.substring(12));
				//Read till end, put data in structure 
				while((str=bi.readLine()) != null)
					{
					data.offer(str);
					}
				//Close source file
				bi.close();
			}
			//Write total frame count
			bo.println("Frames: "+frames);
			//Write frame time
			bo.println("Frame Time: "+frameTime);
			//Write all the data
			while(!data.isEmpty())
				{
				bo.println(data.poll());
				}
			//Close output file
			bo.close();
			return lengths;
			}
		catch (IOException e)
			{
			System.err.println("BVH join fail: " + output);
			e.printStackTrace();
			}
		return null;
		}
	public void corrupt()
		{
		for(Bone b : bones)
			{
			if(b.recoverData == null)
				b.recoverData = b.data.clone();
			Calc.corruptFloats(b.data,2);
			}
		}

	public void recover()
		{
		for(Bone b : bones)
			{
			if(b.recoverData != null)
				System.arraycopy(b.recoverData, 0, b.data, 0, b.data.length);
			}
		}
	}
