package calc;

/**
 * This is a class providing static methods for doing various common computations.
 */
public class Calc
	{
	
	public final static int bytesPerFloat = Float.SIZE / Byte.SIZE;
	public final static int bytesPerInt = Integer.SIZE / Byte.SIZE; 
	/**
	 * Convert radians to degrees
	 */
	public static double toDeg(double radians)
		{
		return 180*radians/Math.PI;
		}
	/**
	 * Convert degrees to radians
	 */
	public static double toRad(double degrees)
		{
		return Math.PI*degrees/180;
		}
	/**
	 * Calculate the distance from the origin to (x,y)
	 */
	public static double dist(double x,double y)
		{
		return Math.sqrt(x*x+y*y);
		}
	
	/**
	 * Calculate the squared distance from the origin to (x,y) (cheap)
	 */
	public static double sqDist(double x, double y)
		{
		return x*x+y*y;
		}
	/**
	 * Calculate the distance from the (x1,y1) to (x2,y2)
	 */
	public static double dist(double x1,double y1,double x2,double y2)
		{
		return Math.sqrt((x2-x1)*(x2-x1)+(y2-y1)*(y2-y1));
		}
	/**
	 * Calculate the distance from the origin to (x,y,z)
	 */
	public static double dist(double x,double y,double z)
		{
		return Math.sqrt(x*x+y*y+z*z);
		}
	/**
	 * Calculate the distance from the (x1,y1,z1) to (x2,y2,z2)
	 */
	public static double dist(double x1,double y1,double z1,double x2,double y2,double z2)
		{
		return Math.sqrt((x2-x1)*(x2-x1)+(y2-y1)*(y2-y1)+(z2-z1)*(z2-z1));
		}
	/**
	 * Calculate the direction from (x1,y1) to (x2,y2)
	 */
	public static double dir(double x1, double y1, double x2, double y2)
		{
		double dx = x2-x1;
		double dy = y2-y1;
		double dist = dist(dx,dy);
		dx/=dist;
		dy/=dist;
		if(dx==0 && dy==0)
			return 0;
		if(dx>=0 && dy<=0)
			{
			dy*=-1;
			return Math.atan(dy/dx);
			}
		if(dx<0 && dy<=0)
			{
			dx*=-1;
			dy*=-1;
			return Math.PI-Math.atan(dy/dx);
			}
		if(dx>=0 && dy>0)
			{
			return 2*Math.PI-Math.atan(dy/dx);
			}
		if(dx<0 && dy>0)
			{
			dx*=-1;
			return Math.PI+Math.atan(dy/dx);
			}
		return 0;
		}
	/**
	 * Calculate the direction from (x1,y1) to (x2,y2)
	 */
	public static double dir(int x1, int y1, int x2, int y2)
		{
		return dir((double)x1,(double)y1,(double)x2,(double)y2);
		}
	/**
	 * Force the given number into the interval by adding or subtracting the size of the range 
	 */
	public static int cyclic(int number,int min, int max)
		{
		number = (number-min)%(max-min);
		if (number<0) number+=(max-min);
		return number+min;
		}
		/**
	 * Force the given number into the interval by adding or subtracting the size of the range 
	 */
	public static double cyclic(double d,double e, double f)
		{
		d = (d-e)%(f-e);
		if (d<0) d+=(f-e);
		return d+e;
		}
	/**
	 * Force the given number into the interval by adding or subtracting the size of the range. Min is implicitly 0. 
	 */
	public static int cyclic(int number, int max)
		{
		return (number<0)?(number%max)+max:number%max;
		}
	/**Calculate the shortest difference(never above Math.PI, never below -Math.PI) between any 2 angles*/
	public static double dirDiff(double dir1, double dir2)
		{
		double diff = dir2-dir1;
		while (diff<-Math.PI)
			diff+=2*Math.PI;
		while (diff>Math.PI)
			diff-=2*Math.PI;
		return diff;
		}
	/**
	 * Normalize a float array(as if it were a mathematical vector)
	 * @param vector the vector to be normalized
	 * @return the same array as the input, but now normalized. 
	 */
	public static float[] normalize(float[] vector)
		{
		float length = 0;
		//vector = vector.clone();??
		for(int i=0; i<vector.length; i++)
			length+=Math.pow(vector[i],2);
		length=(float) Math.sqrt(length);
		for(int i=0; i<vector.length; i++)
			vector[i]/=length;
		return vector;
		}
	/**
	 * Add two vectors(same size) with multipliers
	 */
	public static float[] add(float[] v1, int m1, float[] v2, int m2)
		{
		float[] r = new float[v1.length];
		for(int i=0; i<r.length; i++)
			{
			r[i] = v1[i]*m1+v2[i]*m2;
			}
		return r;
		}
	/**
	 * Add two vectors(same size) with a floating point multiplier for the last one
	 */
	public static float[] add(float[] v1, float[] v2, float m2)
		{
		float[] r = new float[v1.length];
		for(int i=0; i<r.length; i++)
			{
			r[i] = v1[i]+v2[i]*m2;
			}
		return r;
		}
	/**Create a concatenated X*Y*Z euler rotation matrix, putting the result in the indicated destination
	 * Optional y-mirroring*/
	public static void makeXYZMatrixOpt(float rX,float rY,float rZ, boolean mirror, float[][] destination)
		{
		//generate units
		float cx = (float) Math.cos(rX);
		float sx = (float) Math.sin(rX);
		float cy = (float) Math.cos(rY);
		float sy = (float) Math.sin(rY);
		float cz = (float) Math.cos(rZ);
		float sz = (float) Math.sin(rZ);
		if(mirror) {sy*=-1; sz*=-1;}
		destination[0][0] = cy*cz;
		destination[0][1] = -cy*sz;
		destination[0][2] = sy;
		destination[1][0] = cx*sz + sy*cz*sx;
		destination[1][1] = cx*cz - sx*sy*sz;
		destination[1][2] = - sx*cy;
		destination[2][0] = sx*sz - sy*cz*cx;
		destination[2][1] = sx*cz + cx*sy*sz;
		destination[2][2] = cx*cy;
		}
	/**Create a concatenated X*Y*Z euler rotation matrix, allocating the result in a new matrix
	 * Optional y-mirroring*/
	public static float[][] makeXYZMatrix(float rX,float rY,float rZ, boolean mirror)
		{
		//generate units
		float cx = (float) Math.cos(rX);
		float sx = (float) Math.sin(rX);
		float cy = (float) Math.cos(rY);
		float sy = (float) Math.sin(rY);
		float cz = (float) Math.cos(rZ);
		float sz = (float) Math.sin(rZ);
		if(mirror) {sy*=-1; sz*=-1;}
		
		float[][] destination = new float[3][3];
		destination[0][0] = cy*cz;
		destination[0][1] = -cy*sz;
		destination[0][2] = sy;
		destination[1][0] = cx*sz + sy*cz*sx;
		destination[1][1] = cx*cz - sx*sy*sz;
		destination[1][2] = - sx*cy;
		destination[2][0] = sx*sz - sy*cz*cx;
		destination[2][1] = sx*cz + cx*sy*sz;
		destination[2][2] = cx*cy;
		return destination;
		/*float[][] rx = new float[][]{
					{1,0,0},
					{0,cx,-sx},
					{0,sx,cx}
				};
		float[][] ry = new float[][]{
					{cy,0,sy},
					{0,1,0},
					{-sy,0,cy}
				};
		float[][] rz = new float[][]{
					{cz,-sz,0},
					{sz,cz,0},
					{0,0,1}
				};
		return matrixMultiply3(rz,matrixMultiply3(ry,rx));*/
		}
	
	public static double log2(double d)
		{
		return Math.log(d)/Math.log(2);
		}
	
	public static float[][] makeZMatrix(float rZ)
		{
		float cz = (float) Math.cos(rZ);
		float sz = (float) Math.sin(rZ);
		
		return new float[][]{
					{cz,-sz,0},
					{sz,cz,0},
					{0,0,1}
				};
		}
	
	/**
	 * Postmultiply a float[3] to a float[3][3], return a new, resulting float[3]
	 */
	public static float[] rotationMatrixPostMultiply(float[][] m, float[] v)
		{
		return new float[]{
				v[0]*m[0][0]+v[1]*m[0][1]+v[2]*m[0][2],
				v[0]*m[1][0]+v[1]*m[1][1]+v[2]*m[1][2],
				v[0]*m[2][0]+v[1]*m[2][1]+v[2]*m[2][2]
				};
		}
	/**
	 * Postmultiply a float[3] to a float[3][3], replacing the contents of the vector.
	 * The matrix is unchanged.
	 */
	public static void rotationMatrixPostMultiplyOpt(float[][] m, float[] v)
		{
		float x=v[0];
		float y=v[1];
		float z=v[2];
		v[0]=x*m[0][0]+y*m[0][1]+z*m[0][2];
		v[1]=x*m[1][0]+y*m[1][1]+z*m[1][2];
		v[2]=x*m[2][0]+y*m[2][1]+z*m[2][2];
		}
	/**
	 * Premultiply a float[3] to a float[3][3], return a new, resulting float[3]
	 */
	public static float[] rotationMatrixPreMultiply(float[] v,float[][] m)
		{
		return new float[]{
				v[0]*m[0][0]+v[1]*m[1][0]+v[2]*m[2][0],
				v[0]*m[0][1]+v[1]*m[1][1]+v[2]*m[2][1],
				v[0]*m[0][2]+v[1]*m[1][2]+v[2]*m[2][2]
				};
		}

	/**Multiply the matrices m1 x m2, putting the result in m1*/
	public static void matrixMultiply3OptFirst(float[][]m1, float[][]m2)
		{
		float m100=m1[0][0],m101=m1[0][1],m102=m1[0][2];
		float m110=m1[1][0],m111=m1[1][1],m112=m1[1][2];
		float m120=m1[2][0],m121=m1[2][1],m122=m1[2][2];
		
		m1[0][0]=m100*m2[0][0]+m101*m2[1][0]+m102*m2[2][0];
		m1[0][1]=m100*m2[0][1]+m101*m2[1][1]+m102*m2[2][1];
		m1[0][2]=m100*m2[0][2]+m101*m2[1][2]+m102*m2[2][2];
		
		m1[1][0]=m110*m2[0][0]+m111*m2[1][0]+m112*m2[2][0];
		m1[1][1]=m110*m2[0][1]+m111*m2[1][1]+m112*m2[2][1];
		m1[1][2]=m110*m2[0][2]+m111*m2[1][2]+m112*m2[2][2];
		
		m1[2][0]=m120*m2[0][0]+m121*m2[1][0]+m122*m2[2][0];
		m1[2][1]=m120*m2[0][1]+m121*m2[1][1]+m122*m2[2][1];
		m1[2][2]=m120*m2[0][2]+m121*m2[1][2]+m122*m2[2][2];
		}
	
	/**Multiply the matrices m1 x m2, putting the result in m2*/
	public static void matrixMultiply3OptLast(float[][]m1, float[][]m2)
		{
		float m200=m2[0][0],m201=m2[0][1],m202=m2[0][2];
		float m210=m2[1][0],m211=m2[1][1],m212=m2[1][2];
		float m220=m2[2][0],m221=m2[2][1],m222=m2[2][2];
		
		m2[0][0]=m1[0][0]*m200+m1[0][1]*m210+m1[0][2]*m220;
		m2[0][1]=m1[0][0]*m201+m1[0][1]*m211+m1[0][2]*m221;
		m2[0][2]=m1[0][0]*m202+m1[0][1]*m212+m1[0][2]*m222;
		
		m2[1][0]=m1[1][0]*m200+m1[1][1]*m210+m1[1][2]*m220;
		m2[1][1]=m1[1][0]*m201+m1[1][1]*m211+m1[1][2]*m221;
		m2[1][2]=m1[1][0]*m202+m1[1][1]*m212+m1[1][2]*m222;
		
		m2[2][0]=m1[2][0]*m200+m1[2][1]*m210+m1[2][2]*m220;
		m2[2][1]=m1[2][0]*m201+m1[2][1]*m211+m1[2][2]*m221;
		m2[2][2]=m1[2][0]*m202+m1[2][1]*m212+m1[2][2]*m222;
		}
	
	public static float[][] matrixMultiply3(float[][]m1, float[][]m2)
		{
		return new float[][]{
					{m1[0][0]*m2[0][0]+m1[0][1]*m2[1][0]+m1[0][2]*m2[2][0],m1[0][0]*m2[0][1]+m1[0][1]*m2[1][1]+m1[0][2]*m2[2][1],m1[0][0]*m2[0][2]+m1[0][1]*m2[1][2]+m1[0][2]*m2[2][2]},
					{m1[1][0]*m2[0][0]+m1[1][1]*m2[1][0]+m1[1][2]*m2[2][0],m1[1][0]*m2[0][1]+m1[1][1]*m2[1][1]+m1[1][2]*m2[2][1],m1[1][0]*m2[0][2]+m1[1][1]*m2[1][2]+m1[1][2]*m2[2][2]},
					{m1[2][0]*m2[0][0]+m1[2][1]*m2[1][0]+m1[2][2]*m2[2][0],m1[2][0]*m2[0][1]+m1[2][1]*m2[1][1]+m1[2][2]*m2[2][1],m1[2][0]*m2[0][2]+m1[2][1]*m2[1][2]+m1[2][2]*m2[2][2]}
			};
		}
	/**
	 * Parses a string to an array of floats. Must be separated with whitespaces.
	 */
	public static float[] getFloatArray(String line)
		{
		String[] split = line.split(" ");
		float[] c = new float[split.length];
		for(int i=0; i<split.length; i++)
			{
			c[i] = Float.parseFloat(split[i]);
			}
		return c;
		}
	public static double mirror(double d, boolean mirror)
		{
		return d*(mirror?-1:1);
		}
	public static float[][] transpose3(float[][] a)
		{
		return new float[][]{
					{a[0][0],a[1][0],a[2][0]},
					{a[0][1],a[1][1],a[2][1]},
					{a[0][2],a[1][2],a[2][2]}};
		}
	/**Force the number into the bounds*/
	public static double limit(double number, double minimum, double maximum)
		{
		if (number<minimum)
			return minimum;
		if (number>maximum)
			return maximum;
		return number;
		}
	/**Force the number into the bounds*/
	public static int limit(int number, int minimum, int maximum)
		{
		if (number<minimum)
			return minimum;
		if (number>maximum)
			return maximum;
		return number;
		}

	/**Failsafe version of arccos*/
	public static double acos(double d)
		{
		if (d<=-1)
			return  Math.PI;
		else if (d>=1)
			return  0;
		return Math.acos(d);
		}
	public static float[][] makeEulerMatrix(float[] r, float f)
		{
		float c = (float)Math.cos(f);
		float s = (float)Math.sin(f);
		float[][] m = new float[][]{
					{c+r[0]*r[0]*(1-c)		,r[0]*r[1]*(1-c)-r[2]*s	,r[0]*r[2]*(1-c)+r[1]*s },
					{r[1]*r[0]*(1-c)+r[2]*s	,c+r[1]*r[1]*(1-c)		,r[1]*r[2]*(1-c)-r[0]*s	},
					{r[2]*r[0]*(1-c)-r[1]*s	,r[2]*r[1]*(1-c)+r[0]*s	,c+r[2]*r[2]*(1-c)		}};
		return m;
		}
	/**Deeper cloning of 2D array*/
	public static float[][] clone2D(float[][] src)
		{
		float[][] result = new float[src.length][];
		for(int i=0; i<result.length; i++)
			{
			result[i] = src[i].clone();
			}
		return result;
		}
	/**Manhattan distance*/
	public static double manDist(double x1, double y1, double x2, double y2)
		{
		return Math.abs(x2-x1)+Math.abs(y2-y1);
		}
	public static void fill(float[] array, float a,float b,float c,float d)
		{
		array[0]=a;
		array[1]=b;
		array[2]=c;
		array[3]=d;
		}
	public static Object choose(Object... strings)
		{
		return strings[(int)(strings.length*Math.random())];
		}
	public static double signSquare(double d)
		{
		if(d<0)
			return -d*d;
		else
			return d*d;
		}
	
	public static double linesIntersection(V3D pos, V3D pos2, V3D origin, V3D direction)
		{
		V3D off1 = pos.clone().sub(origin);
		off1.set(
				off1.p[0]*direction.p[1]-off1.p[1]*direction.p[0],
				off1.p[0]*direction.p[0]+off1.p[1]*direction.p[1],0);
		V3D off2 = pos2.clone().sub(origin);
		off2.set(
				off2.p[0]*direction.p[1]-off2.p[1]*direction.p[0],
				off2.p[0]*direction.p[0]+off2.p[1]*direction.p[1],0);
		if (Math.abs(off1.x()-off2.x())<0.01)
			{
			if(Math.abs(off1.x())>0.01 || (off1.y()<0 && off2.y()<0))
				return Double.MAX_VALUE;//Parallel but not on this line
			else
				return Math.max(Math.min(off1.y(), off2.y()),0);//Parallel and on this line
			}
		else
			{
			if(off1.x()<=0 ^ off2.x()<=0)//ends be on different sides
				{
				double f = -off1.x()/(off2.x()-off1.x());
				return off1.y()+(off2.y()-off1.y())*f;
				}
			}
		return Double.MAX_VALUE;
		}
	public static void corruptBytes(byte[] dataArray, int minLength, int randomLength, int stride)
		{
		int len = minLength+stride*(int)(Math.random()*randomLength/stride);
		int srcPos = stride*(int)(Math.random()*(dataArray.length-len)/stride);
		int dstPos = stride*(int)(Math.random()*(dataArray.length-len)/stride);
		if(dstPos<srcPos+len || dstPos+len>=srcPos)
			{
			if(srcPos>dataArray.length/2)
				dstPos = stride*(int)(Math.random()*(dataArray.length/2-len)/stride);
			else
				dstPos = dataArray.length/2+len+stride*(int)(Math.random()*(dataArray.length/2-len-len)/stride);	
			}
		if(Math.random()<0.33 && stride==2)
			{
			for(int i=0; i<len; i+=2)
				{
				byte tmp = dataArray[dstPos+i+1]; 
				dataArray[dstPos+i+1]=dataArray[srcPos+i+1];
				dataArray[srcPos+i+1]=tmp;
				tmp = dataArray[dstPos+i]; 
				dataArray[dstPos+i]=dataArray[srcPos+i];
				dataArray[srcPos+i]=tmp;
				}
			}
		else if(Math.random()<0.5)
			{
			for(int i=0; i<len; i+=stride)
				{
				for(int j=0; j<stride; j++)
					{
					dataArray[dstPos+i+j]= dataArray[srcPos+len-i-stride+j];
					}
				}
			}
		else
			System.arraycopy(dataArray, srcPos, dataArray, dstPos, len);
		}
	public static void corruptFloats(float[] data, float multiplier)
		{
		int stride = 1+(int)(Math.random()*4);
		double n = Math.random()*0.0015;
		double a = 0.0005;
		for(int i = (int)(Math.random()*4); i<data.length; i+=stride)
			{
			data[i] += n*multiplier;
			a*=1+Math.random();
			if(Math.abs(n)>0.005)
				a = -Math.signum(n)*Math.random()*0.001;
			n+=a;
			}
		}
	}
