import { EditorState, TextSelection, Transaction } from 'prosemirror-state'
import { Node } from 'prosemirror-model'
import { CSSProperties, forwardRef, UIEvent, useCallback, useEffect, useImperativeHandle, useRef } from 'react'
import { EditorView } from 'prosemirror-view'

import { ProseMirrorHandle, ProseMirror, useProseMirror } from './prosemirror'
import { clipboardTransform, placeholder, propsOverride, nodeIds, devTool, inspectNodeChanges } from './plugins'
import { schema } from './schema'
import { baseKeymap } from 'prosemirror-commands'
import { updateOriginMarks } from './actions'
import { UniqueId } from './util'
import { keymap } from 'prosemirror-keymap'

export interface EditorSelection {
    from: number
    to: number
    left: number
    right: number
    top: number
    bottom: number
    text: string
}

interface EditorInnerState {
    blocked: boolean
}
export type EditorHandle = {
    view: EditorView | null
    root: HTMLDivElement | null
    state: EditorInnerState
    focus: (focus?: boolean) => void
}

export type EditorId = string
export type EditorProps = {
    style?: CSSProperties
    className?: string
    editorId: EditorId
    onTransaction?: (transaction: Transaction, editorId: EditorId) => void
    onSelectionChange?: (selection: EditorSelection, editorId: EditorId) => void
}

export const Editor = forwardRef<EditorHandle, EditorProps>(function Editor(
    { style, className, editorId, onTransaction, onSelectionChange },
    ref
) {
    const editorIdRef = useRef(editorId)

    const blockedRef = useRef(false)
    const viewRef = useRef<ProseMirrorHandle>(null)

    const onTransactionRef = useRef(onTransaction)
    onTransactionRef.current = onTransaction

    const onSelectionChangeRef = useRef(onSelectionChange)
    onSelectionChangeRef.current = onSelectionChange

    const onNodesChanged = useCallback((nodes: Map<UniqueId, { node?: Node; after?: Node }>) => {
        console.debug('changed nodes', nodes)
    }, [])

    const updateSelectionCallbackRef = useRef(0)
    const updateSelection = useCallback((tr: Transaction) => {
        const selection = tr.selection
        clearTimeout(updateSelectionCallbackRef.current)
        const updateSelectionCallback = () => {
            if (onSelectionChangeRef.current) {
                const content = selection.content().content
                const from = Math.min(selection.from, viewRef.current?.view?.state.doc.content.size ?? 0)
                const to = Math.min(selection.to, viewRef.current?.view?.state.doc.content.size ?? 0)
                const left = viewRef.current?.view?.coordsAtPos(from).left ?? 0
                const right = viewRef.current?.view?.coordsAtPos(to).right ?? 0
                const top = viewRef.current?.view?.coordsAtPos(from).top ?? 0
                const bottom = viewRef.current?.view?.coordsAtPos(to).bottom ?? 0
                const scroll = viewRef.current?.root?.getBoundingClientRect().y ?? 0
                const offset = viewRef.current?.root?.offsetTop ?? 0
                const editorScroll = viewRef.current?.view?.dom?.parentElement?.scrollTop ?? 0
                onSelectionChangeRef.current?.call(
                    onSelectionChangeRef.current,
                    {
                        from: from,
                        to: to,
                        left,
                        right,
                        top: top - (scroll - offset - editorScroll),
                        bottom: bottom - (scroll - offset - editorScroll),
                        text: content.textBetween(0, content.size, '\n'),
                    },
                    editorIdRef.current
                )
            }
        }
        updateSelectionCallbackRef.current = setTimeout(updateSelectionCallback, 20) as unknown as number
    }, [])

    const [state, transformState] = useProseMirror({
        schema: schema,
        plugins: [
            placeholder(),
            keymap(baseKeymap),
            clipboardTransform(),
            nodeIds(),
            propsOverride({
                handlePaste: () => false,
                editable: () => true,
                handleKeyDown: () => false,
                handleClick: () => {
                    return
                },
            }),
            inspectNodeChanges(onNodesChanged),
            devTool(),
        ],
    })

    const dispatchTransaction = useCallback(
        (tr: Transaction, state: EditorState) => {
            if (blockedRef.current) return
            // updateSelection(tr)
            // updateOriginMarks(tr, state)
            transformState(() => tr)
            onTransactionRef.current?.call(onTransactionRef.current, tr, editorIdRef.current)
        },
        [transformState]
    )

    const focus = useCallback((focus = true) => {
        if (focus) {
            viewRef.current?.view?.dom.focus()
        } else {
            viewRef.current?.view?.dom.blur()
        }
    }, [])

    const scrollPositionRef = useRef(0)
    const scrollCallbackRef = useRef(0)
    const handleScroll = useCallback((view: EditorView, event: UIEvent<HTMLDivElement>) => {
        clearTimeout(scrollCallbackRef.current)
        scrollCallbackRef.current = setTimeout(() => {
            scrollPositionRef.current =
                (event.target as HTMLDivElement).scrollHeight -
                (event.target as HTMLDivElement).scrollTop -
                (event.target as HTMLDivElement).clientHeight
        }, 20) as unknown as number
    }, [])

    useImperativeHandle(ref, () => ({
        get view() {
            return viewRef.current?.view ?? null
        },
        get root() {
            return viewRef.current?.root ?? null
        },
        set state({ blocked }: EditorInnerState) {
            blockedRef.current = blocked
        },
        focus: (f) => queueMicrotask(() => focus(f)),
    }))

    const resizeCallbackRef = useRef(0)
    useEffect(() => {
        queueMicrotask(() => {
            blockedRef.current = false
            editorIdRef.current = editorId
            focus()
        })
        // eslint-disable-next-line compat/compat
        const observer = new ResizeObserver(() => {
            if (!viewRef.current?.root) return
            if (scrollPositionRef.current <= 5) {
                viewRef.current.root.scrollTop = viewRef.current.root.scrollHeight
            }
            clearTimeout(resizeCallbackRef.current)
            resizeCallbackRef.current = setTimeout(() => {
                transformState((state) => {
                    updateSelection(state.tr)
                    return
                })
            }, 50) as unknown as number
        })
        if (viewRef.current?.root) observer.observe(viewRef.current.root)
        return () => observer.disconnect()
    }, [editorId, transformState, updateSelection, focus])

    return (
        <ProseMirror
            style={style}
            className={className}
            ref={viewRef}
            state={state}
            dispatchTransaction={dispatchTransaction}
            handleScroll={handleScroll}
        />
    )
})

export default Editor
