import { Transaction } from 'prosemirror-state'
import styled from 'styled-components'
import { MutableRefObject, PropsWithRef, startTransition, useEffect, useRef, useState } from 'react'

import { Editor, EditorHandle } from './editor/editor'

const HBox = styled.div`
    display: flex;
    flex-direction: row;
    flex-wrap: wrap;
    max-height: 100%;
`
const VBox = styled.div`
    display: flex;
    flex-direction: column;
    flex: 0;
    max-height: 100%;
`
const AppBox = styled(HBox)`
    height: 100%;
    width: 100%;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans',
        'Helvetica Neue', sans-serif;
    @supports (-webkit-touch-callout: none) {
        height: -webkit-fill-available;
    }
    flex-wrap: wrap;
    align-items: stretch;
    overflow: hidden;
`
const EditorContainer = styled(Editor)`
    background: rgba(101, 154, 255, 0.06);
    display: flex;
    flex: 1;
    flex-direction: column;
    overflow: auto;
    min-width: 400px;
    min-height: 200px;
    .ProseMirror {
        height: 100%;
        width: 100%;
    }
`

export const App = () => {
    const [transactionList, setTransactionList] = useState([] as Array<[number, Transaction]>)
    const editorRef = useRef<EditorHandle>(null)
    const transactionCount = useRef(0)
    return (
        <AppBox>
            <VBox style={{ flex: 1 }}>
                <EditorContainer
                    editorId="1"
                    ref={editorRef}
                    onTransaction={(tr) =>
                        startTransition(() => {
                            setTransactionList((list) => [[transactionCount.current, tr], ...list.slice(-2)])
                            transactionCount.current += 1
                        })
                    }
                />
            </VBox>
            <VBox style={{ width: '100%', minWidth: 'calc(max(300px, 25vw))' }}>
                <TransactionList transactionList={transactionList} />
            </VBox>
        </AppBox>
    )
}
export default App

const TransactionListBackgrounds = ['rgba(250, 50, 80, 0.05)', 'rgba(30, 400, 20, 0.05)']
type TransactionListProps = PropsWithRef<{ transactionList: Array<[number, Transaction]> }>
const TransactionList = ({ transactionList }: TransactionListProps) => {
    const containerRef: MutableRefObject<null | HTMLDivElement> = useRef(null)
    const colorRef = useRef(0)
    const [listRender, setListRender] = useState([<div key="-1"></div>])
    useEffect(() => {
        setListRender(
            transactionList.map(([i, tr], idx) => (
                <div key={idx} style={{ background: TransactionListBackgrounds[(idx + colorRef.current) % 2] }}>
                    {i}: {JSON.stringify(tr, undefined, 3)}
                </div>
            ))
        )
        colorRef.current += 1
        setTimeout(() => containerRef.current?.scroll({ top: -Number.MAX_SAFE_INTEGER, behavior: 'auto' }), 10)
    }, [transactionList])
    return <TransactionListContainer ref={containerRef}>{listRender}</TransactionListContainer>
}
const TransactionListContainer = styled.div`
    display: flex;
    flex-direction: column;
    flex: 1;
    overflow: auto;
    font-size: 0.675rem;
    line-height: 0.975rem;
    & > div {
        padding: 8px;
        white-space: pre;
    }
`
