import React, { useMemo } from 'react'
import { FlatList } from 'react-native-gesture-handler'
import { useRecoilCallback, useRecoilValue, useSetRecoilState } from 'recoil'
import {
    GenerationRequestActive,
    SelectedShelf,
    SelectedStory,
    SessionValue,
    Stories,
    StorySearch,
    StoryShelves,
    StorySort,
} from '../../../shared/globals/state'
import { MetadataMatchResult, SearchFilter } from '../../../shared/data/storage/search'
import { StoryChildContent, StoryMetadata } from '../../../shared/data/story/storycontainer'
import { GlobalUserContext } from '../../../shared/globals/globals'
import { UserSettings } from '../../../shared/data/user/user'
import { StoryElement, ShelfElement, StoryNoResultPlaceholder } from './elements'
import { FocusedMenubarElement } from '../../globals/state'
import { noBounceProps } from '../../styles/noBounce'

const storyFilter = new SearchFilter()

type StoryData = {
    id: string
    key: string
    story: StoryMetadata
    highlight: MetadataMatchResult
    onPress: (long?: boolean) => void
    parent?: StoryMetadata
}

type ShelfData = {
    id: string
    key: string
}

const isStory = (toCheck: StoryData | ShelfData): toCheck is StoryData =>
    (toCheck as StoryData).story !== undefined

const renderItem = (rProps: { item: StoryData | ShelfData }) => {
    return isStory(rProps.item) ? (
        <StoryElement
            id={rProps.item.id}
            key={rProps.item.key}
            story={rProps.item.story}
            highlight={rProps.item.highlight}
            onPress={rProps.item.onPress}
            parent={rProps.item.parent}
        />
    ) : (
        <ShelfElement id={rProps.item?.id ?? ''} key={rProps.item.id} />
    )
}

export default function StoryList(props: { switchEditor(): void }) {
    const setSelected = useSetRecoilState(SelectedStory)
    const stories = useRecoilValue(Stories)
    const shelves = useRecoilValue(StoryShelves)
    const searchValue = useRecoilValue(StorySearch)
    const sortValue = useRecoilValue(StorySort)
    const selectedShelf = useRecoilValue(SelectedShelf)
    const sessionSettings = useRecoilValue(SessionValue('settings')) as UserSettings
    const setFocusedElement = useSetRecoilState(FocusedMenubarElement)

    const onStoryElementClick = useRecoilCallback(
        ({ snapshot }) =>
            async (result: MetadataMatchResult, long?: boolean) => {
                const generationRequestActive = await snapshot.getPromise(GenerationRequestActive)
                const selected = await snapshot.getPromise(SelectedStory)

                if (generationRequestActive) {
                    return
                }

                if (result.metadata?.id) {
                    if (selected.id !== result.metadata.id) {
                        setSelected({
                            id: result.metadata.id,
                        })
                    }
                    setFocusedElement(result.metadata?.id)

                    //change to editor screen if not long press
                    if (!long) props.switchEditor()
                }
            },
        []
    )

    //TODO: set max amount and take into account scroll, like on web
    const filteredStories = useMemo(() => {
        const shelfElements = []

        // if shelf is selected, search shelf and sub shelves
        if (selectedShelf) {
            const children = GlobalUserContext.shelves.get(selectedShelf)?.children ?? []
            shelfElements.push(...children)
        }

        // otherwise search unshelved stories
        if (!selectedShelf) {
            for (const story of stories) {
                let found = false
                for (const shelf of shelves) {
                    if (
                        GlobalUserContext.shelves
                            .get(shelf)
                            ?.children?.filter((child) => child.type === 'story')
                            .map((child) => child.id)
                            .includes(story)
                    ) {
                        found = true
                        break
                    }
                }
                if (!found) {
                    shelfElements.push({ type: 'story', id: story } as StoryChildContent)
                }
            }
            for (const shelf of shelves) {
                let found = false
                for (const shelfInner of shelves) {
                    if (
                        GlobalUserContext.shelves
                            .get(shelfInner)
                            ?.children?.filter((child) => child.type === 'shelf')
                            .map((child) => child.id)
                            .includes(shelf)
                    ) {
                        found = true
                        break
                    }
                }
                if (!found) {
                    shelfElements.push({ type: 'shelf', id: shelf } as StoryChildContent)
                }
            }
        }
        let results = storyFilter.metadataMatch(shelfElements, searchValue)
        switch (sortValue.by.value) {
            case 'recent': {
                if (sortValue.reverse)
                    results.sort((a, b) =>
                        a.metadata && b.metadata
                            ? a.metadata.lastUpdatedAt.getTime() - b.metadata.lastUpdatedAt.getTime()
                            : 0
                    )
                else
                    results.sort((a, b) =>
                        a.metadata && b.metadata
                            ? b.metadata.lastUpdatedAt.getTime() - a.metadata.lastUpdatedAt.getTime()
                            : 0
                    )
                break
            }
            case 'alphabetical': {
                if (sortValue.reverse)
                    results.sort((a, b) =>
                        a.metadata && b.metadata ? b.metadata.title.localeCompare(a.metadata.title) : 0
                    )
                else
                    results.sort((a, b) =>
                        a.metadata && b.metadata ? a.metadata.title.localeCompare(b.metadata.title) : 0
                    )
                break
            }
            case 'creation': {
                if (sortValue.reverse)
                    results.sort((a, b) =>
                        a.metadata && b.metadata
                            ? a.metadata.createdAt.getTime() - b.metadata.createdAt.getTime()
                            : 0
                    )
                else
                    results.sort((a, b) =>
                        a.metadata && b.metadata
                            ? b.metadata.createdAt.getTime() - a.metadata.createdAt.getTime()
                            : 0
                    )
                break
            }
        }
        if (sessionSettings.sortShelvesOnTop ?? true) {
            results.sort((a, b) => (a.elementType === b.elementType ? 0 : a.elementType === 'shelf' ? -1 : 1))
        }

        const elements = results.map((result): StoryData | ShelfData => {
            return result.elementType === 'story'
                ? {
                      id: result.metadata.id,
                      key: result.metadata.id,
                      story: result.metadata,
                      highlight: result,
                      onPress: (long) => onStoryElementClick(result, long),
                      parent: result.parent,
                  }
                : {
                      id: result.metadata?.id ?? '',
                      key: result.metadata.id,
                  }
        })
        return elements
    }, [
        selectedShelf,
        searchValue,
        sortValue.by.value,
        sortValue.reverse,
        stories,
        shelves,
        onStoryElementClick,
        sessionSettings.sortShelvesOnTop,
    ])

    /*
    const [scrollTop, setScrollTop] = useState(false)
    const [scrollBottom, setScrollBottom] = useState(false)
    useEffect(() => {
        const onScroll = () => {
            setScrollTop((storyListRef.current?.scrollTop ?? 0) >= 10)
            setScrollBottom(
                (storyListRef.current?.scrollTop ?? 0) + (storyListRef.current?.offsetHeight ?? 0) <=
                    (storyListRef.current?.scrollHeight ?? 0) - 10
            )
        }
        const element = storyListRef.current
        element?.addEventListener('scroll', onScroll)
        return () => element?.removeEventListener('scroll', onScroll)
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [storyListRef.current])
    */

    return filteredStories.length > 0 ? (
        <FlatList
            data={filteredStories}
            renderItem={renderItem}
            keyExtractor={(item) => item.id}
            style={{ height: '100%' }}
            {...noBounceProps}
        />
    ) : (
        <StoryNoResultPlaceholder />
    )
}
