import React, { useCallback } from 'react'
import { BackHandler, Pressable } from 'react-native'
import { useRecoilState, useRecoilValue, useSetRecoilState, useRecoilCallback } from 'recoil'
import { MaterialTopTabNavigationProp } from '@react-navigation/material-top-tabs'

import {
    SelectedStory,
    Stories,
    StoryUpdate,
    StorySearch,
    GenerationRequestActive,
    SiteTheme,
    Session,
    SelectedShelf,
    SelectedStoryId,
} from '../../../shared/globals/state'
import { StoryContainer } from '../../../shared/data/story/storycontainer'
import { GlobalUserContext } from '../../../shared/globals/globals'
import { Menubar as StyledMenubar, Header, HeaderTitle, HeaderSubTitle } from './menubar.style'

import { PenTipIcon } from '../common/icons'
import { StoryAppScreenParams } from '../storyapp'
import { FocusedMenubarElement } from '../../globals/state'
import StoryList from './storylist'
import { DefaultPrefixOption, StorySettings } from '../../../shared/data/story/storysettings'
import { getModelPresets } from '../../../shared/data/story/defaultpresets'
import { getStorage } from '../../../shared/data/storage/storage'
import { DefaultModel } from '../../../shared/data/request/model'
import ButtonsBlock from './buttonsblock'
import { StoryListPlaceholder } from './elements'
import { FilterControls } from './filtercontrols'
import { NoSubscriptionOnly } from '../../../shared/components/util/accountrequired'
import TrialUsageDisplay from '../modals/trialactions'
import { FlexRow } from '../common/common.style'
import { GearSelectedIcon } from '../../assets/images/navigation/navicons'
import { RootStackParamList } from '../app'

export default function MenuBar(props: {
    navigation: MaterialTopTabNavigationProp<StoryAppScreenParams & RootStackParamList, 'stories'>
}): JSX.Element {
    const theme = useRecoilValue(SiteTheme)

    const setSearchValue = useSetRecoilState(StorySearch)
    const [stories, setStories] = useRecoilState(Stories)

    const setSelectedStory = useSetRecoilState(SelectedStory)
    const setStoryUpdate = useSetRecoilState(StoryUpdate(''))
    const [selectedShelf, setSelectedShelf] = useRecoilState(SelectedShelf)

    const selectedStoryId = useRecoilValue(SelectedStoryId)
    const setFocusedElement = useSetRecoilState(FocusedMenubarElement)

    const handleBackPress = useCallback(() => {
        if (props.navigation.isFocused() && selectedShelf !== '') {
            setSelectedShelf('')
            return true
        }
        return false
    }, [selectedShelf])

    React.useEffect(() => {
        BackHandler.addEventListener('hardwareBackPress', handleBackPress)
        return () => {
            BackHandler.removeEventListener('hardwareBackPress', handleBackPress)
        }
    }, [handleBackPress])

    React.useEffect(() => {
        setFocusedElement(selectedStoryId)
    }, [selectedStoryId])

    React.useEffect(() => {
        if (selectedShelf === '') setFocusedElement(selectedStoryId)
    }, [selectedShelf])

    //could be isolated? identical from web
    const addStoryPress = useRecoilCallback(
        ({ snapshot }) =>
            async () => {
                const generationRequestActive = await snapshot.getPromise(GenerationRequestActive)
                const session = await snapshot.getPromise(Session)

                if (generationRequestActive) {
                    return
                }
                const story = new StoryContainer()

                story.content.settings = new StorySettings()
                GlobalUserContext.stories.set(story.metadata.id, story.metadata)
                GlobalUserContext.storyContentCache.set(story.metadata.id, story.content)
                story.metadata.remote = session.settings.remoteDefault ?? false
                story.content.settings.prefix = session.settings.defaultModule || DefaultPrefixOption
                story.content.settings.model = session.settings.defaultModel || DefaultModel
                story.content.settings.preset =
                    session.settings.defaultPreset ?? getModelPresets(story.content.settings.model)[0].id

                setSearchValue('')
                setStories([story.metadata.id, ...stories])
                setSelectedStory({ loaded: false, id: story.metadata.id })
                setStoryUpdate(story.metadata.save(false))

                const selectedShelf = await snapshot.getPromise(SelectedShelf)

                await getStorage(session).saveStory(story, story.metadata.remote)

                if (selectedShelf) {
                    const shelf = GlobalUserContext.shelves.get(selectedShelf)
                    if (!shelf) return
                    setTimeout(() => {
                        shelf.children = [
                            ...(shelf.children ?? []),
                            {
                                type: 'story',
                                id: story.metadata.id,
                            },
                        ]
                        setStories((stories) => [...stories])
                        getStorage(session).saveStoryShelf(shelf)
                    }, 50)
                }
            },
        [stories]
    )

    return (
        <StyledMenubar>
            <FlexRow grow={false}>
                <Header>
                    <PenTipIcon fill={theme.colors.textMain} />
                    <HeaderTitle>{'NovelAI'}</HeaderTitle>
                    <HeaderSubTitle>{' beta'}</HeaderSubTitle>
                </Header>
                <Pressable onPress={() => props.navigation.navigate('settings')}>
                    <GearSelectedIcon primary={theme.colors.textMain} style={{ marginRight: 20 }} />
                </Pressable>
            </FlexRow>
            <NoSubscriptionOnly>
                <TrialUsageDisplay />
            </NoSubscriptionOnly>
            {stories.length > 0 ? (
                <>
                    <FilterControls />
                    <StoryList switchEditor={() => props.navigation.navigate('editor')} />
                    <ButtonsBlock addStoryPress={addStoryPress} />
                </>
            ) : (
                <StoryListPlaceholder addStoryPress={addStoryPress} />
            )}
        </StyledMenubar>
    )
}
