﻿import React, { useEffect, useRef, useState } from 'react'
import { useRecoilState, useRecoilValue, useSetRecoilState } from 'recoil'
import {
    SelectedShelf,
    Session,
    Stories,
    StorySearch,
    StoryShelves,
    StorySort,
} from '../../../shared/globals/state'
import { getStorage } from '../../../shared/data/storage/storage'
import { GlobalUserContext } from '../../../shared/globals/globals'
import { EmitterSubscription, Keyboard, Platform, Pressable, TextInput, View } from 'react-native'
import {
    Search,
    SearchInner,
    SmallLookingGlass,
    FunnelEmpty,
    FunnelFilled,
    BarView,
    IconsRow,
    FilterOptionsView,
    ArrowDown,
    ArrowUp,
    DisplayOrder,
    OptionContainer,
    OptionLabel,
    Home,
    LeftArrow,
    Edit,
} from './filtercontrols.style'
import { BodyMedium400, HeadingLarge400 } from '../../styles/fonts'
import { HoverModal } from '../modals/common'
import Checkbox from '../common/checkbox'
import { InputSelected } from '../../globals/state'
import { FlatList } from 'react-native-gesture-handler'
import { ShelfMetadataModal } from '../modals/shelfmetadatamodal'
import { ModalTitleRow } from '../modals/common'
import { noBounceProps } from '../../styles/noBounce'

type Option = { label: string; value: string }

function SimpleSelector(props: {
    options: Option[]
    value: Option
    onChange: (value: Option) => void
}): JSX.Element {
    const renderItem = (itemProps: { item: Option }) => {
        return (
            <OptionContainer
                selected={itemProps.item.value == props.value.value}
                onPress={() => {
                    props.onChange(itemProps.item)
                }}
            >
                <OptionLabel>{itemProps.item.label}</OptionLabel>
            </OptionContainer>
        )
    }

    return <FlatList data={props.options} renderItem={renderItem} {...noBounceProps} />
}

export function FilterControls() {
    const [searchValue, setSearchValue] = useRecoilState(StorySearch)
    const [selectedShelf, setSelectedShelf] = useRecoilState(SelectedShelf)
    const [session, setSession] = useRecoilState(Session)

    const [searchVisible, setSearchVisible] = useState(false)
    const [filterVisible, setFilterVisible] = useState(false)

    const [order, setOrder] = useRecoilState(StorySort)

    const setStories = useSetRecoilState(Stories)

    const prevSearchValue = useRef('')
    const blurRef = useRef(0)
    const searchInputRef = useRef<TextInput>(null)

    useRecoilValue(StoryShelves)

    const setInputSelected = useSetRecoilState(InputSelected)
    const [shelfModalVisible, setShelfModalVisible] = useState(false)

    const onSearchPress = (closing?: boolean) => {
        clearTimeout(blurRef.current)
        if (!searchVisible && !closing) {
            setSearchValue(prevSearchValue.current)
            setInputSelected(true)
        } else {
            prevSearchValue.current = searchValue
            setSearchValue('')
        }
        setSearchVisible(!searchVisible)
    }

    const onFilterPress = () => {
        searchVisible && setSearchVisible(filterVisible)
        setFilterVisible(!filterVisible)
    }

    useEffect(() => {
        if (searchVisible && searchInputRef.current) searchInputRef.current.focus()
    }, [searchVisible])

    //if used for other such components, an alternative should be considered
    useEffect(() => {
        const handleKeyboardHide = () => searchInputRef.current?.blur()
        const subscription = Keyboard.addListener(
            Platform.OS === 'ios' ? 'keyboardWillHide' : 'keyboardDidHide',
            handleKeyboardHide
        )
        return () => subscription.remove()
    }, [searchInputRef])

    return (
        <View>
            <BarView
                darkened={searchVisible}
                onPress={() => {
                    if (!selectedShelf) onSearchPress()
                }}
            >
                <Search>
                    {searchVisible ? (
                        <SearchInner
                            placeholder={selectedShelf ? 'Search this shelf' : 'Search your stories'}
                            value={searchValue}
                            onChangeText={(text) => setSearchValue(text)}
                            onPressIn={() => setInputSelected(true)}
                            onBlur={() => onSearchPress(true)}
                            ref={searchInputRef}
                        />
                    ) : selectedShelf && GlobalUserContext.shelves.get(selectedShelf) ? (
                        <View style={{ flexDirection: 'row' }}>
                            <Pressable style={{ flexDirection: 'row' }} onPress={() => setSelectedShelf('')}>
                                <Home />
                                <LeftArrow />
                            </Pressable>
                            <HeadingLarge400>
                                {GlobalUserContext.shelves.get(selectedShelf)?.title}
                            </HeadingLarge400>
                        </View>
                    ) : (
                        <View>
                            <HeadingLarge400>Your Stories</HeadingLarge400>
                        </View>
                    )}
                </Search>
                <IconsRow>
                    {selectedShelf ? (
                        <Pressable
                            onPress={() => {
                                setShelfModalVisible(true)
                                onSearchPress(true)
                            }}
                        >
                            <Edit />
                        </Pressable>
                    ) : null}
                    <Pressable
                        aria-label="Open Sort Settings"
                        onPress={(e) => {
                            e.stopPropagation()
                            onFilterPress()
                        }}
                    >
                        {filterVisible ? <FunnelFilled /> : <FunnelEmpty />}
                    </Pressable>
                    <SmallLookingGlass
                        aria-label="Search"
                        highlight={searchVisible}
                        //style={{ transform: [{ scaleX: -1 }] }}
                        onPress={(e) => {
                            e.stopPropagation()
                            onSearchPress()
                        }}
                    />
                    <ShelfMetadataModal
                        key={selectedShelf} //forces component to reinitialize all states
                        id={selectedShelf}
                        visible={shelfModalVisible}
                        closeModal={() => setShelfModalVisible(false)}
                    />
                </IconsRow>
            </BarView>
            <HoverModal visible={filterVisible} onRequestClose={() => setFilterVisible(false)}>
                <ModalTitleRow title={'Sorting Settings'} onClose={() => setFilterVisible(false)} />
                <FilterOptionsView>
                    <Checkbox
                        value={session.settings.sortShelvesOnTop ?? true}
                        setValue={(v) => {
                            setSession((session) => {
                                const newSession = {
                                    ...session,
                                    settings: { ...session.settings, sortShelvesOnTop: v },
                                }
                                getStorage(newSession).saveSettings(newSession.settings)
                                return newSession
                            })
                        }}
                        label="Always show shelves on top"
                        alternate={true}
                        style={{ paddingHorizontal: 10 }}
                    />
                    <DisplayOrder
                        aria-label="Change display order"
                        onPress={() => setOrder({ ...order, reverse: !order.reverse })}
                        style={{
                            flexDirection: 'row',
                            alignItems: 'center',
                            paddingHorizontal: 10,
                        }}
                    >
                        <BodyMedium400>Sort By </BodyMedium400>
                        {order.reverse ? <ArrowUp /> : <ArrowDown />}
                    </DisplayOrder>
                    <SimpleSelector
                        value={order.by}
                        onChange={(value) => value && setOrder({ ...order, by: value })}
                        options={[
                            {
                                label: 'Most Recent',
                                value: 'recent',
                            },
                            {
                                label: 'Alphabetical',
                                value: 'alphabetical',
                            },
                            {
                                label: 'Creation Date',
                                value: 'creation',
                            },
                        ]}
                    />
                </FilterOptionsView>
            </HoverModal>
        </View>
    )
}
