import React, { useState } from 'react'
import { useRecoilValue } from 'recoil'
import { View } from 'react-native'
import sodium from 'libsodium-wrappers'
import { UserSubscription, UserPriority, UserInformation, User } from '../../../shared/data/user/user'
import { BackgroundSplash, ContentView, GreetingText, TopGreetingText } from './common.style'
import { removeLocalStorage } from '../../util/overrides'
import { Bar, Field } from './common'
import { logError, logInfo, logWarning } from '../../../shared/util/browser'
import { Session, SessionValue, SiteTheme } from '../../../shared/globals/state'
import { getLoginRequest } from '../../../shared/data/request/request'
import { GlobalUserContext } from '../../../shared/globals/globals'
import { emailConfirmationRequiredDate } from '../../../shared/util/util'
import { useUserSetup } from '../../hooks/useUserSetup'
import { NativeStackNavigationProp } from '@react-navigation/native-stack'
import { RootStackParamList } from '../app'
import { SubmitButton, SubmitButtonText } from '../common/button.style'
import { UserSettings } from '../../../shared/data/user/settings'

interface ILoginData {
    email: undefined | string
    access_key: undefined | string
    encryption_key: undefined | string
    subscription: undefined | UserSubscription
    session: undefined | any
    settings: undefined | UserSettings
    priority: undefined | UserPriority
    information: undefined | UserInformation
}

export default function Login(props: {
    navigation: NativeStackNavigationProp<RootStackParamList, 'login'>
}): JSX.Element {
    const authenticated = useRecoilValue(SessionValue('authenticated'))
    const session = useRecoilValue(Session)
    const theme = useRecoilValue(SiteTheme)

    const [rawUsername, setRawUsername] = useState('')
    const [password, setPassword] = useState('')

    const [error, setError] = useState('')
    const [inputsEnabled, setInputsEnabled] = useState(true)

    const [remember, setRemember] = useState(true)
    const [accountSelection, setAccountSelection] = useState<Array<ILoginData>>([])

    //TODO: add loading indicator
    const [loading, setLoading] = useState(true)

    const setupUser = useUserSetup()

    //redirect when authenticated
    React.useEffect(() => {
        if (
            authenticated &&
            (session.information.emailVerified ||
                session.information.trialActivated ||
                session.information.accountCreatedAt <= emailConfirmationRequiredDate)
        )
            props.navigation.replace('storyapp')
        else if (
            authenticated &&
            !(
                session.information.emailVerified ||
                session.information.trialActivated ||
                session.information.accountCreatedAt <= emailConfirmationRequiredDate
            )
        ) {
            logWarning('unauthorized session, not logging in')
            setInputsEnabled(true)
            setLoading(false)
        }
    }, [
        authenticated,
        session.information.emailVerified,
        session.information.emailVerificationLetterSent,
        session.information.trialActivated,
        session.information.accountCreatedAt,
    ])

    // process user data
    const process = async (user: User, email: string) => {
        if (
            !user.information.emailVerified &&
            !user.information.trialActivated &&
            user.information.accountCreatedAt > emailConfirmationRequiredDate
        ) {
            //TODO: Implement trial
            //setLocalTrialState(-1)
            //router.push('/confirm?email=' + encodeURIComponent(Buffer.from(email).toString('base64')))
            setInputsEnabled(true)
            return
        }
        try {
            //TODO: Implement trial
            //setLocalTrialState(-1)
            await setupUser(user, remember)
            setInputsEnabled(true)
        } catch (error: any) {
            switch (error?.message) {
                case 'A mutation operation was attempted on a database that did not allow mutations.':
                    setError('Firefox private browsing is not supported as it does not support IndexedDB.')
                    break
                case 'Invalid Access Key':
                    setError('Invalid email or password.')
                    break
                default:
                    setError(error?.message ?? error)
                    break
            }
            setInputsEnabled(true)
            logError(error, false)
        }
    }

    // process login data
    const login = async (login_data: ILoginData) => {
        if (!login_data.access_key || !login_data.encryption_key) {
            throw new Error('Something went wrong, account data incomplete')
        }
        const user = new User(login_data.session.auth_token, login_data.encryption_key)

        user.subscription = login_data.subscription ?? new UserSubscription()
        user.settings = login_data.settings ?? new UserSettings()
        user.authenticated = login_data.session.authenticated
        user.priority = login_data.priority ?? new UserPriority()
        user.information = login_data.information ?? new UserInformation()

        GlobalUserContext.keystore = login_data.session.keystore

        await process(user, login_data.email || '')
    }

    const authenticate = async () => {
        const username = rawUsername.toLocaleLowerCase()

        if (!username || !password) {
            setError('Email and Password cannot be empty')
            return
        }

        if (!username.includes('@') || username.length <= 3) {
            setError('Email must be in email format')
            return
        }
        if (password.length < 8) {
            setError('Password must be 8 characters or more')
            return
        }

        setInputsEnabled(false)
        setError('')
        await sodium.ready
        removeLocalStorage('session')

        try {
            const login_data = {
                email: undefined,
                access_key: undefined,
                encryption_key: undefined,
                subscription: undefined,
                session: undefined,
                settings: undefined,
                priority: undefined,
                information: undefined,
            } as ILoginData
            const legacy_login_data = {
                email: undefined,
                access_key: undefined,
                encryption_key: undefined,
                subscription: undefined,
                session: undefined,
                settings: undefined,
                priority: undefined,
                information: undefined,
            } as ILoginData

            let loginError

            try {
                login_data.access_key = sodium
                    .crypto_pwhash(
                        64,
                        new Uint8Array(Buffer.from(password)),
                        sodium.crypto_generichash(
                            sodium.crypto_pwhash_SALTBYTES,
                            password.slice(0, 6) + username + 'novelai_data_access_key'
                        ),
                        2,
                        2000000,
                        sodium.crypto_pwhash_ALG_ARGON2ID13,
                        'base64'
                    )
                    .slice(0, 64)
                login_data.encryption_key = sodium.crypto_pwhash(
                    128,
                    new Uint8Array(Buffer.from(password)),
                    sodium.crypto_generichash(
                        sodium.crypto_pwhash_SALTBYTES,
                        password.slice(0, 6) + username + 'novelai_data_encryption_key'
                    ),
                    2,
                    2000000,
                    sodium.crypto_pwhash_ALG_ARGON2ID13,
                    'base64'
                )

                const login = getLoginRequest(login_data.access_key, login_data.encryption_key)

                const result = await login.login()
                login_data.subscription = result.subscription
                login_data.session = result.session
                login_data.settings = result.settings
                login_data.priority = result.priority
                login_data.information = result.information
                login_data.email = username
            } catch (error: any) {
                logInfo(error, false, 'login failed:')
                loginError = error
            }

            // try non-lowercased username
            if (rawUsername !== username) {
                try {
                    legacy_login_data.access_key = sodium
                        .crypto_pwhash(
                            64,
                            new Uint8Array(Buffer.from(password)),
                            sodium.crypto_generichash(
                                sodium.crypto_pwhash_SALTBYTES,
                                password.slice(0, 6) + rawUsername + 'novelai_data_access_key'
                            ),
                            2,
                            2000000,
                            sodium.crypto_pwhash_ALG_ARGON2ID13,
                            'base64'
                        )
                        .slice(0, 64)
                    legacy_login_data.encryption_key = sodium.crypto_pwhash(
                        128,
                        new Uint8Array(Buffer.from(password)),
                        sodium.crypto_generichash(
                            sodium.crypto_pwhash_SALTBYTES,
                            password.slice(0, 6) + rawUsername + 'novelai_data_encryption_key'
                        ),
                        2,
                        2000000,
                        sodium.crypto_pwhash_ALG_ARGON2ID13,
                        'base64'
                    )

                    const login = getLoginRequest(
                        legacy_login_data.access_key,
                        legacy_login_data.encryption_key
                    )

                    const result = await login.login()
                    legacy_login_data.subscription = result.subscription
                    legacy_login_data.session = result.session
                    legacy_login_data.settings = result.settings
                    legacy_login_data.priority = result.priority
                    legacy_login_data.email = rawUsername
                } catch (error: any) {
                    logInfo(error, false, 'legacy login failed:')
                }
            }

            // try login with uppercase first character if both logins failed
            const legacyUsername = username.slice(0, 1).toUpperCase() + username.slice(1)
            if (
                !legacy_login_data.session &&
                !login_data.session &&
                legacyUsername !== rawUsername &&
                legacyUsername !== username
            ) {
                try {
                    if (legacyUsername !== username) {
                        legacy_login_data.access_key = sodium
                            .crypto_pwhash(
                                64,
                                new Uint8Array(Buffer.from(password)),
                                sodium.crypto_generichash(
                                    sodium.crypto_pwhash_SALTBYTES,
                                    password.slice(0, 6) + legacyUsername + 'novelai_data_access_key'
                                ),
                                2,
                                2000000,
                                sodium.crypto_pwhash_ALG_ARGON2ID13,
                                'base64'
                            )
                            .slice(0, 64)
                        legacy_login_data.encryption_key = sodium.crypto_pwhash(
                            128,
                            new Uint8Array(Buffer.from(password)),
                            sodium.crypto_generichash(
                                sodium.crypto_pwhash_SALTBYTES,
                                password.slice(0, 6) + legacyUsername + 'novelai_data_encryption_key'
                            ),
                            2,
                            2000000,
                            sodium.crypto_pwhash_ALG_ARGON2ID13,
                            'base64'
                        )

                        const login = getLoginRequest(
                            legacy_login_data.access_key,
                            legacy_login_data.encryption_key
                        )

                        const result = await login.login()
                        legacy_login_data.subscription = result.subscription
                        legacy_login_data.session = result.session
                        legacy_login_data.settings = result.settings
                        legacy_login_data.priority = result.priority
                        legacy_login_data.email = legacyUsername
                    }
                } catch (error: any) {
                    logInfo(error, false, 'legacy login failed:')
                }
            }

            if (!legacy_login_data.session && !login_data.session) {
                throw loginError
            } else if (legacy_login_data.session && !login_data.session) {
                /* alright, just one account with capitalization */
                login_data.subscription = legacy_login_data.subscription
                login_data.session = legacy_login_data.session
                login_data.settings = legacy_login_data.settings
                login_data.priority = legacy_login_data.priority
                login_data.access_key = legacy_login_data.access_key
                login_data.encryption_key = legacy_login_data.encryption_key
            } else if (!legacy_login_data.session && login_data.session) {
                /* nice, just one correct account */
            } else {
                /* oh no, two correct accounts */
                setAccountSelection([login_data, legacy_login_data])
                setInputsEnabled(true)
                return
            }
            login(login_data)
        } catch (error: any) {
            switch (error?.message) {
                case 'A mutation operation was attempted on a database that did not allow mutations.':
                    setError('Firefox private browsing is not supported as it does not support IndexedDB.')
                    break
                case 'Invalid Access Key':
                    setError('Invalid email or password.')
                    break
                default:
                    setError(error?.message ?? error)
                    break
            }
            setInputsEnabled(true)
            logError(error, false)
        }
    }

    return (
        <View>
            <BackgroundSplash />
            <ContentView>
                <Bar
                    pushBack={() => props.navigation.navigate('landing')}
                    pushOtherScreen={() => props.navigation.replace('signup')}
                    otherScreenName={'Signup'}
                />
                <TopGreetingText>Log In</TopGreetingText>
                <GreetingText>Welcome back!</GreetingText>
                <Field
                    name={'Email'}
                    value={rawUsername}
                    editable={inputsEnabled}
                    onChangeText={(text) => {
                        setRawUsername(text)
                    }}
                    autoCorrect={false}
                    textContentType={'emailAddress'}
                />
                <Field
                    name={'Password'}
                    value={password}
                    editable={inputsEnabled}
                    onChangeText={(text) => {
                        setPassword(text)
                    }}
                    secureTextEntry={true}
                    autoCorrect={false}
                    textContentType={'password'}
                />
                <SubmitButton onPress={authenticate}>
                    <SubmitButtonText>Submit</SubmitButtonText>
                </SubmitButton>
            </ContentView>
        </View>
    )
}
