import React, { useState } from 'react'
import { Footer, Separator, SidebarElementsContainer } from './infobar.style'
import LorebookQuickAccess from '../lorebook/lorebookquickaccess'
import ContextEditor from './items/contexteditor'
import EditorCard from '../common/editorcard'
import { Image, Text, View } from 'react-native'
import { LinkIcon } from '../common/icons'
import ModuleSelector from './items/moduleselector'
import { modelSupportsModules } from '../../../shared/data/ai/model'
import { useRecoilValue, useSetRecoilState } from 'recoil'
import {
    SelectedStoryLoaded,
    Session,
    SettingsModalOpen,
    SiteTheme,
    StoryUpdate,
    UserPresets,
} from '../../../shared/globals/state'
import {
    getAvailiableModels,
    modelName,
    modelsHaveSamePresets,
    prefixIsDefault,
} from '../../../shared/util/models'
import { DefaultModel, normalizeModel, TextGenerationModel } from '../../../shared/data/request/model'
import { GlobalUserContext } from '../../../shared/globals/globals'
import { useReload } from '../../../shared/hooks/useReload'
import { SetterPackage, updateStory } from '../../../shared/component-logic/optionslogic'
import StorySettings from './items/storysettings'
import StoryStats from './items/storystats'
import StoryExporter from './items/storyexporter'
import { StoryDeleteButton } from '../modals/deletemodal'
import { SidebarPlaceholder } from './infobar'
import { SettingsPages } from '../../../shared/components/settings/constants'
import SelectorModal from '../modals/selectormodal'
import { NoModule } from '../../../shared/data/story/storysettings'
import { StoryMode } from '../../../shared/data/story/story'
import { getModelPresets } from '../../../shared/data/story/defaultpresets'
import { copyPresetToStory } from '../../../shared/util/presets'
import { ModuleOptionVisual } from '../../hooks/useModuleOptions'
import { BodyLarge600, BodyMedium400 } from '../../styles/fonts'

export default function StoryTab(props: { selectedStory: string }): JSX.Element {
    const [presetModalVisible, setPresetModalVisible] = useState(false)

    const session = useRecoilValue(Session)
    useRecoilValue(SelectedStoryLoaded)

    const setSettingsModalOpen = useSetRecoilState(SettingsModalOpen)

    const currentStory = GlobalUserContext.stories.get(props.selectedStory)
    const currentStoryContent = GlobalUserContext.storyContentCache.get(props.selectedStory)
    const settings = currentStoryContent?.settings
    const setStoryUpdate = useSetRecoilState(StoryUpdate(props.selectedStory))
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    const userPresets = useRecoilValue(UserPresets)

    const reload = useReload()
    const setterPackage: SetterPackage = {
        currentStory: currentStory,
        currentStoryContent: currentStoryContent,
        genSettings: currentStoryContent?.settings.parameters,
        updateState: setStoryUpdate,
    }
    const siteTheme = useRecoilValue(SiteTheme)

    console.log('currentStoryContent as of last StoryTab refresh: ', currentStoryContent)

    if (!settings) return <SidebarPlaceholder status={''} />

    const modelOptions = getAvailiableModels(session.subscription.tier >= 3)
    const selectedOption =
        modelOptions.find((m) => m.str === normalizeModel(currentStoryContent.settings.model)) ??
        modelOptions.find((m) => m.str === session.settings.defaultModel) ??
        modelOptions.find((m) => m.str === DefaultModel) ??
        modelOptions[0]

    return (
        <SidebarElementsContainer>
            <EditorCard
                title={'AI Model'}
                hint={
                    <Text>
                        Change default <LinkIcon />
                    </Text>
                }
                onHintClick={() => setSettingsModalOpen(SettingsPages.Defaults)}
            >
                <SelectorModal
                    noSeparators={true}
                    selectedID={selectedOption.str}
                    selectedLabel={
                        <ModuleOptionVisual
                            image={selectedOption.img}
                            name={selectedOption.label}
                            selected={false}
                        />
                    }
                    setID={(modelID) => {
                        const model = Object.values(TextGenerationModel).find((model) => model === modelID)
                        if (!model) return
                        updateStory(() => {
                            currentStoryContent.settings.model = model
                            if (
                                !prefixIsDefault(currentStoryContent.settings.prefix ?? '') ||
                                !modelSupportsModules(model)
                            ) {
                                currentStoryContent.settings.prefix = NoModule
                                currentStoryContent.settings.prefixMode = StoryMode.normal
                            }
                            let defaultPreset = getModelPresets(model)[0]
                            if (session.settings.defaultPreset) {
                                let tempPreset = getModelPresets(model).find(
                                    (p) => p.id === session.settings.defaultPreset
                                )
                                if (!tempPreset) {
                                    tempPreset = userPresets.find(
                                        (p) => p.id === session.settings.defaultPreset
                                    )
                                }
                                if (tempPreset && modelsHaveSamePresets(tempPreset.model, model)) {
                                    defaultPreset = tempPreset
                                }
                            }

                            copyPresetToStory(defaultPreset, currentStoryContent)
                            reload()
                        }, setterPackage)
                    }}
                    data={[
                        {
                            title: '',
                            data: modelOptions.map((model) => ({
                                value: model.str,
                                description: `${model.label}: ${model.description}`,
                                rawLabel: '',
                                label: (
                                    <View style={{ flexDirection: 'row', paddingVertical: 5 }}>
                                        <Image source={model.img} style={{ height: 100, width: 100 }} />
                                        <View style={{ padding: 10, flex: 1 }}>
                                            <BodyLarge600>{model.label}</BodyLarge600>
                                            {model.description ? (
                                                <BodyMedium400>{model.description}</BodyMedium400>
                                            ) : null}
                                        </View>
                                    </View>
                                ),
                            })),
                        },
                    ]}
                />
            </EditorCard>
            {modelSupportsModules(selectedOption.str) ? (
                <EditorCard
                    title={'AI Module'}
                    hint={
                        <Text>
                            All Modules <LinkIcon />
                        </Text>
                    }
                    description={'Changes the style of the AI-generated text.'}
                    onHintClick={() => setPresetModalVisible(true)}
                >
                    <ModuleSelector
                        selectedStory={props.selectedStory}
                        presetModalVisible={presetModalVisible}
                        setPresetModalVisible={setPresetModalVisible}
                    />
                </EditorCard>
            ) : (
                <EditorCard
                    title={'AI Module'}
                    description={
                        <Text>
                            You are currently using {modelName(selectedOption.str)}, which does not support AI
                            Modules.
                        </Text>
                    }
                >
                    <View />
                </EditorCard>
            )}
            <EditorCard title="Memory" description="The AI will better remember info placed here.">
                <ContextEditor placeholder="" contextIndex={0} />
            </EditorCard>
            <EditorCard
                title="Author's Note"
                description="Info placed here will strongly influence AI output."
            >
                <ContextEditor placeholder="" contextIndex={1} />
            </EditorCard>
            <EditorCard title="Lorebook Quick Access" style={{ paddingTop: 25 }}>
                <LorebookQuickAccess />
            </EditorCard>
            <EditorCard title={'Story Options'}>
                <View>
                    <StoryStats selectedStory={props.selectedStory} />
                    <StorySettings selectedStory={props.selectedStory} />
                    <StoryExporter selectedStory={props.selectedStory} />
                </View>
            </EditorCard>
            <Separator />
            <EditorCard title="Delete Story" hint={'(cannot be undone)'}>
                <StoryDeleteButton id={props.selectedStory} />
            </EditorCard>
            <Footer />
        </SidebarElementsContainer>
    )
}
