import React, { useState } from 'react'
import { Modal, View } from 'react-native'
import { TextInput } from 'react-native-gesture-handler'
import { useRecoilValue, useSetRecoilState } from 'recoil'
import {
    buildEphemeralContext,
    entryToText,
    EphemeralEntry,
    isEphemeralEntryActive,
} from '../../../../shared/data/ai/ephemeralcontext'
import { GlobalUserContext } from '../../../../shared/globals/globals'
import { SelectedStory, StoryUpdate } from '../../../../shared/globals/state'
import { BodyLarge600 } from '../../../styles/fonts'
import { FlexRow } from '../../common/common.style'
import { ModalTitleRow } from '../../modals/common'
import { FullModalView } from '../../modals/common.style'

export function EphemeralContextDisplay(): JSX.Element {
    const selectedStory = useRecoilValue(SelectedStory)
    const currentStoryContent = GlobalUserContext.storyContentCache.get(selectedStory.id)

    if (!currentStoryContent) {
        return <></>
    }

    return (
        <FlexRow>
            {currentStoryContent.ephemeralContext
                .filter((entry) => isEphemeralEntryActive(entry, currentStoryContent.story.step))
                .map((entry, index) => {
                    return entry !== null ? (
                        //<SidebarListItem key={i}>{e.text.replace(/\n/g, '\\n')}</SidebarListItem>
                        <></>
                    ) : (
                        <></>
                    )
                })}
        </FlexRow>
    )
}

export function EphemeralContextModal(props: { visible: boolean; closeModal: () => void }): JSX.Element {
    const selectedStory = useRecoilValue(SelectedStory)
    const currentStoryContent = GlobalUserContext.storyContentCache.get(selectedStory.id)
    const currentStoryMetadata = GlobalUserContext.stories.get(selectedStory.id)
    const setStoryUpdate = useSetRecoilState(StoryUpdate(''))
    const [entryInput, setEntryInput] = useState('')
    const [selectedEntry, setSelectedEntry] = useState(-1)
    const [confirmDelete, setConfirmDelete] = useState(-1)

    if (!currentStoryContent || !currentStoryMetadata) {
        return <></>
    }

    const crateContextEntry = () => {
        if (selectedEntry < 0) {
            currentStoryContent.ephemeralContext.push(
                buildEphemeralContext(entryInput, currentStoryContent.story.step)
            )
            setStoryUpdate(currentStoryMetadata.save())
            setEntryInput('')
        } else {
            currentStoryContent.ephemeralContext[selectedEntry] = buildEphemeralContext(
                entryInput,
                currentStoryContent.story.step
            )

            setStoryUpdate(currentStoryMetadata.save())
            setSelectedEntry(-1)
            setEntryInput('')
        }
    }

    return (
        <Modal visible={props.visible} onRequestClose={props.closeModal}>
            <ModalTitleRow title={'Ephemeral Context'} onClose={props.closeModal} />
            <FullModalView>
                <TextInput
                    placeholder={'Add ephemeral context entry'}
                    value={entryInput}
                    onChangeText={(text: string) => setEntryInput(text)}
                />
                <View>
                    {currentStoryContent.ephemeralContext.map((entry, index) => (
                        <EphemeralListItem
                            selected={index === selectedEntry}
                            onPress={() => {
                                if (index === selectedEntry) {
                                    setSelectedEntry(-1)
                                    setEntryInput('')
                                } else {
                                    setSelectedEntry(index)
                                    setEntryInput(entryToText(entry))
                                }
                            }}
                            confirmDelete={index === confirmDelete}
                            onDelete={() => {
                                setSelectedEntry(-1)
                                setEntryInput('')
                                if (confirmDelete !== index) {
                                    setConfirmDelete(index)
                                    return
                                }
                                setConfirmDelete(-1)
                                currentStoryContent.ephemeralContext = [
                                    ...currentStoryContent.ephemeralContext.slice(0, index),
                                    ...currentStoryContent.ephemeralContext.slice(index + 1),
                                ]
                                setStoryUpdate(currentStoryMetadata.save())
                            }}
                            unsetDelete={() => setConfirmDelete(-1)}
                            key={index}
                            entry={entry}
                            step={currentStoryContent.story.step}
                        />
                    ))}
                </View>
            </FullModalView>
        </Modal>
    )
}

function EphemeralListItem(props: {
    entry: EphemeralEntry
    step: number
    selected: boolean
    onPress: () => void
    onDelete: () => void
    confirmDelete: boolean
    unsetDelete: () => void
}): JSX.Element {
    if (props.entry === null) {
        return <></>
    }
    return (
        <View>
            {/** Main Area */}
            <View>
                <BodyLarge600>
                    step: {props.entry.startingStep}; {props.entry.reverse ? 'reverse; ' : ''}{' '}
                    {props.entry.delay !== 1 ? 'delay: ' + props.entry.delay + '; ' : ''}{' '}
                    {props.entry.duration !== 1 ? 'duration: ' + props.entry.duration + '; ' : ''}{' '}
                    {props.entry.repeat ? 'repeat;' : ''} pos: {props.entry.contextConfig.insertionPosition}{' '}
                    text: {props.entry.text.replace(/\n/g, '\\n')}
                </BodyLarge600>
            </View>
            {/** Delete Button Area */}
            <View>
                <BodyLarge600>
                    {props.confirmDelete ? 'Delete' : '' /** <ListEntryDeleteIcon /> */}
                </BodyLarge600>
            </View>
        </View>
    )
}
