import React, {
    Dispatch,
    forwardRef,
    MutableRefObject,
    SetStateAction,
    useEffect,
    useImperativeHandle,
    useRef,
    useState,
} from 'react'
import { Text, TextInput } from 'react-native'
import { useSetRecoilState } from 'recoil'
import styled from 'styled-components/native'
import { DataOrigin } from '../../../shared/components/editor/glue'
import { Change } from '../../../shared/data/document/document'
import { Section, SectionId } from '../../../shared/data/document/section'
import { UniqueId } from '../../../shared/data/document/util'
import { useReload } from '../../../shared/hooks/useReload'
import { InputSelected } from '../../globals/state'
import { BodyLarge400M } from '../../styles/fonts'
import { hexOpacity } from '../../util/colors'
import { EditorModel } from './editormodel'
import { Segment, SegmentDecorations, SegmentStyle, VisualSegment } from './model'

export type EditorViewHandle = {
    focused: boolean
    selection: EditorSelection
    width: number
    empty: boolean
    redraw: () => void
    paragraphRefresh: (...toRefresh: UniqueId[]) => void
    segmentRefresh: (...toRefresh: Segment[]) => void
    restoreFirstChar: () => void
    onModelChanged: (changes: Map<SectionId, Change<Section>>) => void
    //Set their heights ASAP? Maybe there could be a way to return a promise.
    //trackHeight: (...toTrack: Paragraph[]) => void
}

export type EditorViewProps = {
    model: EditorModel
    onModelChanged: (changes: Map<SectionId, Change<Section>>) => void
}

export type EditorSelection = {
    start: number
    end: number
}

export const EditorView = forwardRef<EditorViewHandle, EditorViewProps>(function Editor(
    { model, onModelChanged },
    ref
): JSX.Element {
    const setInputSelected = useSetRecoilState(InputSelected)
    const [empty, setEmpty] = useState(model.order.length === 0)
    const segmentRefs = useRef<Map<UniqueId, Dispatch<SetStateAction<VisualSegment>>>>(new Map())
    const paragraphRefs = useRef<Map<UniqueId, Dispatch<SetStateAction<VisualSegment[]>>>>(new Map())
    const restoreFirstCharRef = useRef<() => void>(() => null)
    const width = useRef(0)
    const reload = useReload()

    const selectionRef = useRef<EditorSelection>({ start: 0, end: 0 })
    const textInputRef = useRef<TextInput>(null)

    const refreshSegments = (...segs: Segment[]) => {
        segs.forEach((seg) => {
            const updater = segmentRefs.current.get(seg.id)
            if (updater) updater(seg.getVisual())
            else console.warn('Segment', seg.id, ' has no component.')
        })
    }

    const refreshParagraphs = (...ids: UniqueId[]) => {
        ids.forEach((id) => {
            const updater = paragraphRefs.current.get(id)
            const value = model.getVisuals(id)
            if (updater && value) updater(value)
            else console.warn('Paragraph', id, 'either does not exist, or has no component.')
        })
    }

    useImperativeHandle(ref, () => ({
        get focused() {
            return textInputRef.current?.isFocused() ?? false
        },
        get width() {
            return width.current
        },
        get empty() {
            return empty
        },
        get selection() {
            return selectionRef.current
        },
        set empty(f) {
            setEmpty(f)
        },
        set selection(f) {
            selectionRef.current = f
        },
        redraw: () => queueMicrotask(() => reload()),
        paragraphRefresh: (...f) => queueMicrotask(() => refreshParagraphs(...f)),
        segmentRefresh: (...f) => queueMicrotask(() => refreshSegments(...f)),
        restoreFirstChar: () => queueMicrotask(() => restoreFirstCharRef.current()),
        onModelChanged: (f) => queueMicrotask(() => onModelChanged(f)),
    }))

    return (
        <EditorTextInput
            ref={textInputRef}
            multiline={true}
            placeholder={'Enter your prompt here...'}
            onTextInput={(event) => model.processInput(event.nativeEvent)}
            onChangeText={(text) => {
                const isEmpty = text.length === 0
                if (empty != isEmpty) setEmpty(isEmpty)
            }}
            onLayout={(event) => (width.current = event.nativeEvent.layout.width)}
            onSelectionChange={(event) => (selectionRef.current = event.nativeEvent.selection)}
            onPressIn={() => setInputSelected(true)}
        >
            {!empty && <FirstChar restoreFirstCharRef={restoreFirstCharRef} />}
            {model.order.map((id, index) => {
                return (
                    <DisplayedParagraph
                        key={id}
                        segmentRefs={segmentRefs}
                        updateRef={(value) => paragraphRefs.current.set(id, value)}
                        removeRef={() => paragraphRefs.current.delete(id)}
                        spaced={index < model.order.length - 1}
                        initialSegs={model.getVisuals(id)}
                    />
                )
            })}
        </EditorTextInput>
    )
})

function FirstChar(props: { restoreFirstCharRef: MutableRefObject<() => void> }): JSX.Element {
    const [value, setValue] = useState(false)
    props.restoreFirstCharRef.current = () => setValue((val) => !val)
    return <Text style={{ fontSize: 0 }}>{value ? <Text>.</Text> : '.'}</Text>
}

function DisplayedParagraph(props: {
    segmentRefs: MutableRefObject<Map<UniqueId, Dispatch<SetStateAction<VisualSegment>>>>
    updateRef: (value: Dispatch<SetStateAction<VisualSegment[]>>) => void
    removeRef: () => void
    spaced: boolean
    initialSegs?: VisualSegment[]
}): JSX.Element {
    const [segments, setSegments] = useState<VisualSegment[]>(props.initialSegs ?? [])

    useEffect(() => {
        props.updateRef(setSegments)
        return () => {
            props.removeRef()
        }
    }, [])

    console.log('Paragraph redrawn!')

    return (
        <Text>
            {segments.map((seg) => {
                return (
                    <DisplayedSegment
                        key={seg.id}
                        updateRef={(value) => props.segmentRefs.current.set(seg.id, value)}
                        removeRef={() => {
                            props.segmentRefs.current.delete(seg.id)
                        }}
                        initialValue={seg}
                    />
                )
            })}
            {props.spaced && '\n'}
        </Text>
    )
}

function DisplayedSegment(props: {
    updateRef: (value: Dispatch<SetStateAction<VisualSegment>>) => void
    removeRef: () => void
    initialValue: VisualSegment
}): JSX.Element {
    const [value, setValue] = useState<VisualSegment>(props.initialValue)
    const [alternate, setAlternate] = useState(false) //to force reset with setAlternate((old) => !old)

    //useEffect(() => setAlternate((old) => !old), [value])

    useEffect(() => {
        props.updateRef(setValue)
        return () => {
            props.removeRef()
        }
    }, [])

    return (
        <SegmentText origin={value.origin} marks={value.style}>
            {alternate ? value.data : <Text>{value.data}</Text>}
        </SegmentText>
    )
}

const EditorTextInput = styled(TextInput).attrs((props) => ({
    placeholderTextColor: hexOpacity(props.theme.colors.textMain, 0.5),
}))`
    ${BodyLarge400M};
    padding-horizontal: 0px;
    text-align-vertical: top;
    flex: 1;
`

const SegmentText = styled.Text<{ origin: DataOrigin; marks: SegmentStyle }>`
    ${(props) => props.marks.bold && 'font-weight: 700;'}
    ${(props) => props.marks.italic && 'font-style: italic;'}
    text-decoration-line: ${(props) => {
        switch (props.marks.decorations) {
            case SegmentDecorations.struck:
                return 'line-through;'
            case SegmentDecorations.underlined:
                return 'underline;'
            case SegmentDecorations.both:
                return 'line-through underline;'
            default:
                return 'none;'
        }
    }}
    color: ${(props) => {
        switch (props.origin) {
            case DataOrigin.user:
                return props.theme.colors.textUser
            case DataOrigin.ai:
                return props.theme.colors.textAI
            case DataOrigin.edit:
                return props.theme.colors.textEdit
            case DataOrigin.prompt:
                return props.theme.colors.textPrompt
            default:
                return props.theme.colors.textPrompt
        }
    }};
`
