import { TextFormatting } from '../../../shared/components/editor/glue'
import {
    Section,
    SectionType,
    SectionTypeText,
    SectionTypeTextMeta,
} from '../../../shared/data/document/section'
import { UniqueId } from '../../../shared/data/document/util'
import { Document } from '../../../shared/data/document/document'
import { Paragraph, Segment, SegmentDecorations } from './model'
import { serialize } from 'serializr'

export function paragraphToSection(par: Paragraph, detailed?: boolean): Section {
    if (par.length === 0) return { type: SectionType.empty }
    let text = ''
    const originMarks: SectionTypeTextMeta[] = []
    const styleMarks: SectionTypeTextMeta[] = []
    let currentPosition = 0
    par.segments.forEach((seg) => {
        const newStyleMarks: TextFormatting[] = []
        if (seg.style.bold) newStyleMarks.push(TextFormatting.bold)
        if (seg.style.italic) newStyleMarks.push(TextFormatting.italic)
        switch (seg.style.decorations) {
            // @ts-ignore: Fallthrough case in switch
            case SegmentDecorations.both:
                newStyleMarks.push(TextFormatting.underline)
            case SegmentDecorations.struck:
                newStyleMarks.push(TextFormatting.strikethrough)
                break
            case SegmentDecorations.underlined:
                newStyleMarks.push(TextFormatting.underline)
        }
        styleMarks.push(
            ...newStyleMarks.map((newMark) => ({
                position: currentPosition,
                length: seg.data.length,
                data: newMark,
            }))
        )
        originMarks.push({
            position: currentPosition,
            length: seg.data.length,
            data: seg.origin,
        })
        text += seg.data
        currentPosition += seg.data.length
    })
    if (detailed)
        console.log(
            'Original Pragraph:',
            JSON.stringify(serialize(par)),
            'Origin Marks:',
            originMarks,
            'Style Marks:',
            styleMarks
        )
    return {
        type: SectionType.text,
        text: text,
        meta: new Map([
            [1, originMarks],
            [2, styleMarks],
        ]),
    }
}

type CategorizedMark = {
    origin: number
} & SectionTypeTextMeta

function applyMarks(segment: Segment, ...marks: CategorizedMark[]) {
    marks.forEach((mark) => {
        if (mark.origin === 1) segment.origin = mark.data
        else {
            switch (mark.data) {
                case TextFormatting.bold:
                    segment.style.bold = true
                    break
                case TextFormatting.italic:
                    segment.style.italic = true
                    break
                case TextFormatting.strikethrough:
                    segment.style.decorations =
                        segment.style.decorations ?? 0 > 1
                            ? SegmentDecorations.both
                            : SegmentDecorations.struck
                    break
                case TextFormatting.underline:
                    segment.style.decorations = [SegmentDecorations.struck, SegmentDecorations.both].includes(
                        segment.style.decorations ?? 0
                    )
                        ? SegmentDecorations.both
                        : SegmentDecorations.underlined
                    break
            }
        }
    })
}

export function sectionToParagraph(section: Section, start?: number, detailed?: boolean): Paragraph {
    if (section.type === 0) return new Paragraph()
    const castSection = section as SectionTypeText

    let combinedMarks: CategorizedMark[] = [...castSection.meta.entries()].flatMap((metas) =>
        metas[1].map((meta) => ({ origin: metas[0], ...meta }))
    )
    //get all separation points between segments
    const stops = combinedMarks
        .reduce((stops: number[], mark: CategorizedMark) => {
            const endPos = mark.position + mark.length
            if (!stops.includes(mark.position)) stops.push(mark.position)
            if (!stops.includes(endPos)) stops.push(endPos)
            return stops
        }, [])
        .sort((a, b) => a - b)
    if (detailed) console.log('Stops:', stops)
    let activeMarks: CategorizedMark[] = []
    let previousStop = 0
    const segments = stops.reduce((segments: Segment[], stop: number) => {
        const text = castSection.text.substring(previousStop, stop)
        if (activeMarks.length > 0) {
            const seg = new Segment()
            applyMarks(seg, ...activeMarks)
            seg.data = text
            segments.push(seg)
            if (detailed) console.log('Current Stop:', stop, 'Active Marks:', activeMarks, 'Data:', text)
            activeMarks = activeMarks.filter((mark) => mark.position + mark.length <= stop)
        }
        combinedMarks = combinedMarks.filter((mark) => {
            if (mark.position <= stop) {
                activeMarks.push(mark)
                return false
            }
            return true
        })
        previousStop = stop
        if (detailed) console.log('Next Marks:', activeMarks, 'Remaining Combined Marks:', combinedMarks)
        return segments
    }, [])
    if (detailed) console.log('Generated Segments:', segments)
    const paragraph = new Paragraph(segments, start)
    if (start !== undefined && segments) paragraph.updateSegStarts()
    if (detailed) console.log('Original Section:', section, 'Generated Paragraph:', paragraph)
    return paragraph
}

export type ParagraphHandle = { id: UniqueId; paragraph: Paragraph }

export function sectionsToParagraphs(document: Document): ParagraphHandle[] {
    let currentPosition = 0
    return document.getSections().map((handle) => {
        const newParagraph = { id: handle.id, paragraph: sectionToParagraph(handle.section, currentPosition) }
        currentPosition += newParagraph.paragraph.length
        return newParagraph
    })
}
