from concurrent.futures import ThreadPoolExecutor, as_completed
from datetime import datetime
from typing import List

import numpy as np

from scraper.log import root_logger
from scraper.risuai.char_process import process_risuai_char
from scraper.stop_event import global_stop_event
from scraper.time import calculate_elapsed_time

logger = root_logger.get_child('RISUAI.CHAR_RUNNER')


def run_risuai_char_scrape(urls: List[str], threads: int):
    start = datetime.now()
    processed_count = 0
    new_items = 0
    updated_items = 0
    execution_duration = []
    found_users = set()

    with (ThreadPoolExecutor(max_workers=threads) as executor):
        futures = {executor.submit(process_risuai_char, url) for url in urls}
        for future in as_completed(futures):
            result = future.result()
            if global_stop_event.is_set():
                return

            node, char_card, card_raw, elapsed, did_update_def, is_new_node = result
            if not node:
                # The downloader skipped this item
                continue

            found_users.add(node.authorname)

            mod_str = ''
            if is_new_node:
                mod_str = '(new)'
                new_items += 1
            elif did_update_def:
                mod_str = '(modified)'
                updated_items += 1
            processed_count += 1
            logger.info(f'Character completed in {int(elapsed)}s: "{node.id}" {mod_str}')
            execution_duration.append(elapsed)

    time_str = calculate_elapsed_time(start)
    logger.info(f'-- Characters scraping completed in {time_str} --'.upper())

    if len(execution_duration):
        avg_execution_duration = np.average(execution_duration)
    else:
        avg_execution_duration = 0
    return processed_count, new_items, updated_items, avg_execution_duration, found_users
