import json
import os
import shutil
import tempfile
import time
import traceback
import zipfile
from concurrent.futures import ThreadPoolExecutor

from scraper import http_queue
from scraper.card_types.ccv3.infer import infer_specv3_card
from scraper.card_types.funcs import test_card_name
from scraper.download import fetch_json
from scraper.log import root_logger
from scraper.risuai.char_insert import insert_risuai_character
from scraper.risuai.download import get_risuai_avatar
from scraper.risuai.fetch import get_risuai_character

_logger = root_logger.get_child('RISUAI.CHAR_PROCESS')


def process_risuai_char(url: str):
    start = time.time()

    try:
        node = get_risuai_character(url)
        if not node:
            _logger.error(f'Failed to fetch character for {url}')
            return None, None, None, None, None, None

        with ThreadPoolExecutor() as executor:
            if not node.is_charx:
                def_future = executor.submit(fetch_json, f'https://realm.risuai.net/api/v1/download/json-v3/{node.id}')
            else:
                def_future = executor.submit(fetch_charx_card_json, f'https://realm.risuai.net/api/v1/download/charx-v3/{node.id}')

            avatar_future = executor.submit(get_risuai_avatar, node.img_url)

        card_json, card_raw, err, def_status_code = def_future.result()
        if err or card_json is None:
            _logger.error(f'Failed to fetch character def for "{node.id}" - code: {def_status_code} - {err}')
            return None, None, None, None, None, None

        png_bytes: bytes = avatar_future.result()
        if not png_bytes:
            _logger.error(f'No avatar for card {node.id}')
            return None, None, None, None, None, None

        if not test_card_name(card_json['data']['name']):
            _logger.error(f'Character name is empty: {node.id}')
            return None, None, None, None, None, None

        char_card = infer_specv3_card(card_json, source='risuai', specific_author=node.authorname, specific_name=node.name)
        did_update_def, is_new_node = insert_risuai_character(node, char_card, png_bytes, card_raw)

        return node, char_card, card_raw, time.time() - start, did_update_def, is_new_node
    except:
        _logger.error(f'Exception in process_risuai_char for {url} -- {traceback.format_exc()}')
        return None, None, None, None, None, None


def fetch_charx_card_json(url: str):
    last_traceback = None
    last_status_code = None

    for i in range(3):
        temp_dir = None
        try:
            r = http_queue.add(url)
            if r is None:
                return None, None, 'no response', None

            last_status_code = r.status_code
            if r.status_code == 404 or r.status_code == 500:
                return None, None, None, last_status_code
            elif r.status_code != 200:
                time.sleep(5)
                continue

            temp_dir = tempfile.mkdtemp()
            zip_path = os.path.join(temp_dir, 'temp.zip')
            with open(zip_path, 'wb') as f:
                f.write(r.content)

            with zipfile.ZipFile(zip_path, 'r') as zip_ref:
                zip_ref.extractall(temp_dir)

            card_json_path = os.path.join(temp_dir, 'card.json')
            if not os.path.exists(card_json_path):
                return None, None, 'card.json not found in zip', last_status_code

            with open(card_json_path, 'r', encoding='utf-8') as f:
                json_text = f.read()
                json_data = json.loads(json_text)

            return json_data, json_text, None, None

        except Exception as e:
            last_traceback = traceback.format_exc()
            time.sleep(5)
            continue
        finally:
            if temp_dir and os.path.exists(temp_dir):
                shutil.rmtree(temp_dir)

    return None, None, last_traceback, last_status_code
