import io
import json
import zlib
from datetime import timezone, datetime

from PIL import Image
from psycopg2.extras import RealDictCursor

from scraper.card_types.ccv3.character import CharacterCardV3
from scraper.card_types.ccv3.funcs import count_v3char_tokens, normalize_v3card_for_comparison
from scraper.card_types.ccv3.infer import normalize_cardv3
from scraper.database.connection import CursorFromConnectionFromPool
from scraper.database.exif import strip_exif
from scraper.database.hash import hash_bytestream
from scraper.globals import GLOBALS
from scraper.insert import write_bytes_to_hashed_file
from scraper.log import root_logger
from scraper.risuai.types import RisuaiNode

_logger = root_logger.get_child('CHUB.CHAR.INSERT')


def insert_risuai_character(node: RisuaiNode, char_card: CharacterCardV3, png_bytes: bytes, card_raw: str, extra_metadata: dict = None):
    try:
        img = strip_exif(Image.open(io.BytesIO(png_bytes))).convert('RGB')
    except ValueError as e:
        _logger.error(f'Failed to convert image for {node.id}: {e}')
        return False, False

    # Card image is always updated and is not versioned.
    img.thumbnail(GLOBALS.card_image_max_dimensions, Image.LANCZOS)
    byte_stream = io.BytesIO()
    img.save(byte_stream, format='PNG', optimize=True)

    original_hash = hash_bytestream(io.BytesIO(png_bytes))
    char_card.data.extensions.char_archive.original_card_hash = original_hash
    char_card.data.extensions.char_archive.source_id = node.id

    # Always update the file in case it was missing
    png_hash = write_bytes_to_hashed_file(byte_stream)

    def_metadata = {
        'totalTokens': count_v3char_tokens(char_card)
    }
    node_metadata = {}
    if extra_metadata:
        def_metadata.update(extra_metadata)
        node_metadata.update(extra_metadata)

    # TODO: implement comments once the site implements them
    comments = []

    # ===============================================================================
    # SQL
    with CursorFromConnectionFromPool(cursor_factory=RealDictCursor) as cursor:
        # Get the original "added" timestamp
        cursor.execute(
            'SELECT added FROM risuai_character WHERE author = %s AND name = %s AND id = %s',
            (node.authorname, node.name, node.id)
        )
        added_result = cursor.fetchone()
        if added_result is not None:
            added_ts = added_result['added']
        else:
            added_ts = datetime.now(timezone.utc)

        # Check if the node name was modified. When a card name is modified on risuai, the ID will stay the same but the name will change.
        # So, we must update the existing entry instead of inserting something new.
        cursor.execute('SELECT * from risuai_character where id = %s', (node.id,))
        node_exists = cursor.fetchone()
        if node_exists is not None and node_exists['name'] != node.name:
            cursor.execute('UPDATE risuai_character SET name = %s WHERE id = %s', (node.name, node.id))

        # Update the node.
        cursor.execute(
            """
            DELETE FROM risuai_character WHERE author = %s AND name = %s AND id = %s;
            INSERT INTO risuai_character (author, name, id, node, metadata, added) VALUES (%s, %s, %s, %s, %s, %s);
            """,
            (
                node.authorname, node.name, node.id,
                node.authorname, node.name, node.id, node.model_dump_json(), json.dumps(node_metadata), added_ts
            )
        )

        # Check if we need to update the def.
        cursor.execute(
            'SELECT definition FROM risuai_character_def WHERE author = %s AND name = %s AND id = %s ORDER BY added DESC',
            (node.authorname, node.name, node.id)
        )
        existing_card = cursor.fetchone()
        is_new_node = should_insert = False
        if existing_card is not None:
            old_card = normalize_v3card_for_comparison(normalize_cardv3(existing_card['definition'], source='risuai'))
            new_card = normalize_v3card_for_comparison(normalize_cardv3(json.loads(char_card.model_dump_json()), source='risuai'))
            assert new_card['data'].get('extensions') is None
            assert old_card['data'].get('extensions') is None
            if old_card != new_card:
                is_new_node = False
                should_insert = True
        else:
            is_new_node = should_insert = True

        if should_insert:
            cursor.execute(
                """
                INSERT INTO risuai_character_def (author, name, id, definition, "raw", image_hash, original_card_hash, metadata) VALUES (%s, %s, %s, %s, %s, %s, %s, %s)""",
                (node.authorname, node.name, node.id, char_card.model_dump_json(), zlib.compress(card_raw.encode()), png_hash, original_hash, json.dumps(def_metadata))
            )
        return should_insert, is_new_node
