import multiprocessing
import re
import sys
import threading
import time
from threading import Thread
from typing import List

from scraper.globals import GLOBALS
from scraper.log import root_logger
from scraper.request_worker import worker_process

logger = root_logger.get_child('REQUEST QUEUE')

"""
This request queue/worker thing was originally used as a form of ratelimiting since the number of concurrent requests
can never be greater than the number of active workers. But since we now have access to 1,000 proxies to route our
requests through, this ratelimiting is not longer needed. 
"""


class RequestQueueManager:
    """
    Note: we no longer use `stop_event` or do any sort of shutdown management. We just kill ourselves because who the fuck cares.
    """

    def __init__(self, num_workers_per_proxy):
        self.num_workers_per_proxy = num_workers_per_proxy
        self.job_queue = multiprocessing.Queue()
        self.workers: List[threading.Thread] = []
        self.request_counters = {proxy: multiprocessing.Value('i', 0) for proxy in GLOBALS.proxies}
        self.results = multiprocessing.Manager().dict()
        self.printer_thread = None
        self.manager = multiprocessing.Manager()
        self.stop_event = threading.Event()

        self.start_workers()

        if GLOBALS.log_requests_per_sec:
            self.printer_thread = Thread(target=self.print_requests_per_second, daemon=True)
            self.printer_thread.start()

    def start_workers(self):
        for proxy in GLOBALS.proxies:
            for i in range(self.num_workers_per_proxy):
                worker = Thread(target=worker_process, args=(self.job_queue, self.request_counters[proxy], self.stop_event, i), daemon=True)
                worker.start()
                self.workers.append(worker)

    def add(self, url, method='GET', post_data=None, json_data=None, headers: dict = None, retries=5, delay=10, timeout: int = None, handle_err: bool = True, proxy=None, suicide_on_429: bool = True, cookies: dict = None, blocking: bool = True):
        result_queue = self.manager.Queue()
        self.job_queue.put(((url, method, post_data, json_data, headers, retries, delay, timeout, handle_err, proxy, suicide_on_429, cookies), result_queue))

        if not blocking:
            return result_queue
        else:
            return self.get_result(url, result_queue)

    def get_result(self, url, result_queue):
        response, returned_url = result_queue.get()

        # Be extra sure the HTTP workers gave us the right response!
        if returned_url != returned_url:
            logger.critical(f'Sent URL {url} does not equal returned URL {returned_url}')
            sys.exit(1)

        return response

    def quit(self):
        if not self.stop_event.is_set():
            logger.debug('RequestQueue is stopping...')
            self.stop_event.set()
            for worker in self.workers:
                worker.join()
            logger.debug('RequestQueue has stopped.')

    def print_requests_per_second(self):
        interval = 3
        while True:
            print_str = ''
            for proxy, counter in self.request_counters.items():
                proxy_str = re.match(r'http://(.*?):\d*', proxy).group(1)
                print_str += f' | {proxy_str}: {int(counter.value / interval)}'
                counter.value = 0
            logger.info(print_str.strip().strip('|'))
            time.sleep(interval)
