import itertools
import json
import threading
import time
from json import JSONDecodeError
from typing import Any

import curl_cffi.requests
import requests

from .globals import GLOBALS
from .log import root_logger
from .suicide import watchdog_suicide

logger = root_logger.get_child('HTTP')

lock = threading.Lock()

proxy_pool: itertools.cycle
proxy_pool_lock = threading.Lock()


def init_proxy_pool():
    global proxy_pool
    proxy_pool = itertools.cycle(GLOBALS.proxies)


def get_next_proxy():
    with proxy_pool_lock:
        try:
            return next(proxy_pool)
        except StopIteration:
            raise Exception("Proxy list is empty")
        except NameError:
            raise Exception('You forgot to do init_proxy_pool()')


def handle_request(
        url,
        method='GET',
        post_data=None,
        json_data=None,
        headers: dict = None,
        retries=5,
        delay=10,
        timeout: int = None,
        handle_err: bool = True,
        proxy=None,
        suicide_on_429: bool = False,
        cookies: dict = None
):
    """
    Returns the response (may be None if it failed) and the URL that was sent. This URL is used by RequestQueueManager.get_result() to verify there was
    no mixup in the queue. The URL is not returned to the caller.
    """
    if not headers:
        headers = {}
    headers = GLOBALS.headers | headers

    if not timeout:
        timeout = GLOBALS.request_timeout

    if not cookies:
        cookies = {}

    if not proxy:
        proxy = get_next_proxy()
    elif proxy is False:
        # No proxy
        proxy = None

    clean_url = prepare_url(url)

    # Bump the retries count up by 1 so that we can ask for 0 retries.
    retries = retries + 1

    for i in range(retries):
        if GLOBALS.log_http_requests:
            logger.debug(f'Making {method} using {proxy} to {clean_url}')

        try:
            # =======================================================================================================================================
            # THE HTTP REQUEST
            if method == 'POST':
                # curl_cffi.
                r = requests.post(clean_url, headers=headers, json=json_data, data=post_data, timeout=timeout, cookies=cookies, proxies={"http": proxy, "https": proxy})  # , impersonate=random.choice(IMPERSONATE_AGENTS))
            elif method == 'GET':
                # curl_cffi.
                r = requests.get(clean_url, headers=headers, timeout=timeout, cookies=cookies, proxies={"http": proxy, "https": proxy})  # , impersonate=random.choice(IMPERSONATE_AGENTS))
            elif method == 'HEAD':
                # curl_cffi.
                r = requests.head(clean_url, headers=headers, timeout=timeout, cookies=cookies, proxies={"http": proxy, "https": proxy})  # , impersonate=random.choice(IMPERSONATE_AGENTS))
            else:
                raise ValueError(f'Unrecognized method: {method}')
            # r = cffi_to_requests(r)
            # =======================================================================================================================================
        except Exception as e:
            if handle_err:
                continue_trying = chttp_handle_err(e, clean_url, proxy, handle_err)
                if continue_trying:
                    continue
                else:
                    return None, clean_url
            else:
                raise

        if r and suicide_on_429 and r.status_code == 429:
            # Really lazy
            try:
                after = round(int(r.headers.get("retry-after")) / 60, 1) if int(r.headers.get("retry-after", 0)) > 0 else None
            except:
                after = None
            logger.critical(f'GOT 429 ERROR, QUITTING!!! {clean_url} - Retry after: {after} minutes.')
            watchdog_suicide()
            return None, clean_url

        if handle_err:
            if 400 < r.status_code < 500:
                # We can't recover from a 4xx error
                logger.debug(f'Got HTTP code {r.status_code} - {clean_url}')
                return r, clean_url
            elif r.status_code == 500:
                logger.debug(f'500 Internal Server Error - {clean_url}')
                if i < retries - 1:  # i is zero indexed
                    time.sleep(delay + i)  # increase delay by 1 second each retry
                    continue
                else:
                    logger.debug(f'Status 500 encountered after {retries} retries - {clean_url}')
                    return r, clean_url
        return r, clean_url
    logger.error(f'HTTP REQUEST FATAL ERROR: failed to reach "{clean_url}"')
    return None, clean_url


def chttp_handle_err(e: Any, url, proxy, handle_err: bool):
    """
    False: don't retry
    True: do retry
    """
    if isinstance(e, requests.exceptions.SSLError):
        # This error is common and should be transient. We can ignore it
        logger.debug(f'{e.__class__.__name__}: {e} - {url} - {proxy}')
        return True
    elif isinstance(e, curl_cffi.requests.RequestsError) and e.code == 60:
        logger.error(f'{e.__class__.__name__}: {e} - {url} - {proxy}')
        return False
    elif isinstance(e, (requests.exceptions.ProxyError, curl_cffi.requests.RequestsError)):
        if (isinstance(e, curl_cffi.requests.RequestsError) and e.code == 56) and 'Received HTTP code' in str(e) and 'from proxy after CONNECT' in str(e):
            # This is probably an error with Smartproxy, and we cannot do anything but kill ourselves.
            logger.error(f'Connection failed: {strip_curl_msg(e)} - "{url}"')
            return False
        elif isinstance(e, curl_cffi.requests.errors.CookieConflict):
            # This is a bug.
            logger.error(f'Encountered curl_cffi bug: {e} - "{url}"')
            return False
        else:
            logger.debug(f'Connection to proxy failed: {strip_curl_msg(e)} - "{url}"')
            time.sleep(5)
            return True
    else:
        logger.warn(f'{e.__class__.__name__}: {e} - {url} - {proxy}')
        if not handle_err:
            raise
        return True


def strip_curl_msg(e) -> str:
    return str(e).strip(" This may be a libcurl error, See https://curl.se/libcurl/c/libcurl-errors.html first for more details.")


IMPERSONATE_AGENTS = [
    'chrome99',
    'chrome100',
    'chrome101',
    'chrome104',
    'chrome107',
    'chrome110',
    'chrome99_android',
    'edge99',
    'edge101',
    'safari15_3',
    'safari15_5',
]


class SimpleResponse:
    """
    This class simulates a requests.Response object.
    It's used to convert an unpickle-able curl_cffi.requests.Response to something pickle-able.
    """

    def __init__(self, url, status_code, headers, content, text, cookies):
        self.url = url
        self.status_code = status_code
        self.headers = headers
        self.content = content
        self.text = text
        self.cookies = cookies

    def json(self) -> dict:
        try:
            return json.loads(self.text)
        except JSONDecodeError as e:
            raise requests.exceptions.JSONDecodeError(e.msg)


def cffi_to_requests(curl_response: curl_cffi.requests.Response):
    """
    Simple helper method.
    :param curl_response:
    :return:
    """
    return SimpleResponse(
        url=curl_response.url,
        status_code=curl_response.status_code,
        headers=dict(curl_response.headers),
        content=curl_response.content,
        text=curl_response.text,
        cookies=dict(curl_response.cookies)
    )


def prepare_url(url: str):
    # curl_cffi uses curl which doesn't like spaces in URLs.
    url = url.replace(' ', '%20')
    return url
